﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_MemoryAttribute.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/usb/usb_Host.h>
#include <nn/eth/eth_EthTypes.h>

#include "eth_Asix88772.h"
#include "eth_Asix8817X.h"
#include "eth_UsbDeviceInfo.h"

namespace nn     {
namespace eth    {
namespace device {
namespace usb    {

nn::util::IntrusiveList <
    UsbDeviceInfo,
    nn::util::IntrusiveListBaseNodeTraits<UsbDeviceInfo>
    > g_UsbDeviceInfoList;


uint32_t UsbDeviceInfo::m_InterfaceCount = 0;

template <class T>
class EthAdapter : public UsbDeviceInfo
{

public:
    EthAdapter(uint16_t idVendor, uint16_t idProduct)
    NN_NOEXCEPT :
        UsbDeviceInfo(idVendor, idProduct)
    {

    }

    static void Finalize(UsbDevice* pUsbDevice, void* arg)
    NN_NOEXCEPT
    {
        T* pAdapter = reinterpret_cast<T*>(arg);

        if (arg != nullptr)
        {
            pAdapter->Finalize();
            delete pAdapter;

            NN_ABORT_UNLESS(UsbDeviceInfo::m_InterfaceCount > 0);
            UsbDeviceInfo::m_InterfaceCount--;
        }

        pUsbDevice->Finalize();

        delete pUsbDevice;
    }

    nn::Result Initialize(nn::usb::Host* pHost, nn::usb::InterfaceProfile* pProfile)
    NN_NOEXCEPT
    {
        UsbDevice* pUsbDevice = nullptr;
        T*         pAdapter   = nullptr;
        nn::Result result     = nn::eth::ResultOutOfMemory();

        if (UsbDeviceInfo::m_InterfaceCount >= MaxInterfaceCount)
        {
            // Even though we won't use it, claim this device to receive detach event.
            // NIFM requires us to keep accurate count for attached devices.
            // Calls below set up detach event without initializing Eps.

            if ((pUsbDevice = new UsbDevice(nullptr, 0)) != nullptr)
            {
                result = pUsbDevice->Initialize(pHost, pProfile, Finalize, nullptr);
            }
        }
        else if ((pUsbDevice = new UsbDevice(T::EndPoints, T::NumberOfEndPoints)) != nullptr &&
                 (pAdapter   = new(nn::os::MemoryPageSize) T(pUsbDevice)) != nullptr)
        {
            if ((result = pUsbDevice->Initialize(pHost, pProfile, Finalize, reinterpret_cast<void*>(pAdapter))).IsSuccess())
            {
                if ((result = pAdapter->Initialize()).IsSuccess())
                {
                    UsbDeviceInfo::m_InterfaceCount++;
                }
                else
                {
                    pUsbDevice->Finalize();
                }
            }
        }

        if (result.IsFailure())
        {
            if (pAdapter != nullptr)
            {
                delete pAdapter;
            }
            if (pUsbDevice != nullptr)
            {
                delete pUsbDevice;
            }
        }

        return result;
    }
};

// supported adapters
EthAdapter<Asix88772> g_Asix88772(ASIX_88772_VID, ASIX_88772_PID);
EthAdapter<Asix8817X> g_Asix8817X(ASIX_8817X_VID, ASIX_88179_PID);

}}}}
