﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Result.h>
#include <nn/usb/usb_Host.h>
#include <nn/eth/eth_Result.h>

#include "../eth_Allocator.h"
#include "../eth_Packet.h"

#include "eth_UsbDevice.h"

namespace nn     {
namespace eth    {
namespace device {
namespace usb    {

extern nn::os::MultiWaitType g_MultiWait;

nn::util::IntrusiveList<UsbDevice, nn::util::IntrusiveListBaseNodeTraits<UsbDevice>> UsbDevice::m_DeviceList;

nn::Result UsbDevice::AssociateEndpoints(nn::usb::InterfaceProfile* pProfile)
NN_NOEXCEPT
{
    int activatedEndPoints = 0;
    nn::Result result;

    for (int i = 0; i < EpCount; i++)
    {
        for (int j = 0; j < nn::usb::UsbLimitMaxEndpointPairCount; j++)
        {
            nn::usb::UsbEndpointDescriptor* pTempIn  = &(pProfile->epInDesc[j]);
            nn::usb::UsbEndpointDescriptor* pTempOut = &(pProfile->epOutDesc[j]);

            if (EpDescriptors[i].type                  == nn::usb::UsbEndpointType_Bulk &&
                EpDescriptors[i].direction             == nn::usb::UsbEndpointDirection_ToHost &&
                nn::usb::UsbGetEndpointType(pTempIn)   == nn::usb::UsbEndpointType_Bulk)
            {
                if ((result = m_EpSession[i].Initialize(&m_IfSession, pTempIn, MaxInPacketCount, BulkInMaxSize)).IsFailure())
                {
                    return result;
                }
                activatedEndPoints++;
                break;
            }

            if (EpDescriptors[i].type                  == nn::usb::UsbEndpointType_Bulk &&
                EpDescriptors[i].direction             == nn::usb::UsbEndpointDirection_ToDevice &&
                nn::usb::UsbGetEndpointType(pTempOut)  == nn::usb::UsbEndpointType_Bulk)
            {
                if ((result = m_EpSession[i].Initialize(&m_IfSession, pTempOut, 2, BulkOutMaxSize)).IsFailure())
                {
                    return result;
                }
                activatedEndPoints++;
                break;
            }

            if (EpDescriptors[i].type                  == nn::usb::UsbEndpointType_Int &&
                EpDescriptors[i].direction             == nn::usb::UsbEndpointDirection_ToHost &&
                nn::usb::UsbGetEndpointType(pTempIn)   == nn::usb::UsbEndpointType_Int)
            {
                if ((result = m_EpSession[i].Initialize(&m_IfSession, pTempIn, 2, InterruptInMaxSize)).IsFailure())
                {
                    return result;
                }
                activatedEndPoints++;
                break;
            }
        }

        if (activatedEndPoints == EpCount)
        {
            break;
        }
    }

    if (activatedEndPoints != EpCount)
    {
        return nn::eth::ResultDeviceError();
    }

    return ResultSuccess();
}

nn::Result UsbDevice::SubmitControlOutRequest(
    size_t*  pLengthOut,
    void*    pData,
    uint8_t  request,
    uint8_t  type,
    uint16_t value,
    uint16_t index,
    uint16_t length)
NN_NOEXCEPT
{
    nn::Result result;
    Packet     packet;

    if (length >= nn::usb::HwLimitDmaBufferAlignmentSize)
    {
        return ResultInvalidArgument();
    }

    if ((result = packet.Initialize(nn::usb::HwLimitDmaBufferAlignmentSize)).IsSuccess())
    {
        memcpy(packet.m_pData, pData, length);

        result = m_IfSession.ControlRequest(
                                pLengthOut,
                                packet.m_pData,
                                type,
                                request,
                                value,
                                index,
                                length);
        packet.Finalize();
    }

    return result;
}

nn::Result UsbDevice::SubmitControlInRequest(
    size_t*  pLengthOut,
    void*    pData,
    uint8_t  request,
    uint8_t  type,
    uint16_t value,
    uint16_t index,
    uint16_t length)
NN_NOEXCEPT
{
    nn::Result result;
    Packet     packet;

    if (length >= nn::usb::HwLimitDmaBufferAlignmentSize)
    {
        return ResultInvalidArgument();
    }

    if ((result = packet.Initialize(nn::usb::HwLimitDmaBufferAlignmentSize)).IsSuccess())
    {
        result = m_IfSession.ControlRequest(
                                pLengthOut,
                                packet.m_pData,
                                type,
                                request,
                                value,
                                index,
                                length);

        if (result.IsSuccess())
        {
            memcpy(pData, packet.m_pData, *pLengthOut < length ? *pLengthOut : length);
        }

        packet.Finalize();
    }

    return result;
}

nn::Result UsbDevice::RunAsyncInOperation(
    Packet*  pPacket,
    uint32_t packetCount,
    std::function<void(Packet* pPacket)>ProcessPacketFunction,
    int ep)
NN_NOEXCEPT
{
    nn::Result result;
    uint32_t xferCount;
    nn::usb::XferReport xferReports[MaxInPacketCount];
    nn::os::SystemEventType* pCompletionEvent = m_EpSession[ep].GetCompletionEvent();

    NN_ABORT_UNLESS(packetCount <= MaxInPacketCount);

    for (uint32_t i = 0; i < packetCount; i++)
    {
        if ((result = m_EpSession[ep].PostBufferAsync(
                        &pPacket[i].m_Id,
                        pPacket[i].m_pData,
                        pPacket[i].m_TotalLength,
                        i)
                      ).IsFailure())
        {
            return result;
        }
    }

    while (NN_STATIC_CONDITION(true))
    {
        nn::os::WaitSystemEvent(pCompletionEvent);
        nn::os::ClearSystemEvent(pCompletionEvent);

        if ((result = m_EpSession[ep].GetXferReport(
                        &xferCount,
                        xferReports,
                        sizeof(xferReports) / sizeof(xferReports[0]))
                      ).IsFailure())
        {
            return result;
        }

        NN_ABORT_UNLESS(xferCount <= packetCount);

        for (uint32_t i = 0; i < xferCount; i++)
        {
            uint32_t index = static_cast<uint32_t>(xferReports[i].context);

            NN_ABORT_UNLESS(index < packetCount);
            NN_ABORT_UNLESS(pPacket[index].m_Id == xferReports[i].id);

            if ((result = xferReports[i].result).IsFailure())
            {
                m_IfSession.ResetDevice();
                m_EpSession[ep].Finalize();
                return result;
            }

            pPacket[index].m_ValidLength = xferReports[i].transferredSize;
            ProcessPacketFunction(&pPacket[index]);
        }

        for (uint32_t i = 0; i < xferCount; i++)
        {
            uint32_t index = static_cast<uint32_t>(xferReports[i].context);

            if ((result = m_EpSession[ep].PostBufferAsync(
                            &pPacket[index].m_Id,
                            pPacket[index].m_pData,
                            pPacket[index].m_TotalLength,
                            index)
                          ).IsFailure())
            {
                return result;
            }
        }
    }

    return result;
}

nn::Result UsbDevice::SubmitInRequest(size_t* pLengthOut, int ep, void* pData, size_t dataLength)
NN_NOEXCEPT
{
    return m_EpSession[ep].PostBuffer(pLengthOut, pData, dataLength);
}

nn::Result UsbDevice::SubmitOutRequest(size_t* pLengthOut, int ep, void* pData, size_t dataLength)
NN_NOEXCEPT
{
    return m_EpSession[ep].PostBuffer(pLengthOut, pData, dataLength);
}

void UsbDevice::CancelTransfers()
NN_NOEXCEPT
{
    for (int i = 0; i < EpCount; i++)
    {
        m_EpSession[i].Terminate();
    }
}

nn::Result UsbDevice::Initialize(
    nn::usb::Host*             pHost,
    nn::usb::InterfaceProfile* pProfile,
    FinalizeFunction            finalizeFunction,
    void*                       finalizeArgument)
NN_NOEXCEPT
{
    nn::Result result;

    m_FinalizeFunction = finalizeFunction;
    m_FinalizeArgument = finalizeArgument;

    if ((result = m_IfSession.Initialize(pHost, pProfile->handle)).IsSuccess())
    {
        m_pEvent = m_IfSession.GetStateChangeEvent();

        nn::os::InitializeMultiWaitHolder(&m_MultiWaitHolder, m_pEvent);
        nn::os::SetMultiWaitHolderUserData(&m_MultiWaitHolder, reinterpret_cast<uintptr_t>(this));
        nn::os::LinkMultiWaitHolder(&g_MultiWait, &m_MultiWaitHolder);

        if ((result = AssociateEndpoints(pProfile)).IsFailure())
        {
            for (int i = 0; i < EpCount; i++)
            {
                m_EpSession[i].Finalize();
            }

            nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
            nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);

            m_IfSession.Finalize();
        }
    }

    return result;
}

nn::Result UsbDevice::Finalize()
NN_NOEXCEPT
{
    for (int i = 0; i < EpCount; i++)
    {
        m_EpSession[i].Finalize();
    }

    nn::os::UnlinkMultiWaitHolder(&m_MultiWaitHolder);
    nn::os::FinalizeMultiWaitHolder(&m_MultiWaitHolder);

    m_IfSession.Finalize();

    return ResultSuccess();
}

UsbDevice::UsbDevice(const EndPointDescriptor* descriptors, int count)
NN_NOEXCEPT :
    EpDescriptors(descriptors),
    EpCount(count),
    m_EpSession(nullptr),
    m_FinalizeFunction(nullptr)
{
    if (EpCount > 0)
    {
        m_EpSession = reinterpret_cast<nn::usb::HostEndpoint*>(
                lmem::AllocateFromExpHeap(
                            g_HeapHandle,
                            sizeof(nn::usb::HostEndpoint) * EpCount)
        );
        NN_ABORT_UNLESS(m_EpSession != nullptr);

        for (int i = 0; i < EpCount; i++)
        {
            new(&m_EpSession[i]) nn::usb::HostEndpoint();
        }
    }

    m_DeviceList.push_back(*this);
}

UsbDevice::~UsbDevice()
NN_NOEXCEPT
{
    m_DeviceList.erase(m_DeviceList.iterator_to(*this));

    if (m_EpSession != nullptr)
    {
        for (int i = 0; i < EpCount; i++)
        {
            m_EpSession[i].Finalize();
            m_EpSession[i].~HostEndpoint();
        }

        lmem::FreeToExpHeap(g_HeapHandle, m_EpSession);
    }
}

}}}}
