﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SdkThreadCommon.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/eth/eth_EthTypes.h>
#include <net/if_dl.h>

#include "../mii_Registers.h"
#include "eth_Asix8817X.reg.h"
#include "eth_Asix8817X.h"

#include <siglo/device.h>

namespace nn     {
namespace eth    {
namespace device {
namespace usb    {

#define RETURN_IF_FAILURE(f)                                \
do                                                          \
{                                                           \
    nn::Result result = (f);                                \
    if (result.IsFailure())                                 \
    {                                                       \
        NN_SDK_LOG("[asix]: failed at %d (%d:%d)\n",        \
                    __LINE__,                               \
                    result.GetModule(),                     \
                    result.GetDescription());               \
        return result;                                      \
    }                                                       \
}                                                           \
while(NN_STATIC_CONDITION(0))

const int Asix8817X::NumberOfEndPoints = EndpointCount;

const EndPointDescriptor Asix8817X::EndPoints[EndpointCount]
{
    [BulkOutEp] =
    {
        .type      = nn::usb::UsbEndpointType_Bulk,
        .direction = nn::usb::UsbEndpointDirection_ToDevice,
    },
    [BulkInEp] =
    {
        .type      = nn::usb::UsbEndpointType_Bulk,
        .direction = nn::usb::UsbEndpointDirection_ToHost,
    },
    [InterruptInEp] =
    {
        .type      = nn::usb::UsbEndpointType_Int,
        .direction = nn::usb::UsbEndpointDirection_ToHost,
    },
};

nn::Result Asix8817X::WriteMemory(uint8_t command, uint16_t index, uint16_t value, void* pData, int dataLength)
NN_NOEXCEPT
{
    size_t lengthOut;
    return m_pUsbDevice->SubmitControlOutRequest(
                &lengthOut,
                pData,
                command,
                UT_WRITE_VENDOR_DEVICE,
                value,
                index,
                dataLength);
}

nn::Result Asix8817X::ReadMemory(uint8_t command, uint16_t index, uint16_t value, void* pData, int dataLength)
NN_NOEXCEPT
{
    size_t lengthOut;
    return m_pUsbDevice->SubmitControlInRequest(
                &lengthOut,
                pData,
                command,
                UT_READ_VENDOR_DEVICE,
                value,
                index,
                dataLength);
}

nn::Result Asix8817X::WriteCommand(uint8_t command, uint16_t address, uint8_t data)
NN_NOEXCEPT
{
    return WriteMemory(command, 1, address, &data, sizeof(data));
}

nn::Result Asix8817X::WriteCommand(uint8_t command, uint16_t index, uint16_t address, uint16_t data)
NN_NOEXCEPT
{
    return WriteMemory(command, index, address, &data, sizeof(data));
}

nn::Result Asix8817X::ReadCommand(uint8_t command, uint16_t address, uint8_t* pData)
NN_NOEXCEPT
{
    return ReadMemory(command, 1, address, pData, sizeof(*pData));
}

nn::Result Asix8817X::ReadCommand(uint8_t command, uint16_t index, uint16_t address, uint16_t* pData)
NN_NOEXCEPT
{
    return ReadMemory(command, index, address, pData, sizeof(*pData));
}

nn::Result Asix8817X::PhyReadReg(uint16_t miiRegister, uint16_t* pMiiValue)
NN_NOEXCEPT
{
    return ReadCommand(AXGE_ACCESS_PHY, miiRegister, AXGE_PHY_ADDRESS, pMiiValue);
}

nn::Result Asix8817X::PhyWriteReg(uint16_t miiRegister, uint16_t miiValue)
NN_NOEXCEPT
{
    return WriteCommand(AXGE_ACCESS_PHY, miiRegister, AXGE_PHY_ADDRESS, miiValue);
}

nn::Result Asix8817X::PhyReset()
NN_NOEXCEPT
{
    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, 2, AXGE_EPPRCR, 0));
    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, 2, AXGE_EPPRCR, EPPRCR_IPRL));
    DriverSleepMs(250);

    RETURN_IF_FAILURE(
            WriteCommand(
                AXGE_ACCESS_MAC,
                AXGE_CLK_SELECT,
                AXGE_CLK_SELECT_ACS | AXGE_CLK_SELECT_BCS));

    DriverSleepMs(100);

    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, AXGE_PWLLR, 0x34));
    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, AXGE_PWLHR, 0x52));
    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, AXGE_CRCR,  0));
    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, AXGE_CTCR,  0));

    RETURN_IF_FAILURE(
            WriteCommand(
                AXGE_ACCESS_MAC,
                AXGE_MMSR,
                MMSR_PME_TYPE | MMSR_PME_POL));

    RETURN_IF_FAILURE(
            WriteCommand(
                AXGE_ACCESS_MAC,
                2,
                AXGE_RCR,
                RCR_IPE | RCR_DROP_CRCE | RCR_SO | RCR_AP | RCR_AB | RCR_AMALL));

    RETURN_IF_FAILURE(
            WriteCommand(
                AXGE_ACCESS_MAC,
                2,
                AXGE_MSR,
                MSR_RE | MSR_RFC | MSR_TFC | MSR_GM | MSR_FD | MSR_EN_125MHZ));

    RETURN_IF_FAILURE(
            ReadMemory(
                AXGE_ACCESS_MAC,
                sizeof(Gmii::m_PhyMacAddress),
                AXGE_NIDR,
                &(Gmii::m_PhyMacAddress),
                sizeof(Gmii::m_PhyMacAddress)));

    return ResultSuccess();
}

nn::Result Asix8817X::PhySetMedia(MediaType type)
NN_NOEXCEPT
{
    static const struct
    {
        uint8_t control;
        uint8_t timerLow;
        uint8_t timerHigh;
        uint8_t transferSize;
        uint8_t interFrameGap;
    }
    __packed axgeBulkSize[] =
    {
        { 7, 0xd2, 0x06, 0x3f, 0xff },
        { 7, 0x15, 0x0d, 0x3f, 0xff },
        { 7, 0xae, 0x07, 0x18, 0xff },
        { 7, 0xcc, 0x4c, 0x18, 0x08 }
    };

    uint16_t msrValue   = MSR_RE  | MSR_ALWAYS_ONE;
    uint16_t rcrValue   = RCR_IPE | RCR_SO | RCR_AM | RCR_AB;
    uint8_t* bulkConfig = (uint8_t*)&axgeBulkSize[3];
    uint8_t  plsr;

    RETURN_IF_FAILURE(ReadCommand(AXGE_ACCESS_MAC, AXGE_PLSR, &plsr));

    if (type == MediaType_AUTO)
    {
        msrValue |= MSR_EN_125MHZ | MSR_GM;
        bulkConfig = (uint8_t*)((plsr & PLSR_USB_SS) ? &axgeBulkSize[0] : &axgeBulkSize[1]);
    }

    if (EthFullDuplex(type))
    {
        msrValue |= MSR_FD;
    }

    if (EthFlow(type))
    {
        msrValue |= MSR_TFC | MSR_RFC;
    }

    switch (EthSpeed(type))
    {
    case 1000:
        msrValue |= MSR_GM | MSR_EN_125MHZ;
        if (plsr & PLSR_USB_SS)
        {
            bulkConfig = (uint8_t*)&axgeBulkSize[0];
        }
        else if (plsr & PLSR_USB_HS)
        {
            bulkConfig = (uint8_t*)&axgeBulkSize[1];
        }
        else
        {
            bulkConfig = (uint8_t*)&axgeBulkSize[3];
        }
        break;

    case 100:
        msrValue |= MSR_PS;
        if (plsr & (PLSR_USB_SS | PLSR_USB_HS))
        {
            bulkConfig = (uint8_t*)&axgeBulkSize[2];
        }
        else
        {
            bulkConfig = (uint8_t*)&axgeBulkSize[3];
        }
        break;

    default:
        break;
    }

    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, 2, AXGE_RCR, rcrValue));
    RETURN_IF_FAILURE(WriteMemory(AXGE_ACCESS_MAC,  5, AXGE_RX_BULKIN_QCTRL, bulkConfig, sizeof(axgeBulkSize[0])));
    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, 2, AXGE_MSR, msrValue));

    return ResultSuccess();
}

nn::Result Asix8817X::PhyAdjustMedia(MediaType type)
NN_NOEXCEPT
{
    // after autoneg completes, need to adjust clock settings
    return Asix8817X::PhySetMedia(type);
}

nn::Result Asix8817X::PhyGetMedia(MediaType* pType)
NN_NOEXCEPT
{
    uint16_t   physr;
    uint16_t   anlpar;
    uint16_t   bmcr;

    RETURN_IF_FAILURE(PhyReadReg(MII_ANLPAR, &anlpar));
    RETURN_IF_FAILURE(PhyReadReg(MII_PHYSR, &physr));
    RETURN_IF_FAILURE(PhyReadReg(MII_BMCR, &bmcr));

    *pType = (MediaType)EthMakeMediaType(
                    (physr  & PHYSR_SPEED1)     ? Type_1000_T :
                    (physr  & PHYSR_SPEED0)     ? Type_100_TX : 0,
                    (bmcr   & BMCR_AUTOEN)      ? SubType_AUTO : SubType_MANUAL,
                    (physr  & PHYSR_DUPLEX      ? Option_FDX : Option_HDX) |
                    (anlpar & ANLPAR_PAUSE_MASK ? Option_FLOW : 0),
                    (physr  & PHYSR_LINK        ? LinkState_Up : LinkState_Down));

    return ResultSuccess();
}


void Asix8817X::LinkInterrupt(Packet* pPacket)
NN_NOEXCEPT
{
    NN_ABORT_UNLESS(pPacket->m_ValidLength == sizeof(LinkState));

    LinkState* state = reinterpret_cast<LinkState*>(pPacket->m_pData);

    if ((m_LinkState ^ state->link) & AX_INT_PPLS_LINK)
    {
        m_LinkState = state->link;

        if (m_LinkState & AX_INT_PPLS_LINK)
        {
            SocketIf::SetLinkState(true);
            Gmii::UpdateLinkStatus();
        }
        else
        {
            Gmii::UpdateLinkStatus();
            SocketIf::SetLinkState(false);
        }
    }

    Tick();
}

void Asix8817X::GetFramesFromStack(Packet* pPacket)
NN_NOEXCEPT
{
    TxPacketHeader* pHeader;
    uint8_t* pBuffer   = pPacket->m_pData;
    int      remain    = static_cast<int>(pPacket->m_TotalLength);
    int      frameSize = SocketIf::GetMtu() + sizeof(TxPacketHeader);
    size_t   length;

    while (remain >= frameSize)
    {
        pHeader = reinterpret_cast<TxPacketHeader*>(pBuffer);

        if (SocketIf::GetFrameFromStack(
                &length,
                pBuffer + sizeof(TxPacketHeader),
                remain  - sizeof(TxPacketHeader)
            ).IsFailure() || length == 0)
        {
            break;
        }

        pHeader->mssLength    = 0;
        pHeader->packetLength = length;

        // tx frames are 4 byte aligned
        length   = (length + 3) & ~0x3;
        pBuffer += length + sizeof(TxPacketHeader);
        remain  -= length + sizeof(TxPacketHeader);
    }

    if (remain < 0)
    {
        remain = 0;
    }

    pPacket->m_ValidLength = pPacket->m_TotalLength - remain;
}

void Asix8817X::SendFramesToStack(Packet* pPacket)
NN_NOEXCEPT
{
    RxHeader*       pRxHeader;
    RxPacketHeader* pPktHeaders;
    uint16_t        packetCount;
    uint16_t        headerOffset;
    uint8_t*        pBuffer;
    int             remain;
    size_t          length;

    if (pPacket->m_ValidLength <= sizeof(RxHeader))
    {
        return;
    }

    pRxHeader    = reinterpret_cast<RxHeader*>(pPacket->m_pData + pPacket->m_ValidLength - sizeof(RxHeader));
    packetCount  = pRxHeader->packetCount;
    headerOffset = pRxHeader->headerOffset;

    if (headerOffset + packetCount * sizeof(RxPacketHeader) > pPacket->m_ValidLength - sizeof(RxHeader))
    {
        return;
    }

    pPktHeaders  = reinterpret_cast<RxPacketHeader*>(pPacket->m_pData + headerOffset);

    // packet data is just below packet headers
    pBuffer      = pPacket->m_pData;
    remain       = static_cast<int>(headerOffset);

    for (int i = 0; i < packetCount; i++)
    {
        if ((pPktHeaders[i].packetErrors & (AXGE_RXHDR_CRC_ERR | AXGE_RXHDR_DROP_ERR)) == 0)
        {
            // rx frames are 8 byte aligned
            length  = (pPktHeaders[i].packetLength + 7) & ~7;
            remain -= length;

            if (remain < 0)
            {
                break;
            }

            if (pBuffer[0] == 0xee && pBuffer[1] == 0xee)
            {
                if (SocketIf::SendFrameToStack(&pBuffer[2], pPktHeaders[i].packetLength - 2).IsFailure())
                {
                    break;
                }
            }

            pBuffer += length;
        }
    }
}

void Asix8817X::ReadThread()
NN_NOEXCEPT
{
    Packet     packets[AsyncPacketCount];
    nn::Result result;

    for (uint32_t i = 0; i < sizeof(packets) / sizeof(packets[0]); i++)
    {
        result = packets[i].Initialize(PacketSizeIn);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    m_pUsbDevice->RunAsyncInOperation(
                    packets,
                    sizeof(packets) / sizeof(packets[0]),
                    std::bind(&Asix8817X::SendFramesToStack, this, std::placeholders::_1),
                    BulkInEp);

    for (uint32_t i = 0; i < sizeof(packets) / sizeof(packets[0]); i++)
    {
        packets[i].Finalize();
    }
}

void Asix8817X::WriteThread()
NN_NOEXCEPT
{
    Packet packet;
    nn::Result result;

    result = packet.Initialize(PacketSizeOut);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    while (m_Running)
    {
        GetFramesFromStack(&packet);

        if (packet.m_ValidLength == 0)
        {
            m_DataEvent.Wait();
            continue;
        }

        if (m_pUsbDevice->SubmitOutRequest(
                &packet.m_ValidLength,
                BulkOutEp,
                packet.m_pData,
                packet.m_ValidLength).IsFailure())
        {
            break;
        }
    }

    packet.Finalize();
}

void Asix8817X::InterruptThread()
NN_NOEXCEPT
{
    Packet packet;
    nn::Result result;
    nn::os::SystemEventType* event;

    result = packet.Initialize(PacketSizeInterrupt);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    event = m_pUsbDevice->GetTerminateEventPointer();

    while (m_Running)
    {
        if (nn::os::TimedWaitSystemEvent(event, nn::TimeSpan::FromMilliSeconds(PollRateMs)))
        {
            break;
        }

        if ((result = m_pUsbDevice->SubmitInRequest(
                &packet.m_ValidLength,
                InterruptInEp,
                packet.m_pData,
                sizeof(LinkState))).IsFailure())
        {
            NN_SDK_LOG("[usb eth]: terminating (%d:%d)\n", result.GetModule(), result.GetDescription());
            break;
        }

        if (packet.m_ValidLength > 0)
        {
            LinkInterrupt(&packet);
        }
    }

    packet.Finalize();
}

void Asix8817X::Start()
NN_NOEXCEPT
{
    m_DataEvent.Signal();
}


nn::Result Asix8817X::UpdateMultiFilter(struct ifnet *pIfnet)
NN_NOEXCEPT
{
    struct ifmultiaddr *ifma;
    uint32_t h = 0;
    uint16_t rxmode;
    uint8_t hashtbl[8] = { 0, 0, 0, 0, 0, 0, 0, 0 };

    RETURN_IF_FAILURE(ReadCommand(AXGE_ACCESS_MAC, 2, AXGE_RCR, &rxmode));

    if (pIfnet->if_flags & (IFF_ALLMULTI | IFF_PROMISC))
    {
        rxmode |= RCR_AMALL;
        RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, 2, AXGE_RCR, rxmode));
        return ResultSuccess();
    }
    rxmode &= ~RCR_AMALL;

    TAILQ_FOREACH(ifma, &pIfnet->if_multiaddrs, ifma_link)
    {
        if (ifma->ifma_addr->sa_family != AF_LINK)
        {
            continue;
        }

        h = ether_crc32_be(
                reinterpret_cast<uint8_t *>(LLADDR(reinterpret_cast<struct sockaddr_dl *>(ifma->ifma_addr))),
                ETHER_ADDR_LEN) >> 26;
        hashtbl[h / 8] |= 1 << (h % 8);
    }

    RETURN_IF_FAILURE(WriteMemory(AXGE_ACCESS_MAC, 8, AXGE_MFA, (void *)&hashtbl, 8));
    RETURN_IF_FAILURE(WriteCommand(AXGE_ACCESS_MAC, 2, AXGE_RCR, rxmode));

    return ResultSuccess();
}

nn::Result Asix8817X::Initialize()
NN_NOEXCEPT
{
    nn::Result result;

    if ((result = Gmii::Initialize()).IsFailure())
    {
        return result;
    }

    if ((SocketIf::Initialize(
            Gmii::m_Name,
            Gmii::m_Instance,
            Gmii::m_PhyMacAddress)).IsFailure())
    {
        Gmii::Finalize();
        return result;
    }

    m_Running = true;

    m_BulkInThread.Initialize(
            std::bind(&Asix8817X::ReadThread, this),
            NN_SYSTEM_THREAD_PRIORITY(socket, EthInterfaceIngress),
            NN_SYSTEM_THREAD_NAME(socket, EthInterfaceIngress));

    m_BulkOutThread.Initialize(
            std::bind(&Asix8817X::WriteThread, this),
            NN_SYSTEM_THREAD_PRIORITY(socket, EthInterfaceEgress),
            NN_SYSTEM_THREAD_NAME(socket, EthInterfaceEgress));

    m_InterruptThread.Initialize(
            std::bind(&Asix8817X::InterruptThread, this),
            NN_SYSTEM_THREAD_PRIORITY(socket, EthEvents),
            NN_SYSTEM_THREAD_NAME(socket, EthEvents));

    return nn::ResultSuccess();
}

void Asix8817X::Finalize()
NN_NOEXCEPT
{
    m_Running = false;
    m_pUsbDevice->CancelTransfers();
    m_DataEvent.Signal();
    m_BulkInThread.Finalize();
    m_BulkOutThread.Finalize();
    m_InterruptThread.Finalize();
    SocketIf::Finalize();
    Gmii::Finalize();
}

Asix8817X::Asix8817X(UsbDevice* usb)
NN_NOEXCEPT :
    Gmii(UsbInterfaceName, Asix8817X::PollRateMs),
    m_LinkState(0),
    m_DataEvent(nn::os::EventClearMode_AutoClear, true),
    m_pUsbDevice(usb)
{

}

Asix8817X::~Asix8817X()
NN_NOEXCEPT
{

}

}}}}

