﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Synopsys DesignWare Ethernet QoS memory map
 */

#pragma once

#include <nn/dd.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_BitPack.h>

namespace nn {
namespace eth {
namespace device {
namespace tx2 {

// ------------------------------------------------------------------------------------------------
// Address map

const nn::dd::PhysicalAddress EqosPhysAddr = 0x02490000;
const size_t EqosSize = 0x10000;

// ------------------------------------------------------------------------------------------------
// Interrupt map (TRM 3.3)

const int Interrupt_EqosTx0     = 32 + 186;
const int Interrupt_EqosTx1     = 32 + 187;
const int Interrupt_EqosTx2     = 32 + 188;
const int Interrupt_EqosTx3     = 32 + 189;
const int Interrupt_EqosRx0     = 32 + 190;
const int Interrupt_EqosRx1     = 32 + 191;
const int Interrupt_EqosRx2     = 32 + 192;
const int Interrupt_EqosRx3     = 32 + 193;
const int Interrupt_EqosCommon  = 32 + 194;
const int Interrupt_EqosPower   = 32 + 195;

const size_t InterruptCount = 10;
const nn::os::InterruptName InterruptNameTable[InterruptCount] =
{
    Interrupt_EqosTx0,
    Interrupt_EqosTx1,
    Interrupt_EqosTx2,
    Interrupt_EqosTx3,
    Interrupt_EqosRx0,
    Interrupt_EqosRx1,
    Interrupt_EqosRx2,
    Interrupt_EqosRx3,
    Interrupt_EqosCommon,
    Interrupt_EqosPower,
};

// EQOS DMA registers address offset
const uint32_t RegOffset_DmaMode                      = 0x1000;
const uint32_t RegOffset_DmaSysbusMode                = 0x1004;
const uint32_t RegOffset_DmaInterruptStatus           = 0x1008;
const uint32_t RegOffset_DmaCh0Control                = 0x1100;
const uint32_t RegOffset_DmaCh0TxControl              = 0x1104;
const uint32_t RegOffset_DmaCh0RxControl              = 0x1108;
const uint32_t RegOffset_DmaCh0TxDescRingListHaddress = 0x1110;
const uint32_t RegOffset_DmaCh0TxDescRingListAddress  = 0x1114;
const uint32_t RegOffset_DmaCh0RxDescRingListHaddress = 0x1118;
const uint32_t RegOffset_DmaCh0RxDescRingListAddress  = 0x111c;
const uint32_t RegOffset_DmaCh0TxDescTailPointer      = 0x1120;
const uint32_t RegOffset_DmaCh0RxDescTailPointer      = 0x1128;
const uint32_t RegOffset_DmaCh0TxDescRingLength       = 0x112c;
const uint32_t RegOffset_DmaCh0RxDescRingLength       = 0x1130;
const uint32_t RegOffset_DmaCh0InterruptEnable        = 0x1134;
const uint32_t RegOffset_DmaCh0Status                 = 0x1160;

// DMA_Mode
struct DmaMode : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 18, 14, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 15,  3, uint32_t > Intm;
    typedef nn::util::BitPack32::Field< 12,  3, uint32_t > Pr;
    typedef nn::util::BitPack32::Field< 11,  1, bool     > Txpr;
    typedef nn::util::BitPack32::Field<  5,  6, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  2,  3, uint32_t > Taa;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Da;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Swr; // software reset
};

// DMA_SysBus_Mode
struct DmaSysBusMode : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 31,  1, bool     > EnLpi;
    typedef nn::util::BitPack32::Field< 30,  1, bool     > LpiXitPkt;
    typedef nn::util::BitPack32::Field< 26,  4, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 24,  2, uint32_t > WrOsrLm;
    typedef nn::util::BitPack32::Field< 18,  6, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field< 16,  2, uint32_t > RdOsrLmt;
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Rb;
    typedef nn::util::BitPack32::Field< 14,  1, bool     > Mb;
    typedef nn::util::BitPack32::Field< 13,  1, bool     > Onekbbe;
    typedef nn::util::BitPack32::Field< 12,  1, bool     > Aal;
    typedef nn::util::BitPack32::Field< 11,  1, bool     > Eame; // Enhanced Address Mode
    typedef nn::util::BitPack32::Field<  8,  3, uint32_t > Reserved2;
    typedef nn::util::BitPack32::Field<  7,  1, bool     > Blen256;
    typedef nn::util::BitPack32::Field<  6,  1, bool     > Blen128;
    typedef nn::util::BitPack32::Field<  5,  1, bool     > Blen64;
    typedef nn::util::BitPack32::Field<  4,  1, bool     > Blen32;
    typedef nn::util::BitPack32::Field<  3,  1, bool     > Blen16;
    typedef nn::util::BitPack32::Field<  2,  1, bool     > Blen8;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Blen4;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Fb;
};

// DMA_CHx_Control
struct DmaControl : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 25,  7, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 24,  1, bool     > Sph;
    typedef nn::util::BitPack32::Field< 21,  3, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field< 18,  3, uint32_t > Dsl;
    typedef nn::util::BitPack32::Field< 17,  1, bool     > Reserved2;
    typedef nn::util::BitPack32::Field< 16,  1, bool     > Pblx8;
    typedef nn::util::BitPack32::Field< 14,  2, uint32_t > Reserved3;
    typedef nn::util::BitPack32::Field<  0, 14, uint32_t > Mss;
};

// DMA_CHx_TX_Control
struct DmaTxControl : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 28,  4, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 24,  4, uint32_t > Tqos;
    typedef nn::util::BitPack32::Field< 22,  2, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field< 16,  6, uint32_t > Txpbl;
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Ipbl;
    typedef nn::util::BitPack32::Field< 13,  2, uint32_t > Reserved2;
    typedef nn::util::BitPack32::Field< 12,  1, bool     > Tse;
    typedef nn::util::BitPack32::Field<  5,  7, uint32_t > Reserved3;
    typedef nn::util::BitPack32::Field<  4,  1, bool     > Osp;
    typedef nn::util::BitPack32::Field<  1,  3, uint32_t > Tcw;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > St;
};

// DMA_CHx_RX_Control
struct DmaRxControl : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 31,  1, bool     > Rpf;
    typedef nn::util::BitPack32::Field< 28,  3, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 24,  4, uint32_t > Rqos;
    typedef nn::util::BitPack32::Field< 22,  2, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field< 16,  6, uint32_t > Rxpbl;
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Reserved2;
    typedef nn::util::BitPack32::Field<  1, 14, uint32_t > Rbsz;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Sr;
};

// DMA_CHx_Interrupt_Enable
struct DmaInterruptEnable : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 16, 16, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Nie;
    typedef nn::util::BitPack32::Field< 14,  1, bool     > Aie;
    typedef nn::util::BitPack32::Field< 13,  1, bool     > Cdee;
    typedef nn::util::BitPack32::Field< 12,  1, bool     > Fbee;
    typedef nn::util::BitPack32::Field< 11,  1, bool     > Erie;
    typedef nn::util::BitPack32::Field< 10,  1, bool     > Etie;
    typedef nn::util::BitPack32::Field<  9,  1, bool     > Rwte;
    typedef nn::util::BitPack32::Field<  8,  1, bool     > Rse;
    typedef nn::util::BitPack32::Field<  7,  1, bool     > Rbue;
    typedef nn::util::BitPack32::Field<  6,  1, bool     > Rie;
    typedef nn::util::BitPack32::Field<  3,  3, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  2,  1, bool     > Tbue;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Txse;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Tie;
};

// DMA_CHx_Status
struct DmaStatus : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 22, 10, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 19,  3, uint32_t > Reb;
    typedef nn::util::BitPack32::Field< 16,  3, uint32_t > Teb;
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Nis;
    typedef nn::util::BitPack32::Field< 14,  1, bool     > Ais;
    typedef nn::util::BitPack32::Field< 13,  1, bool     > Cde;
    typedef nn::util::BitPack32::Field< 12,  1, bool     > Fbe;
    typedef nn::util::BitPack32::Field< 11,  1, bool     > Eri;
    typedef nn::util::BitPack32::Field< 10,  1, bool     > Eti;
    typedef nn::util::BitPack32::Field<  9,  1, bool     > Rwt;
    typedef nn::util::BitPack32::Field<  8,  1, bool     > Rps;
    typedef nn::util::BitPack32::Field<  7,  1, bool     > Rbu;
    typedef nn::util::BitPack32::Field<  6,  1, bool     > Ri;
    typedef nn::util::BitPack32::Field<  3,  3, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  2,  1, bool     > Tbu;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Tps;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Ti;
};

// EQOS MTL registers address offset
const uint32_t RegOffset_MtlOperationMode             = 0x0c00;
const uint32_t RegOffset_MtlRxqDmaMap0      = 0x0c30;
const uint32_t RegOffset_MtlRxqDmaMap1      = 0x0c34;
const uint32_t RegOffset_MtlTxq0OperationMode         = 0x0d00;
const uint32_t RegOffset_MtlQ0InterruptControlStatus  = 0x0d2c;
const uint32_t RegOffset_MtlRxq0OperationMode         = 0x0d30;

// MTL_Operation_Mode
struct MtlOperationMode : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 10, 22, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field<  9,  1, bool     > Cntclr;
    typedef nn::util::BitPack32::Field<  8,  1, bool     > Cntprst;
    typedef nn::util::BitPack32::Field<  7,  1, bool     > Reserved1;
    typedef nn::util::BitPack32::Field<  5,  2, uint32_t > Schalg;
    typedef nn::util::BitPack32::Field<  3,  2, uint32_t > Reserved2;
    typedef nn::util::BitPack32::Field<  2,  1, bool     > Raa;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Dtxsts;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Reserved3;
};

// MTL_TxQ0_Operation_Mode
struct MtlTxq0OperationMode : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 25,  7, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 16,  9, uint32_t > Tqs;
    typedef nn::util::BitPack32::Field<  7,  9, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  4,  3, uint32_t > Ttc;
    typedef nn::util::BitPack32::Field<  2,  2, uint32_t > Txqen;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Tsf;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Ftq;
};

// MTL_RxQ0_Operation_Mode
struct MtlRxq0OperationMode : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 30,  2, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 20, 10, uint32_t > Rqs;
    typedef nn::util::BitPack32::Field< 14,  6, uint32_t > Rfd;
    typedef nn::util::BitPack32::Field<  8,  6, uint32_t > Rfa;
    typedef nn::util::BitPack32::Field<  7,  1, bool     > Ehfc;
    typedef nn::util::BitPack32::Field<  6,  1, bool     > DisTcpEf;
    typedef nn::util::BitPack32::Field<  5,  1, bool     > Rsf;
    typedef nn::util::BitPack32::Field<  4,  1, bool     > Fep;
    typedef nn::util::BitPack32::Field<  3,  1, bool     > Fup;
    typedef nn::util::BitPack32::Field<  2,  1, bool     > Reserved1;
    typedef nn::util::BitPack32::Field<  0,  2, uint32_t > Rtc;
};

// MTL_Interrupt_Control_Status
struct MtlInterruptControlStatus : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 25,  7, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 24,  1, bool     > Rxoie;
    typedef nn::util::BitPack32::Field< 17,  7, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field< 16,  1, bool     > Rxovfis;
    typedef nn::util::BitPack32::Field< 10,  6, uint32_t > Reserved2;
    typedef nn::util::BitPack32::Field<  9,  1, bool     > Abpsie;
    typedef nn::util::BitPack32::Field<  8,  1, bool     > Txuie;
    typedef nn::util::BitPack32::Field<  2,  6, uint32_t > Reserved3;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Abpsis;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Txunfis;
};

// EQOS MAC registers address offset
const uint32_t RegOffset_MacConfiguration       = 0x0000;
const uint32_t RegOffset_MacPacketFilter        = 0x0008;
const uint32_t RegOffset_MacTxFlowCtrl          = 0x0070;
const uint32_t RegOffset_MacRxFlowCtrl          = 0x0090;
const uint32_t RegOffset_MacRxqCtrl0            = 0x00a0;
const uint32_t RegOffset_MacInterruptStatus     = 0x00b0;
const uint32_t RegOffset_MacInterruptEnable     = 0x00b4;
const uint32_t RegOffset_MacRxTxStatus          = 0x00b8;
const uint32_t RegOffset_MacPhyifControlStatus  = 0x00f8;
const uint32_t RegOffset_MacMdioAddress         = 0x0200;
const uint32_t RegOffset_MacMdioData            = 0x0204;
const uint32_t RegOffset_MacAddress0High        = 0x0300;
const uint32_t RegOffset_MacAddress0Low         = 0x0304;

// MAC_Configuration
struct MacConfiguration : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 31,  1, bool     > Arpen;
    typedef nn::util::BitPack32::Field< 28,  3, uint32_t > Sarc;
    typedef nn::util::BitPack32::Field< 27,  1, bool     > Ipc;
    typedef nn::util::BitPack32::Field< 24,  3, uint32_t > Ipg;
    typedef nn::util::BitPack32::Field< 23,  1, bool     > Gpslce;
    typedef nn::util::BitPack32::Field< 22,  1, bool     > S2kp;
    typedef nn::util::BitPack32::Field< 21,  1, bool     > Cst;
    typedef nn::util::BitPack32::Field< 20,  1, bool     > Acs;
    typedef nn::util::BitPack32::Field< 19,  1, bool     > Wd;
    typedef nn::util::BitPack32::Field< 18,  1, bool     > Be;
    typedef nn::util::BitPack32::Field< 17,  1, bool     > Jd;
    typedef nn::util::BitPack32::Field< 16,  1, bool     > Je;
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Ps;
    typedef nn::util::BitPack32::Field< 14,  1, bool     > Fes;
    typedef nn::util::BitPack32::Field< 13,  1, bool     > Dm;
    typedef nn::util::BitPack32::Field< 12,  1, bool     > Lm;
    typedef nn::util::BitPack32::Field< 11,  1, bool     > Ecrsfd;
    typedef nn::util::BitPack32::Field< 10,  1, bool     > Do;
    typedef nn::util::BitPack32::Field<  9,  1, bool     > Dcrs;
    typedef nn::util::BitPack32::Field<  8,  1, bool     > Dr;
    typedef nn::util::BitPack32::Field<  7,  1, bool     > Reserved;
    typedef nn::util::BitPack32::Field<  5,  2, uint32_t > Bl;
    typedef nn::util::BitPack32::Field<  4,  1, bool     > Dc;
    typedef nn::util::BitPack32::Field<  2,  2, uint32_t > Prelen;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Te;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Re;
};

// MAC_Tx_Flow_Control
struct MacTxFlowCtrl : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 16, 16, uint32_t > Pt;
    typedef nn::util::BitPack32::Field<  8,  8, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field<  7,  1, bool     > Dzpq;
    typedef nn::util::BitPack32::Field<  4,  3, uint32_t > Plt;
    typedef nn::util::BitPack32::Field<  2,  2, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Tfe;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Fcb;
};

// MAC_Rx_Flow_Control
struct MacRxFlowCtrl : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field<  9, 23, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field<  8,  1, bool     > Pfce;
    typedef nn::util::BitPack32::Field<  2,  6, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Up;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Rfe;
};

// MAX_RxQ_Ctrl0
struct MacRxqCtrl0 : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 16, 16, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 14,  2, uint32_t > Rxq7en;
    typedef nn::util::BitPack32::Field< 12,  2, uint32_t > Rxq6en;
    typedef nn::util::BitPack32::Field< 10,  2, uint32_t > Rxq5en;
    typedef nn::util::BitPack32::Field<  8,  2, uint32_t > Rxq4en;
    typedef nn::util::BitPack32::Field<  6,  2, uint32_t > Rxq3en;
    typedef nn::util::BitPack32::Field<  4,  2, uint32_t > Rxq2en;
    typedef nn::util::BitPack32::Field<  2,  2, uint32_t > Rxq1en;
    typedef nn::util::BitPack32::Field<  0,  2, uint32_t > Rxq0en;
};

// MAC_Interrupt_Status
struct MacInterruptStatus : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 16, 16, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Gpiis;
    typedef nn::util::BitPack32::Field< 14,  1, bool     > Rxstsi;
    typedef nn::util::BitPack32::Field< 13,  1, bool     > Txstsi;
    typedef nn::util::BitPack32::Field< 12,  1, bool     > Tis;
    typedef nn::util::BitPack32::Field< 11,  1, bool     > Mmcrxipis;
    typedef nn::util::BitPack32::Field< 10,  1, bool     > Mmctis;
    typedef nn::util::BitPack32::Field<  9,  1, bool     > Mmcrxis;
    typedef nn::util::BitPack32::Field<  8,  1, bool     > Mmcis;
    typedef nn::util::BitPack32::Field<  6,  2, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  5,  1, bool     > Lpiis;
    typedef nn::util::BitPack32::Field<  4,  1, bool     > Pmtis;
    typedef nn::util::BitPack32::Field<  3,  1, bool     > Phyis;
    typedef nn::util::BitPack32::Field<  2,  1, bool     > Pcsancis;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Pcslchgis;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Rgsmiiis;
};

// MAC_Interrupt_Enable
struct MacInterruptEnable : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 15, 17, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 14,  1, bool     > Rxstsie;
    typedef nn::util::BitPack32::Field< 13,  1, bool     > Txstsie;
    typedef nn::util::BitPack32::Field< 12,  1, bool     > Tsie;
    typedef nn::util::BitPack32::Field<  6,  6, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  5,  1, bool     > Lpiie;
    typedef nn::util::BitPack32::Field<  4,  1, bool     > Pmtie;
    typedef nn::util::BitPack32::Field<  3,  1, bool     > Phyie;
    typedef nn::util::BitPack32::Field<  2,  1, bool     > Pcsancie;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Pcslchgie;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Rgsmiiie;
};

// MAC_PHYIF_Control_Status
struct MacPhyifControlStatus : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 22, 10, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 21,  1, bool     > Falscardet;
    typedef nn::util::BitPack32::Field< 20,  1, bool     > Jabto;
    typedef nn::util::BitPack32::Field< 19,  1, bool     > Lnksts;
    typedef nn::util::BitPack32::Field< 17,  2, uint32_t > Lnkspeed;
    typedef nn::util::BitPack32::Field< 16,  1, bool     > Lnkmod;
    typedef nn::util::BitPack32::Field<  2, 14, uint32_t > Reserved1;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > Lud;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Tc;
};

// MAC_MDIO_Address
struct MacMdioAddress : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 28,  4, uint32_t > Reserved0;
    typedef nn::util::BitPack32::Field< 27,  1, bool     > Pse;
    typedef nn::util::BitPack32::Field< 26,  1, bool     > Btb;
    typedef nn::util::BitPack32::Field< 21,  5, uint32_t > Pa;
    typedef nn::util::BitPack32::Field< 16,  5, uint32_t > Rda;
    typedef nn::util::BitPack32::Field< 15,  1, bool     > Reserved1;
    typedef nn::util::BitPack32::Field< 12,  3, uint32_t > Ntc;
    typedef nn::util::BitPack32::Field<  8,  4, uint32_t > Cr;
    typedef nn::util::BitPack32::Field<  5,  3, uint32_t > Reserved2;
    typedef nn::util::BitPack32::Field<  4,  1, bool     > Skap;
    typedef nn::util::BitPack32::Field<  2,  2, uint32_t > Goc;
    typedef nn::util::BitPack32::Field<  1,  1, bool     > C45e;
    typedef nn::util::BitPack32::Field<  0,  1, bool     > Gb;
};

// MacMdioAddress::Pa に設定するべき値。実験で確認済。
const int MacPhysicalLayerAddress = 0;

// MAC_MDIO_Data
struct MacMdioData : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 16, 32, uint32_t > Ra;
    typedef nn::util::BitPack32::Field<  0, 16, uint32_t > Gd;
};

// MAC_Address0_High
struct MacAddress0High : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 31,  1, bool     > Ae;
    typedef nn::util::BitPack32::Field< 19, 12, uint32_t > Reserved;
    typedef nn::util::BitPack32::Field< 16,  3, uint32_t > Dcs;
    typedef nn::util::BitPack32::Field<  8,  8, uint8_t  > Addr5;
    typedef nn::util::BitPack32::Field<  0,  8, uint8_t  > Addr4;
};

// MAC_Address0_Low
struct MacAddress0Low : public nn::util::BitPack32
{
    typedef nn::util::BitPack32::Field< 24,  8, uint8_t  > Addr3;
    typedef nn::util::BitPack32::Field< 16,  8, uint8_t  > Addr2;
    typedef nn::util::BitPack32::Field<  8,  8, uint8_t  > Addr1;
    typedef nn::util::BitPack32::Field<  0,  8, uint8_t  > Addr0;
};

} // tx2
} // device
} // eth
} // nn
