﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <functional>
#include <mutex>

#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/dd.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/eth/detail/eth_Log.h>

#include "../../util/eth_Util.h"
#include "../eth_Allocator.h"
#include "../mii_Registers.h"
#include "eth_SynopsysDwcEtherQos.h"
#include "eth_SynopsysDwcEtherQos.reg.h"
#include "eth_SynopsysDwcEtherQos.descriptor.h"
#include "pcv/eth_Pcv.h"

namespace nn {
namespace eth {
namespace device {
namespace tx2 {

// ------------------------------------------------------------------------------------------------
// Utility

namespace {

// 4KB ページアラインされたバッファを確保
NN_FORCEINLINE void* AllocatePageAlignedBuffer(size_t size) NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size, nn::os::MemoryPageSize);
}

// EQOS のクロック (156.25MHz) 4クロック以上待つ (同じレジスタに連続して書き込む場合に必要、TRM 40.5 参照)
NN_FORCEINLINE void SleepForRegWriteCompletion() NN_NOEXCEPT
{
    nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(1));
}

// 4GB 境界をまたいでいないかのチェック
NN_FORCEINLINE bool IsInSame4GPage(nn::dd::PhysicalAddress a, nn::dd::PhysicalAddress b) NN_NOEXCEPT
{
    return ADDR_HIGH(a) == ADDR_HIGH(b);
}

} // namespace

// ------------------------------------------------------------------------------------------------
// Public interface

SynopsysDwcEtherQos::SynopsysDwcEtherQos()
NN_NOEXCEPT :
    Gmii(Tx2InterfaceName, PhyThreadPollIntervalMs),
    m_TxRequestEvent(nn::os::EventClearMode_AutoClear),
    m_PhyTimerEvent(nn::os::EventClearMode_AutoClear),
    m_PhyMutex(true),
    m_TxDescMutex(false),
    m_TxDescHead(0),
    m_TxDescTail(0),
    m_RxDescHead(0),
    m_AvailableTxDescCount(0)
{
}

SynopsysDwcEtherQos::~SynopsysDwcEtherQos()
NN_NOEXCEPT
{
}

nn::Result SynopsysDwcEtherQos::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    // Clock and reset initialization
    pcv::Initialize();
    NN_RESULT_DO(pcv::SetClockEnabled(pcv::Module_EqosClockSlaveBus, true));
    NN_RESULT_DO(pcv::SetClockEnabled(pcv::Module_EqosClockMasterBus, true));
    NN_RESULT_DO(pcv::SetClockEnabled(pcv::Module_EqosClockRx, true));
    NN_RESULT_DO(pcv::SetClockEnabled(pcv::Module_EqosClockPtpRef, true));
    NN_RESULT_DO(pcv::SetClockEnabled(pcv::Module_EqosClockTx, true));
    NN_RESULT_DO(pcv::SetReset(pcv::Module_EqosReset, false));

    m_pTxDescRing = reinterpret_cast<TxDesc*>(AllocatePageAlignedBuffer(TxDescTotalSize));
    m_pRxDescRing = reinterpret_cast<RxDesc*>(AllocatePageAlignedBuffer(RxDescTotalSize));

    m_pTxBuffer = reinterpret_cast<TxBuffer*>(AllocatePageAlignedBuffer(TxBufferTotalSize));
    m_pRxBuffer = reinterpret_cast<RxBuffer*>(AllocatePageAlignedBuffer(RxBufferTotalSize));

    // Get virtual address of I/O registers
    m_pEqos = util::GetVirtualAddress(EqosPhysAddr, EqosSize);

    // Get physical address of descriptors
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::dd::QuerySinglePhysicalAddress(&m_TxDescRingPhysAddr, m_pTxDescRing, sizeof(m_pTxDescRing)));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::dd::QuerySinglePhysicalAddress(&m_RxDescRingPhysAddr, m_pRxDescRing, sizeof(m_pRxDescRing)));
    m_TxDescRingTailPhysAddr = m_TxDescRingPhysAddr + TxDescTotalSize;
    m_RxDescRingTailPhysAddr = m_RxDescRingPhysAddr + RxDescTotalSize;

    // Descriptor rings should not cross the 4GB boundary (see Parker TRM 40.5.1)
    // TODO: SMMU を使ってバッファが 4G 境界を超えないことを保証
    NN_ABORT_UNLESS(IsInSame4GPage(m_TxDescRingPhysAddr, m_TxDescRingTailPhysAddr - 1));
    NN_ABORT_UNLESS(IsInSame4GPage(m_RxDescRingPhysAddr, m_RxDescRingTailPhysAddr - 1));

    // Get physical address of buffers
    for (int i = 0; i < TxDescRingLength; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::dd::QuerySinglePhysicalAddress(&m_TxBufferPhysAddr[i], &m_pTxBuffer[i], TxBufferSize));
    }
    for (int i = 0; i < RxDescRingLength; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::dd::QuerySinglePhysicalAddress(&m_RxBufferPhysAddr[i], &m_pRxBuffer[i], RxBufferSize));
    }

    // Event initialization
    nn::os::InitializeMultiWait(&m_InterruptMultiWait);
    for (int i = 0; i < InterruptCount; i++)
    {
        nn::os::InitializeInterruptEvent(&m_InterruptEvent[i], InterruptNameTable[i], nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&m_InterruptEventHolder[i], &m_InterruptEvent[i]);
        nn::os::LinkMultiWaitHolder(&m_InterruptMultiWait, &m_InterruptEventHolder[i]);
        nn::os::SetMultiWaitHolderUserData(&m_InterruptEventHolder[i], static_cast<uintptr_t>(i));
    }

    InitializeMacAddress();
    InitializeDesc();
    InitializeBuffer();
    InitializeDma();
    InitializeMtl();
    InitializeMac();

    NN_RESULT_DO(Gmii::Initialize());

    const nn::Result result = SocketIf::Initialize(Gmii::m_Name, Gmii::m_Instance, Gmii::m_PhyMacAddress);
    if (result.IsFailure())
    {
        Gmii::Finalize();
        return result;
    }

    m_PhyThread.Initialize(
        std::bind(&SynopsysDwcEtherQos::PhyThread, this),
        NN_SYSTEM_THREAD_PRIORITY(socket, EthPhy),
        NN_SYSTEM_THREAD_NAME(socket, EthPhy));

    m_InterruptThread.Initialize(
        std::bind(&SynopsysDwcEtherQos::InterruptThread, this),
        NN_SYSTEM_THREAD_PRIORITY(socket, EthEvents),
        NN_SYSTEM_THREAD_NAME(socket, EthEvents));

    m_TxThread.Initialize(
        std::bind(&SynopsysDwcEtherQos::TxThread, this),
        NN_SYSTEM_THREAD_PRIORITY(socket, EthInterfaceEgress),
        NN_SYSTEM_THREAD_NAME(socket, EthInterfaceEgress));

    return ResultSuccess();
}

void SynopsysDwcEtherQos::Finalize() NN_NOEXCEPT
{
    NN_ABORT("%s is not implemented.\n", __FUNCTION__);
}

// ------------------------------------------------------------------------------------------------
// Debug functions

#if defined(NN_DETAIL_ETH_ENABLE_DEBUG_FUNCTIONS)
nn::dd::PhysicalAddress SynopsysDwcEtherQos::GetPhysicalAddress(uintptr_t base, uint32_t offset)
{
    if (base == m_pEqos)
    {
        return EqosPhysAddr + offset;
    }
    else
    {
        NN_DETAIL_ETH_ERROR("Cannot convert base address %p.\n", base);
        return 0;
    }
}

void SynopsysDwcEtherQos::Print(uintptr_t addr, size_t size)
{
    NN_SDK_ASSERT_EQUAL(addr % 0x10, 0);
    NN_SDK_ASSERT_EQUAL(size % 0x10, 0);

    bool prevDump = false;
    uint32_t prevRegs[4];
    uint32_t regs[4];

    for (int offset = 0; offset < size; offset += 0x10)
    {
        regs[0] = util::Read32(addr, offset + 0);
        regs[1] = util::Read32(addr, offset + 4);
        regs[2] = util::Read32(addr, offset + 8);
        regs[3] = util::Read32(addr, offset + 12);

        if (offset == 0 ||
            regs[0] != prevRegs[0] ||
            regs[1] != prevRegs[1] ||
            regs[2] != prevRegs[2] ||
            regs[3] != prevRegs[3])
        {
            NN_DETAIL_ETH_INFO("%06x %08x %08x %08x %08x\n", offset, regs[0], regs[1], regs[2], regs[3]);

            std::memcpy(prevRegs, regs, sizeof(regs));
            prevDump = true;
        }
        else
        {
            if (prevDump)
            {
                NN_DETAIL_ETH_INFO("*\n");
            }
            prevDump = false;
        }
    }
}

void SynopsysDwcEtherQos::PrintEqosReg() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    Print(m_pEqos, PrintableEqosRegSize);
}

void SynopsysDwcEtherQos::PrintTxDesc(int index) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    InvalidateTxDescCache(index);
    Print(reinterpret_cast<uintptr_t>(&m_pTxDescRing[index]), sizeof(TxDesc));
}

void SynopsysDwcEtherQos::PrintTxDescAll() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    for (int i = 0; i < TxDescRingLength; i++)
    {
        InvalidateTxDescCache(i);
    }
    Print(reinterpret_cast<uintptr_t>(m_pTxDescRing), TxDescTotalSize);
}

void SynopsysDwcEtherQos::PrintRxDesc(int index) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    InvalidateRxDescCache(index);
    Print(reinterpret_cast<uintptr_t>(&m_pRxDescRing[index]), sizeof(RxDesc));
}

void SynopsysDwcEtherQos::PrintRxDescAll() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    for (int i = 0; i < RxDescRingLength; i++)
    {
        InvalidateRxDescCache(i);
    }
    Print(reinterpret_cast<uintptr_t>(m_pRxDescRing), sizeof(RxDesc) * RxDescRingLength);
}

void SynopsysDwcEtherQos::PrintTxBuffer(int index) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    nn::dd::InvalidateDataCache(&m_pTxBuffer[index], sizeof(m_pTxBuffer[index]));
    Print(reinterpret_cast<uintptr_t>(&m_pTxBuffer[index]), sizeof(m_pTxBuffer[index]));
}

void SynopsysDwcEtherQos::PrintRxBuffer(int index) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    nn::dd::InvalidateDataCache(&m_pRxBuffer[index], sizeof(m_pRxBuffer[index]));
    Print(reinterpret_cast<uintptr_t>(&m_pRxBuffer[index]), sizeof(m_pRxBuffer[index]));
}

void SynopsysDwcEtherQos::PrintMiiRegister() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    uint16_t value[PrintableMiiRegSize];
    for (uint16_t i = 0; i < PrintableMiiRegSize; i++)
    {
        PhyReadReg(i, &value[i]);
    }
    for (uint16_t i = 0; i < PrintableMiiRegSize; i+=4)
    {
        NN_DETAIL_ETH_INFO("PHY %2d: %04x %04x %04x %04x\n", i, value[i], value[i + 1], value[i + 2], value[i + 3]);
    }
}
#endif

// ------------------------------------------------------------------------------------------------
// Descriptor control

void SynopsysDwcEtherQos::InitializeTxDesc(int index) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    TxDesc desc;
    desc.tdes0.Clear();
    desc.tdes1.Clear();
    desc.tdes2.Clear();
    desc.tdes3.Clear();
    desc.tdes3.Set<Rdes3::Own>(Owner_Cpu);

    WriteTxDesc(&desc, index);
    FlushTxDescCache(index);
}

void SynopsysDwcEtherQos::InitializeRxDesc(int index) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    // Initialize RxDesc (Read format)
    RxDesc desc;

    desc.rdes0.Clear();
    desc.rdes0.Set<Rdes0::BufferAddressLow>(ADDR_LOW(m_RxBufferPhysAddr[index]));

    desc.rdes1.Clear();
    desc.rdes1.Set<Rdes1::BufferAddressHigh>(ADDR_HIGH(m_RxBufferPhysAddr[index]));

    desc.rdes2.Clear();

    desc.rdes3.Clear();
    desc.rdes3.Set<Rdes3::Ioc>(true); // Interrupt on completion
    desc.rdes3.Set<Rdes3::Buf1v>(true);

    WriteRxDesc(&desc, index);

    nn::dd::EnsureMemoryAccess();

    desc.rdes3.Set<Rdes3::Own>(Owner_Dma);
    WriteRdes3(&desc.rdes3, index);

    FlushRxDescCache(index);
}

void SynopsysDwcEtherQos::PrepareTxDescToSend(int index, size_t dataSize) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    // Initialize TxDesc (Read format)
    TxDesc desc;

    desc.tdes0.Clear();
    desc.tdes0.Set<Tdes0::BufferAddressLow>(ADDR_LOW(m_TxBufferPhysAddr[index]));

    desc.tdes1.Clear();
    desc.tdes1.Set<Tdes1::BufferAddressHigh>(ADDR_HIGH(m_TxBufferPhysAddr[index]));

    desc.tdes2.Clear();
    desc.tdes2.Set<Tdes2::BufferLength>(dataSize);
    desc.tdes2.Set<Tdes2::Vtir>(0);           // Do not add a VLAN tag.
    desc.tdes2.Set<Tdes2::Buffer2Length>(0);  // Do not use Buffer2
    desc.tdes2.Set<Tdes2::Ttse>(0);           // Disable Transmit Timestamp
    desc.tdes2.Set<Tdes2::Ioc>(1);            // Enable interrupt on comletion

    desc.tdes3.Clear();
    desc.tdes3.Set<Tdes3::Fd>(1);     // First Descriptor
    desc.tdes3.Set<Tdes3::Ld>(1);     // Last Descriptor
    desc.tdes3.Set<Tdes3::Cpc>(0);    // Enable CRC and Pad Insertion
    desc.tdes3.Set<Tdes3::Saic>(0);   // Do not insert the Source Address
    desc.tdes3.Set<Tdes3::Thl>(0);
    desc.tdes3.Set<Tdes3::Tse>(0);    // Disable TCP Segmentation
    desc.tdes3.Set<Tdes3::Cic>(0);    // Disable Checksum Insertion

    WriteTxDesc(&desc, index);

    nn::dd::EnsureMemoryAccess();

    desc.tdes3.Set<Tdes3::Own>(Owner_Dma);
    WriteTdes3(&desc.tdes3, index);

    FlushTxDescCache(index);
}

void SynopsysDwcEtherQos::PrepareRxDescToReceive(int index) NN_NOEXCEPT
{
    InitializeRxDesc(index);
}

// DMA から返ってきた送信ディスクリプタをチェックして、エラーが無ければ true を返す
bool SynopsysDwcEtherQos::CheckTxDesc(const TxDesc& desc) NN_NOEXCEPT
{
    if (desc.tdes3.IsAnyBitOn(Tdes3::Ctxt::Mask | Tdes3::Es::Mask))
    {
        return false;
    }

    return true;
}

// DMA から返ってきた受信ディスクリプタをチェックして、エラーが無ければ true を返す
bool SynopsysDwcEtherQos::CheckRxDesc(const RxDesc& desc) NN_NOEXCEPT
{
    if (desc.rdes3.IsAnyBitOff(Rdes3::Ld::Mask | Rdes3::Fd::Mask))
    {
        return false;
    }

    if (desc.rdes3.IsAnyBitOn(
        Rdes3::Ctxt::Mask |
        Rdes3::Ce::Mask |
        Rdes3::Gp::Mask |
        Rdes3::Rwt::Mask |
        Rdes3::Oe::Mask |
        Rdes3::Re::Mask |
        Rdes3::De::Mask))
    {
        return false;
    }

    if (desc.rdes2.IsAnyBitOn(Rdes2::Daf::Mask | Rdes2::Saf::Mask))
    {
        return false;
    }

    if (desc.rdes1.IsAnyBitOn(Rdes1::Ipce::Mask))
    {
        return false;
    }

    return true;
}

void SynopsysDwcEtherQos::PrintTxDescError(const TxDesc& desc) NN_NOEXCEPT
{
    if (desc.tdes3.Get<Tdes3::Ctxt>())
    {
        NN_DETAIL_ETH_DEBUG_INFO("Context descriptor.\n");
    }
    if (desc.tdes3.Get<Tdes3::Es>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("TX Error: TDES3=%08x\n", desc.tdes3.GetMaskedBits(0xffffffff));
    }
}

void SynopsysDwcEtherQos::PrintRxDescError(const RxDesc& desc) NN_NOEXCEPT
{
    if (!desc.rdes3.Get<Rdes3::Ld>() || !desc.rdes3.Get<Rdes3::Fd>())
    {
        NN_DETAIL_ETH_WARN("Long packet received, but it's not supported.\n");
    }
    if (desc.rdes3.Get<Rdes3::Ctxt>())
    {
        NN_DETAIL_ETH_DEBUG_INFO("Context descriptor.\n");
    }
    if (desc.rdes3.Get<Rdes3::Ce>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("CRC Error.\n");
    }
    if (desc.rdes3.Get<Rdes3::Gp>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("Giant Packet.\n");
    }
    if (desc.rdes3.Get<Rdes3::Rwt>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("Receive Watchdog Timeout.\n");
    }
    if (desc.rdes3.Get<Rdes3::Oe>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("Overflow Error.\n");
    }
    if (desc.rdes3.Get<Rdes3::Re>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("Receive Error.\n");
    }
    if (desc.rdes3.Get<Rdes3::De>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("Dribble bit Error.\n");
    }
    if (desc.rdes2.Get<Rdes2::Daf>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("Destination MAC address filter fail.\n");
    }
    if (desc.rdes2.Get<Rdes2::Saf>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("Source MAC address filter fail.\n");
    }
    if (desc.rdes1.Get<Rdes1::Ipce>())
    {
        NN_DETAIL_ETH_DEBUG_WARN("IP payload error.\n");
    }
}

// ------------------------------------------------------------------------------------------------
// Initialize

void SynopsysDwcEtherQos::InitializeMacAddress() NN_NOEXCEPT
{
    // TORIAEZU: ランダムな MAC アドレスを生成
    std::srand(nn::os::GetSystemTick().GetInt64Value());

    Gmii::m_PhyMacAddress[0] = MacAddressVendorCode[0];
    Gmii::m_PhyMacAddress[1] = MacAddressVendorCode[1];
    Gmii::m_PhyMacAddress[2] = MacAddressVendorCode[2];
    Gmii::m_PhyMacAddress[3] = static_cast<uint8_t>(rand());
    Gmii::m_PhyMacAddress[4] = static_cast<uint8_t>(rand());
    Gmii::m_PhyMacAddress[5] = static_cast<uint8_t>(rand());
}

void SynopsysDwcEtherQos::InitializeDesc() NN_NOEXCEPT
{
    for (int i = 0; i < TxDescRingLength; i++)
    {
        InitializeTxDesc(i);
    }
    m_AvailableTxDescCount = TxDescRingLength;

    for (int i = 0; i < RxDescRingLength; i++)
    {
        InitializeRxDesc(i);
    }
}

void SynopsysDwcEtherQos::InitializeBuffer() NN_NOEXCEPT
{
    // バッファ内のデータは初期化不要だが、受信バッファをキャッシュから追い出す必要がある
    nn::dd::InvalidateDataCache(m_pRxBuffer, sizeof(RxBuffer) * RxDescRingLength);
}

void SynopsysDwcEtherQos::InitializeDma() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    // See Parker TRM 40.5.1

    // Software reset
    util::SetBitOn32(m_pEqos, RegOffset_DmaMode, DmaMode::Swr::Mask);

    // DmaMode.Swr == 0 になるのを待つ
    nn::TimeSpan elapsedTime = 0;
    while (true)
    {
        const auto reg = util::Read32(m_pEqos, RegOffset_DmaMode);
        if (!(reg & DmaMode::Swr::Mask))
        {
            break;
        }

        nn::os::SleepThread(DmaResetPollInterval);
        if ((elapsedTime += DmaResetPollInterval) > DmaResetPollTimeout)
        {
            NN_ABORT("Timeout (%s)\n", __FUNCTION__);
        }
    }

    // DMA_SysBus_Mode
    util::Write32(m_pEqos, RegOffset_DmaSysbusMode, DmaSysBusMode::Eame::Mask);

    // Initialize descriptor related registers
    util::Write32(m_pEqos, RegOffset_DmaCh0TxDescRingLength, TxDescRingLength - 1);
    util::Write32(m_pEqos, RegOffset_DmaCh0RxDescRingLength, RxDescRingLength - 1);

    util::Write32(m_pEqos, RegOffset_DmaCh0TxDescRingListHaddress, ADDR_HIGH(m_TxDescRingPhysAddr));
    util::Write32(m_pEqos, RegOffset_DmaCh0TxDescRingListAddress, ADDR_LOW(m_TxDescRingPhysAddr));
    util::Write32(m_pEqos, RegOffset_DmaCh0TxDescTailPointer, ADDR_LOW(m_TxDescRingTailPhysAddr));

    util::Write32(m_pEqos, RegOffset_DmaCh0RxDescRingListHaddress, ADDR_HIGH(m_RxDescRingPhysAddr));
    util::Write32(m_pEqos, RegOffset_DmaCh0RxDescRingListAddress, ADDR_LOW(m_RxDescRingPhysAddr));
    util::Write32(m_pEqos, RegOffset_DmaCh0RxDescTailPointer, ADDR_LOW(m_RxDescRingTailPhysAddr));

    // Initialize control registers
    util::Write32(m_pEqos, RegOffset_DmaCh0Control, ((DescPaddingSize / 16) << DmaControl::Dsl::Pos)); // Dsl は 16 byte 単位で設定
    util::Write32(m_pEqos, RegOffset_DmaCh0TxControl, (DmaTxBurstLength << DmaTxControl::Txpbl::Pos));
    util::Write32(m_pEqos, RegOffset_DmaCh0RxControl,
        (DmaRxBurstLength << DmaRxControl::Rxpbl::Pos) |
        (RxBufferSize << DmaRxControl::Rbsz::Pos));

    // Interrupt Enable
    util::Write32(m_pEqos, RegOffset_DmaCh0InterruptEnable,
        DmaInterruptEnable::Nie::Mask |
        DmaInterruptEnable::Aie::Mask |
        DmaInterruptEnable::Cdee::Mask |
        DmaInterruptEnable::Fbee::Mask |
        DmaInterruptEnable::Tie::Mask |
        DmaInterruptEnable::Rie::Mask);

    nn::dd::EnsureMemoryAccess();
    SleepForRegWriteCompletion();

    // Start DMA
    util::SetBitOn32(m_pEqos, RegOffset_DmaCh0TxControl, DmaTxControl::St::Mask);
    SleepForRegWriteCompletion();

    util::SetBitOn32(m_pEqos, RegOffset_DmaCh0RxControl, DmaRxControl::Sr::Mask);
    util::DummyRead(m_pEqos, RegOffset_DmaCh0RxControl);
}

void SynopsysDwcEtherQos::InitializeMtl() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    // See Parker TRM 40.5.2

    util::Write32(m_pEqos, RegOffset_MtlOperationMode, (3 << MtlOperationMode::Schalg::Pos)); // 3: Strict priority algorithm

    // Use DMA channel 0 for all queues
    util::Write32(m_pEqos, RegOffset_MtlRxqDmaMap0, 0);
    util::Write32(m_pEqos, RegOffset_MtlRxqDmaMap1, 0);

    MtlTxq0OperationMode txOperationMode;
    txOperationMode.Clear();
    txOperationMode.Set<MtlTxq0OperationMode::Tqs>(MtlTxQueueSize / 256 - 1); // 設定値は 256 byte 刻み (0: 256 byte)
    txOperationMode.Set<MtlTxq0OperationMode::Txqen>(2); // 2: Enabled
    txOperationMode.Set<MtlTxq0OperationMode::Tsf>(1);
    util::Write32(m_pEqos, RegOffset_MtlTxq0OperationMode, txOperationMode.GetMaskedBits(0xffffffff));

    MtlRxq0OperationMode rxOperationMode;
    rxOperationMode.Clear();
    rxOperationMode.Set<MtlRxq0OperationMode::Rqs>(MtlRxQueueSize / 256 - 1); // 設定値は 256 byte 刻み (0: 256 byte)
    rxOperationMode.Set<MtlRxq0OperationMode::Rfd>(MtlRxFlowControlOffThreshold / 512 - 2); // 設定値は 512 byte 刻み (0: 1KB)
    rxOperationMode.Set<MtlRxq0OperationMode::Rfa>(MtlRxFlowControlOnThreshold / 512 - 2); // 設定値は 512 byte 刻み (0: 1KB)
    rxOperationMode.Set<MtlRxq0OperationMode::Ehfc>(1);
    rxOperationMode.Set<MtlRxq0OperationMode::Rsf>(1);
    util::Write32(m_pEqos, RegOffset_MtlRxq0OperationMode, rxOperationMode.GetMaskedBits(0xffffffff));

    util::Write32(m_pEqos, RegOffset_MtlQ0InterruptControlStatus, MtlInterruptControlStatus::Txuie::Mask);

    util::DummyRead(m_pEqos, RegOffset_MtlRxq0OperationMode);
}

void SynopsysDwcEtherQos::InitializeMac() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    // See Parker TRM 40.5.3

    // MAC アドレスの設定
    util::Write32(m_pEqos, RegOffset_MacAddress0High,
        1 << MacAddress0High::Ae::Pos |
        Gmii::m_PhyMacAddress[5] << MacAddress0High::Addr5::Pos |
        Gmii::m_PhyMacAddress[4] << MacAddress0High::Addr4::Pos);
    util::Write32(m_pEqos, RegOffset_MacAddress0Low,
        Gmii::m_PhyMacAddress[3] << MacAddress0Low::Addr3::Pos |
        Gmii::m_PhyMacAddress[2] << MacAddress0Low::Addr2::Pos |
        Gmii::m_PhyMacAddress[1] << MacAddress0Low::Addr1::Pos |
        Gmii::m_PhyMacAddress[0] << MacAddress0Low::Addr0::Pos);

    // 特別なパケットフィルタは使わない
    util::Write32(m_pEqos, RegOffset_MacPacketFilter, 0);

    // Flow control
    util::Write32(m_pEqos, RegOffset_MacTxFlowCtrl,
        MacTxFlowCtrl::Tfe::Mask |
        (0xffff << MacTxFlowCtrl::Pt::Pos)); // Pause time (Pt): MAX
    util::Write32(m_pEqos, RegOffset_MacRxFlowCtrl, MacRxFlowCtrl::Rfe::Mask);

    // Interrupt Enable
    util::Write32(m_pEqos, RegOffset_MacInterruptEnable,
        MacInterruptEnable::Rxstsie::Mask |
        MacInterruptEnable::Txstsie::Mask);

    // MAC_Configuration
    util::Write32(m_pEqos, RegOffset_MacConfiguration, MacConfiguration::Cst::Mask | MacConfiguration::Acs::Mask);

    // MAC_RxQ_Ctrl0
    util::Write32(m_pEqos, RegOffset_MacRxqCtrl0, 2 << MacRxqCtrl0::Rxq0en::Pos); // 2: Enabled

    nn::dd::EnsureMemoryAccess();
    SleepForRegWriteCompletion();

    // Start RX/TX
    util::SetBitOn32(m_pEqos, RegOffset_MacConfiguration, MacConfiguration::Te::Mask | MacConfiguration::Re::Mask);
    util::DummyRead(m_pEqos, RegOffset_MacConfiguration);
}

// ------------------------------------------------------------------------------------------------
// Phy access functions (overrides Gmii)

nn::Result SynopsysDwcEtherQos::PhyReset() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    std::lock_guard<nn::os::Mutex> lock(m_PhyMutex);

    uint16_t miiValue;

    PhyReadReg(MII_BMCR, &miiValue);
    PhyWriteReg(MII_BMCR, miiValue | BMCR_RESET);

    // Wait until BMCR_RESET becomes 0
    nn::TimeSpan elapsedTime = 0;
    while (true)
    {
        PhyReadReg(MII_BMCR, &miiValue);

        if ((miiValue & BMCR_RESET) == 0)
        {
            break;
        }

        nn::os::SleepThread(PhyResetPollInterval);
        if ((elapsedTime += PhyResetPollInterval) > PhyResetPollTimeout)
        {
            NN_ABORT("Timeout (%s)\n", __FUNCTION__);
        }
    }

    return ResultSuccess();
}

nn::Result SynopsysDwcEtherQos::PhyReadReg(uint16_t miiRegister, uint16_t* pMiiValue) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    std::lock_guard<nn::os::Mutex> lock(m_PhyMutex);

    MacMdioAddress mdioAddress;
    mdioAddress.Clear();
    mdioAddress.Set<MacMdioAddress::Pa>(MacPhysicalLayerAddress);
    mdioAddress.Set<MacMdioAddress::Rda>(miiRegister);
    mdioAddress.Set<MacMdioAddress::Goc>(3); // 3: Read
    mdioAddress.Set<MacMdioAddress::Gb>(1); // トランザクション前に 1 にする
    mdioAddress.Set<MacMdioAddress::Cr>(1); // CSR clock=100-150 MHz
    util::Write32(m_pEqos, RegOffset_MacMdioAddress, mdioAddress.GetMaskedBits(0xffffffff));

    // mdioAddress.Gb == 0 になるのを待つ
    nn::TimeSpan elapsedTime = 0;
    while (true)
    {
        const auto reg = util::Read32(m_pEqos, RegOffset_MacMdioAddress);
        if (!(reg & MacMdioAddress::Gb::Mask))
        {
            break;
        }

        nn::os::SleepThread(PhyReadPollInterval);
        if ((elapsedTime += PhyReadPollInterval) > PhyReadPollTimeout)
        {
            NN_ABORT("Timeout (%s)\n", __FUNCTION__);
        }
    }

    const uint32_t value = util::Read32(m_pEqos, RegOffset_MacMdioData);
    *pMiiValue = static_cast<uint16_t>((value & MacMdioData::Gd::Mask) >> MacMdioData::Gd::Pos);

    return ResultSuccess();
}

nn::Result SynopsysDwcEtherQos::PhyWriteReg(uint16_t miiRegister, uint16_t miiValue) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    std::lock_guard<nn::os::Mutex> lock(m_PhyMutex);

    MacMdioData mdioData;
    mdioData.Clear();
    mdioData.Set<MacMdioData::Gd>(miiValue);
    util::Write32(m_pEqos, RegOffset_MacMdioData, mdioData.GetMaskedBits(0xffffffff));

    MacMdioAddress mdioAddress;
    mdioAddress.Clear();
    mdioAddress.Set<MacMdioAddress::Pa>(MacPhysicalLayerAddress);
    mdioAddress.Set<MacMdioAddress::Rda>(miiRegister);
    mdioAddress.Set<MacMdioAddress::Goc>(1); // 1: Write
    mdioAddress.Set<MacMdioAddress::Gb>(1); // トランザクション前に 1 にする
    mdioAddress.Set<MacMdioAddress::Cr>(1); // CSR clock=100-150 MHz
    util::Write32(m_pEqos, RegOffset_MacMdioAddress, mdioAddress.GetMaskedBits(0xffffffff));

    // mdioAddress.Gb == 0 になるのを待つ
    nn::TimeSpan elapsedTime = 0;
    while (true)
    {
        const auto reg = util::Read32(m_pEqos, RegOffset_MacMdioAddress);
        if (!(reg & MacMdioAddress::Gb::Mask))
        {
            break;
        }

        nn::os::SleepThread(PhyWritePollInterval);
        if ((elapsedTime += PhyWritePollInterval) > PhyWritePollTimeout)
        {
            NN_ABORT("Timeout (%s)\n", __FUNCTION__);
        }
    }

    return ResultSuccess();
}

nn::Result SynopsysDwcEtherQos::PhyGetMedia(MediaType* pMediaType) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    std::lock_guard<nn::os::Mutex> lock(m_PhyMutex);

    uint16_t   anar;
    uint16_t   aner;
    uint16_t   bmcr;
    uint16_t   bmsr;

    PhyReadReg(MII_ANAR, &anar);
    PhyReadReg(MII_ANER, &aner);
    PhyReadReg(MII_BMCR, &bmcr);
    PhyReadReg(MII_BMSR, &bmsr);

    EthType ethType;
    bool fullDuplex;

    if ((bmsr & BMSR_ACOMP) && !(aner & ANER_LPAN))
    {
        NN_DETAIL_ETH_INFO("Link partner is not auto-negotiation able.\n");
    }

    if ((bmsr & BMSR_ACOMP) && (aner & ANER_LPAN))
    {
        // オートネゴシエーション成立時は、オートネゴシエーション結果から speed/duplex を取得
        uint16_t anlpar;
        uint16_t gtcr;
        uint16_t gtsr;
        PhyReadReg(MII_ANLPAR, &anlpar);
        PhyReadReg(MII_100T2CR, &gtcr);
        PhyReadReg(MII_100T2SR, &gtsr);

        if ((gtcr & GTCR_ADV_1000TFDX) && (gtsr & GTSR_LP_1000TFDX))
        {
            ethType = Type_1000_T;
            fullDuplex = true;
        }
        else if ((gtcr & GTCR_ADV_1000THDX) && (gtsr & GTSR_LP_1000THDX))
        {
            ethType = Type_1000_T;
            fullDuplex = false;
        }
        else if ((anar & ANAR_TX_FD) && (anlpar & ANLPAR_TX_FD))
        {
            ethType = Type_100_TX;
            fullDuplex = true;
        }
        else if ((anar & ANAR_TX) && (anlpar & ANLPAR_TX))
        {
            ethType = Type_100_TX;
            fullDuplex = false;
        }
        else if ((anar & ANAR_10_FD) && (anlpar & ANLPAR_10_FD))
        {
            ethType = Type_10_T;
            fullDuplex = true;
        }
        else if ((anar & ANAR_10) && (anlpar & ANLPAR_10))
        {
            ethType = Type_10_T;
            fullDuplex = false;
        }
        else
        {
            NN_DETAIL_ETH_WARN("No speed/duplex configuration is available as a result of auto-negotiation.\n");
            NN_DETAIL_ETH_WARN("Use 10M half-duplex.\n");
            ethType = Type_10_T;
            fullDuplex = false;
        }
    }
    else
    {
        // それ以外の場合は、BMCR から speed/duplex を取得
        ethType = (bmcr & BMCR_SPEED1) ? Type_1000_T :
                  (bmcr & BMCR_SPEED0) ? Type_100_TX : Type_10_T,
        fullDuplex = bmcr & BMCR_FDX;
    }

    *pMediaType = (MediaType)EthMakeMediaType(ethType,
        (bmcr & BMCR_AUTOEN) ? SubType_AUTO : SubType_MANUAL,
        (fullDuplex ? Option_FDX : Option_HDX) | (anar & ANAR_PAUSE_MASK ? Option_FLOW : 0),
        (bmsr & BMSR_LINK ? LinkState_Up : LinkState_Down));

    return ResultSuccess();
}

nn::Result SynopsysDwcEtherQos::PhySetMedia(MediaType type) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    std::lock_guard<nn::os::Mutex> lock(m_PhyMutex);

    Speed speed;
    bool fullDuplex;
    bool flowControlEnable;

    if (type == MediaType_AUTO)
    {
        speed = Speed_1000;
        fullDuplex = true;
        flowControlEnable = true;
    }
    else
    {
        NN_RESULT_DO(GetSpeed(&speed, type));
        fullDuplex = EthFullDuplex(type);
        flowControlEnable = EthFlow(type);
    }

    UpdateMtl(fullDuplex);
    UpdateMac(speed, fullDuplex, flowControlEnable);
    UpdateClock(speed);


    return ResultSuccess();
}

nn::Result SynopsysDwcEtherQos::PhyAdjustMedia(MediaType type) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);
    std::lock_guard<nn::os::Mutex> lock(m_PhyMutex);

    return PhySetMedia(type);
}

// ------------------------------------------------------------------------------------------------
// Phy related functions

void SynopsysDwcEtherQos::ProcessLinkStateChange(bool linkup) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    if (linkup)
    {
        NN_DETAIL_ETH_DEBUG_INFO("Link up\n");

        SocketIf::SetLinkState(true);
        Gmii::UpdateLinkStatus();
    }
    else
    {
        NN_DETAIL_ETH_DEBUG_INFO("Link down\n");

        Gmii::UpdateLinkStatus();
        SocketIf::SetLinkState(false);
    }
}

// ------------------------------------------------------------------------------------------------
// Mac related functions

void SynopsysDwcEtherQos::UpdateMac(Speed speed, bool fullDuplex, bool flowControlEnable) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    // Speed
    uint32_t macConfiguration = util::Read32(m_pEqos, RegOffset_MacConfiguration);

    switch (speed)
    {
    case Speed_10:
        macConfiguration &= ~MacConfiguration::Fes::Mask;
        macConfiguration |= MacConfiguration::Ps::Mask;
        break;
    case Speed_100:
        macConfiguration |= MacConfiguration::Fes::Mask;
        macConfiguration |= MacConfiguration::Ps::Mask;
        break;
    case Speed_1000:
        macConfiguration &= ~MacConfiguration::Fes::Mask;
        macConfiguration &= ~MacConfiguration::Ps::Mask;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (fullDuplex)
    {
        macConfiguration |= MacConfiguration::Dm::Mask;
    }
    else
    {
        macConfiguration &= ~MacConfiguration::Dm::Mask;
    }

    util::Write32(m_pEqos, RegOffset_MacConfiguration, macConfiguration);

    // Flow control
    if (flowControlEnable)
    {
        util::SetBitOn32(m_pEqos, RegOffset_MacTxFlowCtrl, MacTxFlowCtrl::Tfe::Mask);
        util::SetBitOn32(m_pEqos, RegOffset_MacRxFlowCtrl, MacRxFlowCtrl::Rfe::Mask);
    }
    else
    {
        util::SetBitOff32(m_pEqos, RegOffset_MacTxFlowCtrl, MacTxFlowCtrl::Tfe::Mask);
        util::SetBitOff32(m_pEqos, RegOffset_MacRxFlowCtrl, MacRxFlowCtrl::Rfe::Mask);
    }
}

// ------------------------------------------------------------------------------------------------
// Clock related functions

void SynopsysDwcEtherQos::UpdateMtl(bool fullDuplex) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    // Work around
    if (!fullDuplex)
    {
        util::SetBitOn32(m_pEqos, RegOffset_MtlTxq0OperationMode, MtlTxq0OperationMode::Ftq::Mask);
    }
}

// ------------------------------------------------------------------------------------------------
// Clock related functions

void SynopsysDwcEtherQos::UpdateClock(Speed speed) NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    switch (speed)
    {
    case Speed_10:
        pcv::SetClockRate(pcv::Module_EqosClockTx, TxClockHzForSpeed_10);
        break;
    case Speed_100:
        pcv::SetClockRate(pcv::Module_EqosClockTx, TxClockHzForSpeed_100);
        break;
    case Speed_1000:
        pcv::SetClockRate(pcv::Module_EqosClockTx, TxClockHzForSpeed_1000);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

// ------------------------------------------------------------------------------------------------
// Socket related functions (overrides SocketIf)

void SynopsysDwcEtherQos::Start() NN_NOEXCEPT
{
    m_TxRequestEvent.Signal();
}

// ------------------------------------------------------------------------------------------------
// Thread functions

void SynopsysDwcEtherQos::TxThread() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    while (true)
    {
        // 1. 送信データの到着イベント or DMA 送信完了割り込みイベントを待つ
        m_TxRequestEvent.Wait();

        while (true)
        {
            size_t length; // 送信データの長さ

            {
                std::lock_guard<nn::os::Mutex> lock(m_TxDescMutex);

                // 2.空き送信ディスクリプタ数が 0 の場合は条件変数を待つ
                while (m_AvailableTxDescCount == 0)
                {
                    m_TxDescConditionalVariable.Wait(m_TxDescMutex);
                }

                // 3. プロトコルスタックから送信データを取得
                NN_ABORT_UNLESS_RESULT_SUCCESS(SocketIf::GetFrameFromStack(&length, reinterpret_cast<uint8_t*>(&m_pTxBuffer[m_TxDescHead]), sizeof(TxBuffer)));

                if (length == 0)
                {
                    // イベント待ちに戻る
                    break;
                }
                else if (length == sizeof(TxBuffer))
                {
                    NN_DETAIL_ETH_WARN("GetFrameFromStack returns large value (jumbo packet?)");
                }

                nn::dd::FlushDataCache(&m_pTxBuffer[m_TxDescHead], length);

                // 4. 空き送信ディスクリプタ数を 1 減らす
                m_AvailableTxDescCount--;
                NN_SDK_ASSERT(m_AvailableTxDescCount >= 0);
            }

            // 5. ディスクリプタの内容を設定する
            PrepareTxDescToSend(m_TxDescHead, length);

            // 6. DMA_CH0_TxDesc_Tail_Pointer に書き込み
            util::Write32(m_pEqos, RegOffset_DmaCh0TxDescTailPointer, ADDR_LOW(m_TxDescRingTailPhysAddr));
            util::DummyRead(m_pEqos, RegOffset_DmaCh0TxDescTailPointer);

            // 7. 空きディスクリプタの先頭を更新
            m_TxDescHead = (m_TxDescHead + 1) % TxDescRingLength;
        }
    }
}

void SynopsysDwcEtherQos::ProcessTxCompleteInterrupt() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    Tdes3 tdes3;
    TxDesc desc;

    std::lock_guard<nn::os::Mutex> lock(m_TxDescMutex);

    auto prevAvailableTxDescCount = m_AvailableTxDescCount;

    // 送信ディスクリプタの owner を調べて、空きディスクリプタ数を増やす
    while (m_AvailableTxDescCount < TxDescRingLength)
    {
        InvalidateTxDescCache(m_TxDescTail); // プリフェッチデータをクリア
        ReadTdes3(&tdes3, m_TxDescTail);

        if (tdes3.Get<Tdes3::Own>() == Owner_Cpu)
        {
            nn::dd::EnsureMemoryAccess();
            ReadTxDesc(&desc, m_TxDescTail);
            if (!CheckTxDesc(desc))
            {
                // エラー発生時は原因を print して処理を続行
                PrintTxDescError(desc);
            }

            m_AvailableTxDescCount++;
            m_TxDescTail = (m_TxDescTail + 1) % TxDescRingLength;
        }
        else
        {
            break;
        }
    }

    // 空きディスクリプタ数が 1 以上になったら条件変数をシグナルする
    if (prevAvailableTxDescCount == 0 && m_AvailableTxDescCount > 0)
    {
        m_TxDescConditionalVariable.Signal();
    }
}

void SynopsysDwcEtherQos::ProcessRxInterrupt() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    Rdes3 rdes3;
    RxDesc desc;

    // 受信データが無くなるまで処理を繰り返す
    while (true)
    {
        // 2. ディスクリプタの内容を読み込んで処理
        InvalidateRxDescCache(m_RxDescHead); // プリフェッチデータをクリア
        ReadRdes3(&rdes3, m_RxDescHead);

        if (rdes3.Get<Rdes3::Own>() == Owner_Dma)
        {
            break;
        }

        nn::dd::EnsureMemoryAccess();
        ReadRxDesc(&desc, m_RxDescHead);

        if (CheckRxDesc(desc))
        {
            size_t packetLength = desc.rdes3.Get<Rdes3::PacketLength>();

            if (packetLength != 0)
            {
                // 3. ディスクリプタに対応する受信バッファからデータを読みこんで、プロトコルスタックに渡す
                nn::dd::InvalidateDataCache(&m_pRxBuffer[m_RxDescHead], packetLength); // プリフェッチデータをクリア
                NN_ABORT_UNLESS_RESULT_SUCCESS(SendFrameToStack(reinterpret_cast<uint8_t*>(&m_pRxBuffer[m_RxDescHead]), packetLength));
                nn::dd::FlushDataCache(&m_pRxBuffer[m_RxDescHead], packetLength); // 次の受信のためにフラッシュ
            }
        }
        else
        {
            // エラー発生時は、受信バッファの読み込みをスキップして、エラー原因を表示
            PrintRxDescError(desc);
        }

        // 4. ディスクリプタを再利用するために再初期化
        PrepareRxDescToReceive(m_RxDescHead);

        // 5. DMA_CH0_RxDesc_Tail_Pointer に書き込み
        util::Write32(m_pEqos, RegOffset_DmaCh0RxDescTailPointer, ADDR_LOW(m_RxDescRingTailPhysAddr));
        util::DummyRead(m_pEqos, RegOffset_DmaCh0RxDescTailPointer);

        // 6. 受信ディスクリプタの先頭を更新
        m_RxDescHead = (m_RxDescHead + 1) % RxDescRingLength;
    }
}

void SynopsysDwcEtherQos::InterruptThread() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    while (true)
    {
        nn::os::MultiWaitHolderType* pHolder = nn::os::WaitAny(&m_InterruptMultiWait);

        auto index = static_cast<int>(nn::os::GetMultiWaitHolderUserData(pHolder));
        nn::os::ClearInterruptEvent(&m_InterruptEvent[index]);

        uint32_t dmaStatus = util::Read32(m_pEqos, RegOffset_DmaCh0Status);
        uint32_t mtlStatus = util::Read32(m_pEqos, RegOffset_MtlQ0InterruptControlStatus);
        uint32_t macStatus = util::Read32(m_pEqos, RegOffset_MacInterruptStatus);

        // 割り込み要因解除
        util::Write32(m_pEqos, RegOffset_DmaCh0Status, dmaStatus);

        // TORIAEZU: 発火した InterruptEvent によらず、すべての割り込みステータスを調べて処理する
        // 異常検知
        if ((dmaStatus & DmaStatus::Cde::Mask) ||
            (dmaStatus & DmaStatus::Fbe::Mask) ||
            (mtlStatus & MtlInterruptControlStatus::Txunfis::Mask) ||
            (macStatus & MacInterruptStatus::Rxstsi::Mask) ||
            (macStatus & MacInterruptStatus::Txstsi::Mask))
        {
            uint32_t macRxTxStatus = util::Read32(m_pEqos, RegOffset_MacRxTxStatus);
            NN_UNUSED(macRxTxStatus); // for Release build.

            NN_DETAIL_ETH_ERROR("DmaCh0Status:                %08x\n", dmaStatus);
            NN_DETAIL_ETH_ERROR("MtlQ0InterruptControlStatus: %08x\n", mtlStatus);
            NN_DETAIL_ETH_ERROR("MacInterruptStatus:          %08x\n", macStatus);
            NN_DETAIL_ETH_ERROR("MacRxTxStatus:               %08x\n", macRxTxStatus);
            NN_ABORT("Unexpected interrupt.\n");
        }

        // 受信割り込み
        if (dmaStatus & DmaStatus::Ri::Mask)
        {
            ProcessRxInterrupt();
        }

        // 送信割り込み
        if (dmaStatus & DmaStatus::Ti::Mask)
        {
            ProcessTxCompleteInterrupt();
        }
    }
}

void SynopsysDwcEtherQos::PhyThread() NN_NOEXCEPT
{
    NN_DETAIL_ETH_DEBUG_TRACE("%s\n", __FUNCTION__);

    uint16_t miiValue;

    // 本来は PHY からの割り込み (GPIO) を使うべきだが、現在は TX2 で GPIO 経由の割り込みが使いづらいのでポーリングで実装する。
    // TODO: GPIO 割り込みが容易に使えるようになったら、実装を変更する。
    bool linkup = false;
    bool prevLinkup = false;

    m_PhyTimerEvent.StartPeriodic(PhyThreadPollInterval, PhyThreadPollInterval);

    while (true)
    {
        m_PhyTimerEvent.Wait();

        PhyReadReg(MII_BMSR, &miiValue);

        // Link up/down
        linkup = (miiValue & BMSR_LINK);
        if (linkup != prevLinkup)
        {
            ProcessLinkStateChange(linkup);
            prevLinkup = linkup;
        }

        Gmii::Tick();
    }
}

} // tx2
} // device
} // eth
} // nn
