﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/eth/eth_EthClient.h>
#include <nn/eth/server/eth_EthServer.h>

namespace nn     {
namespace eth    {
namespace client {
namespace        {

const int BitsPerWord     = 8;
const int SessionHeapSize = (8 * 1024);

class InitExpHeap
{
public:
    InitExpHeap()
    NN_NOEXCEPT
    {
        nn::sf::ExpHeapStaticAllocator
                <SessionHeapSize, InterfaceGroupHandler>::
                Initialize(nn::lmem::CreationOption_ThreadSafe);
    }
} g_InitExpHeap;

inline int getBitField(uint8_t* data, uint32_t offset)
NN_NOEXCEPT
{
    uint32_t word = offset / BitsPerWord;
    uint32_t bit  = offset % BitsPerWord;
    return (data[word] & (1<<bit));
}

inline void setBitField(uint8_t* data, uint32_t offset)
NN_NOEXCEPT
{
    uint32_t word = offset / BitsPerWord;
    uint32_t bit  = offset % BitsPerWord;
    data[word] |= (0x1 << bit);
    return;
}

} // namespace

InterfaceGroupHandler::InterfaceGroupHandler()
NN_NOEXCEPT : m_Initialized(false)
{

}

InterfaceGroupHandler::~InterfaceGroupHandler()
NN_NOEXCEPT
{
    m_InterfaceGroupImpl = nullptr;
}

nn::Result InterfaceGroupHandler::Initialize(nn::os::EventClearMode eventClearMode)
NN_NOEXCEPT
{
    nn::sf::NativeHandle sfHandle;
    nn::Result result;

    NN_ABORT_UNLESS(m_Initialized == false, "already initialized\n");

    result = nn::sf::CreateHipcProxyByName
                    <nn::eth::sf::IEthInterfaceGroup, nn::sf::ExpHeapStaticAllocator
                    <SessionHeapSize, InterfaceGroupHandler>::Policy>
                    (&m_InterfaceGroupImpl, InterfaceGroupPortName);

    if (result.IsSuccess())
    {
        result = m_InterfaceGroupImpl->GetReadableHandle(&sfHandle);
        if (result.IsSuccess())
        {
            m_SystemEvent.AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), eventClearMode);
            sfHandle.Detach();
            m_Initialized = true;
        }
        else
        {
            m_InterfaceGroupImpl = nullptr;
        }
    }

    return result;
}

nn::Result InterfaceGroupHandler::Finalize()
NN_NOEXCEPT
{
    m_InterfaceGroupImpl = nullptr;
    nn::os::DestroySystemEvent(m_SystemEvent.GetBase());
    m_Initialized = false;
    return ResultSuccess();
}

nn::Result InterfaceGroupHandler::GetResult()
NN_NOEXCEPT
{
    return m_InterfaceGroupImpl->GetResult();
}

nn::Result InterfaceGroupHandler::Cancel()
NN_NOEXCEPT
{
    return m_InterfaceGroupImpl->Cancel();
}

void InterfaceGroupHandler::ClearEvent()
NN_NOEXCEPT
{
    m_SystemEvent.Clear();
}

nn::os::SystemEvent* InterfaceGroupHandler::GetSystemEventPointer()
NN_NOEXCEPT
{
    return &m_SystemEvent;
}

nn::Result InterfaceGroupHandler::GetInterfaceList(InterfaceList* pInterfaceListOut)
NN_NOEXCEPT
{
    nn::sf::OutBuffer interfaceList(reinterpret_cast<char*>(pInterfaceListOut), sizeof(InterfaceList));
    return m_InterfaceGroupImpl->GetInterfaceList(interfaceList);
}

nn::Result InterfaceGroupHandler::GetInterfaceCount(uint32_t* pInterfaceCount)
NN_NOEXCEPT
{
    return m_InterfaceGroupImpl->GetInterfaceCount(pInterfaceCount);
}

void InterfaceGroupHandler::ParseInterfaceList(
        InterfaceList* pNewList,
        InterfaceList* pOldList,
        InterfaceList* pAddedList,
        InterfaceList* pRemovedList)
NN_NOEXCEPT
{
    uint8_t oldListMask[(nn::eth::MaxInterfaceCount + BitsPerWord - 1) / BitsPerWord] = {0};
    uint8_t newListMask[(nn::eth::MaxInterfaceCount + BitsPerWord - 1) / BitsPerWord] = {0};

    pRemovedList->adapterCount = 0;
    pAddedList->adapterCount   = 0;

    for (uint32_t i = 0; i < pNewList->adapterCount; i++)
    {
        for (uint32_t j = 0; j < pOldList->adapterCount; j++)
        {
            if (strncmp(
                        pNewList->adapterInfo[i].interfaceName,
                        pOldList->adapterInfo[j].interfaceName,
                        nn::eth::MaxInterfaceNameLength) == 0)
            {
                setBitField(newListMask, i);
                setBitField(oldListMask, j);
            }
        }
    }

    for (uint32_t i = 0; i < pNewList->adapterCount; i++)
    {
        if (!getBitField(newListMask, i))
        {
            pAddedList->adapterInfo[pAddedList->adapterCount++] = pNewList->adapterInfo[i];
        }
    }

    for (uint32_t j = 0; j < pOldList->adapterCount; j++)
    {
        if (!getBitField(oldListMask, j))
        {
            pRemovedList->adapterInfo[pRemovedList->adapterCount++] = pOldList->adapterInfo[j];
        }
    }
}

}}}

