﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/dt/dt_Types.h>

namespace nn { namespace dt { namespace server {

class FdtReader
{
public:
    void Initialize(const nn::Bit8* fdt, size_t size) NN_NOEXCEPT;
    nn::Result GetNodePath(char* pOutBuffer, size_t bufferSize, Node* pNode) NN_NOEXCEPT;
    nn::Result DumpNode(Node* pNode, int depth) NN_NOEXCEPT;
    nn::Result FindNodeByPath(Node* pOutNode, const char* path) NN_NOEXCEPT;
    nn::Result FindNodeByPHandle(Node* pOutNode, PHandle phandle) NN_NOEXCEPT;
    nn::Result GetCompatibleNodeCount(size_t* pOutCount, const char* compatible) NN_NOEXCEPT;
    nn::Result ListCompatibleNode(Node* pOutNodeList, size_t* pOutCount, size_t maxCount, const char* compatible) NN_NOEXCEPT;
    nn::Result GetParentNode(Node* pOutNode, Node* pCurrentNode) NN_NOEXCEPT;
    nn::Result GetChildNodeByName(Node* pOutNode, Node* pCurrentNode, const char* name) NN_NOEXCEPT;
    nn::Result GetChildNodeCount(size_t* pOutCount, Node* pCurrentNode) NN_NOEXCEPT;
    nn::Result ListChildNode(Node* pOutNodeList, size_t* pOutCount, size_t maxCount, Node* pCurrentNode) NN_NOEXCEPT;
    nn::Result GetFirstChildNode(Node* pOutNode, Node* pCurrentNode) NN_NOEXCEPT;
    nn::Result GetNextSiblingNode(Node* pOutNode, Node* pCurrentNode) NN_NOEXCEPT;
    nn::Result GetValueOfAddressCells(uint32_t* pOutValue, Node* pNode) NN_NOEXCEPT;
    nn::Result GetValueOfSizeCells(uint32_t* pOutValue, Node* pNode) NN_NOEXCEPT;
    nn::Result IsPropertyExist(bool *pOutExist, Node* pNode, const char* name) NN_NOEXCEPT;
    nn::Result GetPropertySize(size_t *pOutSize, Node* pNode, const char* name) NN_NOEXCEPT;
    nn::Result GetProperty(char* pOutValue, Node* pNode, const char* name, size_t elementSize) NN_NOEXCEPT;
    nn::Result GetPropertyWithIndex(char* pOutValue, Node* pNode, const char* name, size_t index, size_t elementSize) NN_NOEXCEPT;
    nn::Result GetPropertyList(char* pOutList, size_t* pOutPropertySize, size_t maxCount, Node* pNode, const char* name, size_t elementSize) NN_NOEXCEPT;
    nn::Result GetInterruptCount(size_t* pOutCount, Node* pNode) NN_NOEXCEPT;
    nn::Result GetInterruptList(InterruptInfo* pOutList, size_t* pOutActualCount, size_t maxCount, Node* pNode) NN_NOEXCEPT;
    nn::Result GetRegisterAddressList(char* pOutList, size_t* pOutActualCount, size_t maxCount, Node* pNode, size_t elementSize) NN_NOEXCEPT;
    nn::Result GetRegisterSizeList(char* pOutList, size_t* pOutActualCount, size_t maxCount, Node* pNode, size_t elementSize) NN_NOEXCEPT;

private:
    enum RegisterPropertyType
    {
        RegisterPropertyType_Address,
        RegisterPropertyType_Size,
    };

    static size_t GetOffsetInPropertyFrom(RegisterPropertyType type, size_t addressCells)
    {
        switch (type)
        {
        case RegisterPropertyType_Address:
            return 0;
            break;
        case RegisterPropertyType_Size:
            return sizeof(uint32_t) * addressCells;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    nn::Result GetRegisterPropertyRaw(const char** pPointerToProperty, size_t *pOutPropertySize, uint32_t* pOutAddressCells, uint32_t* pOutSizeCells, Node* pNode) NN_NOEXCEPT;
    nn::Result GetRegisterPropertyWithIndex(char* pOutValue, Node* pNode, size_t index, size_t elementSize, RegisterPropertyType type) NN_NOEXCEPT;
    nn::Result GetRegisterPropertyList(char* pOutList, size_t* pOutPropertySize, size_t maxCount, Node* pNode, size_t elementSize, RegisterPropertyType type) NN_NOEXCEPT;

    nn::Result GetValueOfInterruptCells(uint32_t* pOutValue, Node* pNode) NN_NOEXCEPT;

    bool IsValidNodeOffset(int offset) NN_NOEXCEPT
    {
        return (0 <= offset && offset < m_FdtSize);
    }

    nn::Result DumpRecursive(Node* pNode, int depth, int indent) NN_NOEXCEPT;

    /*
    * @brief
    *  value が負の値なら、エラー内容を表示したうえで、該当の result を返して関数を抜ける。
    */
    nn::Result CheckLibFdtError(int error) NN_NOEXCEPT;

    /*
    * @brief
    *  offset が負の値なら、エラー内容を表示したうえで、該当の result を返して関数を抜ける。
    *  offset が範囲内であることもアサートで検査する。
    */
    nn::Result CheckReturnedOffset(int offset) NN_NOEXCEPT;

    /*
    * @brief
    *  pNode が nullptr でないことと pNode->GetOffset() が範囲内であることを検査し、
    *  引っかかった場合は ResultInvalidArgument を返す。
    *  ただし、Device Tree の内容に問題があった場合は ResultLibFdtError を返す。
    */
    nn::Result CheckInputNode(Node* pNode) NN_NOEXCEPT;

    const nn::Bit8* m_pFdt;
    size_t m_FdtSize;
};

}}}
