﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/drivers/eth/driverEth.h>
#include <nn/os.h>
#include <nn/dd.h>

namespace nn { namespace drivers { namespace eth {
class EthernetDriver
{
private:
    static const int      IRQ_PRIORITY         = 3;

public:
    nn::Result Initialize();
    nn::Result Finalize();

    void RegisterFrameReceivedCallBack(FrameReceivedCallBack callBack)
    {
        m_FrameReceivedCallBack = callBack;
    }
    void RegisterSendCompletedCallBack(SendCompletedCallBack callBack)
    {
        m_SendCompletedCallBack = callBack;
    }

    void UnregisterFrameReceivedCallBack()
    {
        m_FrameReceivedCallBack = NULL;
    }
    void UnregisterSendCompletedCallBack()
    {
        m_SendCompletedCallBack = NULL;
    }

    nn::Result Send(const void* frameData, int frameBytes);
    nn::Result Send(nnnetOslMbuf* pMbuf);

    nn::Result GetMacAddress(uint8_t pMacAddress[NN_NET_MAC_ADDRESS_SIZE]) const;
    void HandleInterrupt();
private:
    class InternalConditionVariable;
    class InternalMutex
    {
        friend InternalConditionVariable;
    public:
        void Initialize()
        {
            nn::os::InitializeMutex(&m_Mutex, false, 0);
        }
        void Finalize()
        {
            nn::os::FinalizeMutex(&m_Mutex);
        }
        void lock()
        {
            nn::os::LockMutex(&m_Mutex);
        }
        void unlock()
        {
            nn::os::UnlockMutex(&m_Mutex);
        }
    private:
        nn::os::MutexType m_Mutex;
    };
    class InternalConditionVariable
    {
    public:
        void Initialize()
        {
            nn::os::InitializeConditionVariable(&m_ConditionVariable);
        }
        void Finalize()
        {
            nn::os::FinalizeConditionVariable(&m_ConditionVariable);
        }
        void Signal()
        {
            nn::os::SignalConditionVariable(&m_ConditionVariable);
        }
        void Broadcast()
        {
            nn::os::BroadcastConditionVariable(&m_ConditionVariable);
        }
        void Wait(InternalMutex* lock)
        {
            nn::os::WaitConditionVariable(&m_ConditionVariable, &lock->m_Mutex);
        }
    private:
        nn::os::ConditionVariableType m_ConditionVariable;
    };

    InternalMutex               m_TxMutex;
    InternalConditionVariable   m_ConditionVariable;
    nn::os::ThreadType          m_InterruptTthread;

    bool                        m_RequestedFinalize;
    FrameReceivedCallBack       m_FrameReceivedCallBack;
    SendCompletedCallBack       m_SendCompletedCallBack;
    uint16_t                    m_PacketTag;

    void DumpRegister();
    uint16_t PhyRead(uint8_t addr);
    void PhyWrite(uint8_t addr, uint16_t val);
    void StartAutoNegotiation();
    void Receive();
    void TxReady(uint32_t interruptStatus);
    void ClearInterrupt();
    void Reset();
};

}}}

