﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/diag/detail/diag_DetailTranslation.h>
#include <nn/nn_SdkLog.h>

//#define NN_DIAG_TRANSLATION_DEBUG

#if defined(NN_DIAG_TRANSLATION_DEBUG)
#define NN_DIAG_TRANSLATION_LOG NN_SDK_LOG
#else
#define NN_DIAG_TRANSLATION_LOG(...)
#endif

namespace nn { namespace diag { namespace detail {

namespace {
    const int MagicNumberCount = 4;
    const uint8_t MagicNumbers[MagicNumberCount] = { 0xde, 0x12, 0x04, 0x95 };

    typedef struct {
        uint32_t length;
        uint32_t offset;
    } StringProperty;
}

// TODO: 範囲外アクセスのリスクをなくすため、Dictionary のサイズを渡すようにする
const char* Translate(Dictionary dictionary, const char* pKey) NN_NOEXCEPT
{
    const uint8_t* data = reinterpret_cast<const uint8_t*>(dictionary);

    // マジックナンバーのチェック
    const uint8_t* pMagicNumber = data;
    for (int i = 0; i < MagicNumberCount; i++)
    {
        // 現在、不正な辞書データとして1バイトのデータ 0x00 が渡されることがある。
        // これを範囲外アクセスせずに扱えるよう、マジックナンバーの比較は1バイトずつ行う。
        // TODO: 「不正な辞書データとして1バイトのデータを渡す」というやり方をやめる
        if (pMagicNumber[i] != MagicNumbers[i])
        {
            NN_DIAG_TRANSLATION_LOG("マジックナンバーが不正です。\n");
            return pKey;
        }
    }

    // ファイルフォーマットリビジョンのチェック
    const uint32_t fileFormatRevision = *reinterpret_cast<const uint32_t*>(data + 4);
    if (fileFormatRevision != 0)
    {
        NN_DIAG_TRANSLATION_LOG("ファイルフォーマットリビジョンが不正です。\n");
        return pKey;
    }

    const uint32_t stringCount = *reinterpret_cast<const uint32_t*>(data + 8);
    const uint32_t originalStringsOffset = *reinterpret_cast<const uint32_t*>(data + 12);
    const uint32_t translationStringsOffset = *reinterpret_cast<const uint32_t*>(data + 16);
    const StringProperty* pOriginalStringsProperty = reinterpret_cast<const StringProperty*>(data + originalStringsOffset);
    const StringProperty* pTranslationStringsProperty = reinterpret_cast<const StringProperty*>(data + translationStringsOffset);

    // key に一致するオリジナル文字列の検索
    bool found = false;
    uint32_t index = 0;

    for (uint32_t i = 0; i < stringCount; i++)
    {
        StringProperty property = pOriginalStringsProperty[i];
        const char* pString = reinterpret_cast<const char*>(data + property.offset);

        if (strcmp(pString, pKey) == 0)
        {
            found = true;
            index = i;
            break;
        }
    }

    if (!found)
    {
        NN_DIAG_TRANSLATION_LOG("オリジナル文字列の検索がヒットしませんでした。\n");
        return pKey;
    }

    // ヒットしたオリジナル文字列に対応する翻訳済文字列の取得
    StringProperty property = pTranslationStringsProperty[index];
    const char* pValue = reinterpret_cast<const char*>(data + property.offset);

    return pValue;
}

}}}
