﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Windows.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include "diag_DbgHelp.h"

// SIGLO-21432: DbgHelp.h を VS2015 でビルドすると警告が出るので、抑制する。
#pragma warning(push)
#pragma warning(disable: 4091)
#include <DbgHelp.h>
#pragma warning(pop)

#pragma comment(lib, "DbgHelp.lib")

namespace nn { namespace diag { namespace detail {

namespace
{
    SYMBOL_INFO* GetSymbolInfoFromAddress(uintptr_t address) NN_NOEXCEPT
    {
        diag::detail::LockDbgHelpMutex();
        NN_UTIL_SCOPE_EXIT { diag::detail::UnlockDbgHelpMutex(); };

        auto hProcess = GetCurrentProcess();

        auto pSymbol = static_cast<SYMBOL_INFO*>(calloc(sizeof(SYMBOL_INFO) + MAX_SYM_NAME * sizeof(char), sizeof(char)));
        NN_ABORT_UNLESS(pSymbol != nullptr);
        pSymbol->MaxNameLen = MAX_SYM_NAME;
        pSymbol->SizeOfStruct = sizeof(SYMBOL_INFO);

        if (!SymFromAddr(hProcess, static_cast<DWORD64>(address), 0, pSymbol))
        {
            free(pSymbol);
            return nullptr;
        }

        return pSymbol;
    }
}

uintptr_t GetSymbolNameImpl(char* outNameBuffer, size_t nameBufferSize, uintptr_t address) NN_NOEXCEPT
{
    auto pSymbol = GetSymbolInfoFromAddress(address);

    if (!pSymbol)
    {
        return 0u;
    }

    NN_UTIL_SCOPE_EXIT { free(pSymbol); };

    util::Strlcpy(outNameBuffer, pSymbol->Name, static_cast<int>(nameBufferSize));

    return static_cast<uintptr_t>(pSymbol->Address);
}

size_t GetSymbolSizeImpl(uintptr_t address) NN_NOEXCEPT
{
    auto pSymbol = GetSymbolInfoFromAddress(address);

    if (!pSymbol)
    {
        return 0u;
    }

    NN_UTIL_SCOPE_EXIT { free(pSymbol); };

    if (address != pSymbol->Address)
    {
        return 0u;
    }

    return pSymbol->Size;
}

}}} // nn::diag::detail
