﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/diag/diag_Config.h>

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    #error  "Win32 OS の機能が利用出来るビルド条件ではありません。"
#endif

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Windows.h>
#include "diag_PrintDebugString.h"

namespace nn { namespace diag { namespace detail {

void PrintDebugString(const char* text, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(text || length == 0);
    if (length == 0)
    {
        return;
    }

    // OutputDebugStringA がヌル終端を要求するため、ヌル終端してから渡す
    char* string = reinterpret_cast<char*>(malloc((sizeof(char) * length) + 1));
    std::memcpy(string, text, length);
    string[length] = '\0';
    OutputDebugStringA(string);
    std::fwrite(string, sizeof(char), length, stdout);
    std::free(string);
    std::fflush(stdout);
}

void PrintDebugStringWithConversion(const char* text, size_t length) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(text || length == 0);
    if (length == 0)
    {
        return;
    }

    const UINT consoleCodePage = GetConsoleOutputCP();

    if( consoleCodePage == CP_UTF8 )
    {
        PrintDebugString(text, length);
    }
    else
    {
        // text の文字コードを コマンドプロンプトのコードページに合わせて変更する
        const auto wideStringCount = MultiByteToWideChar(CP_UTF8, 0, text, static_cast<int>(length), NULL, 0);
        LPWSTR pWideString = reinterpret_cast<LPWSTR>(malloc(sizeof(wchar_t) * wideStringCount));
        NN_SDK_ASSERT_NOT_NULL(pWideString);
        const auto multiByteToWideCharResult = MultiByteToWideChar(CP_UTF8, 0, text, static_cast<int>(length), pWideString, wideStringCount);
        NN_SDK_ASSERT(multiByteToWideCharResult != 0, "MultiByteToWideChar() failed.\n");
        NN_UNUSED(multiByteToWideCharResult); // リリースビルド時の未使用変数警告に対応

        const auto multibyteStringCount = WideCharToMultiByte(consoleCodePage, 0, pWideString, wideStringCount, NULL, 0, NULL, NULL);
        auto pMultibyteString = reinterpret_cast<char*>(malloc(multibyteStringCount));
        NN_SDK_ASSERT_NOT_NULL(pMultibyteString);
        const auto wideCharToMultiByteResult = WideCharToMultiByte(consoleCodePage, 0, pWideString, wideStringCount, pMultibyteString, multibyteStringCount, NULL, NULL);
        NN_SDK_ASSERT(wideCharToMultiByteResult != 0, "WideCharToMultiByte() failed.\n");
        NN_UNUSED(wideCharToMultiByteResult); // リリースビルド時の未使用変数警告に対応

        PrintDebugString(pMultibyteString, multibyteStringCount);

        free(pWideString);
        free(pMultibyteString);
    }
}

}}}
