﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/diag/diag_AbortTypes.h>
#include "diag_DumpStackTrace.h"

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <windows.h>
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/svc/svc_Base.h>
#endif

namespace nn { namespace diag { namespace detail {

#if !defined(NN_SDK_BUILD_RELEASE)
namespace {

const char* ToString(
    AbortReason abortReason) NN_NOEXCEPT
{
    switch( abortReason )
    {
    case AbortReason_SdkAssert:
        return "SDK Assertion Failure";
    case AbortReason_SdkRequires:
        return "Precondition not met";
    case AbortReason_UserAssert:
        return "Assertion Failure";
    case AbortReason_UnexpectedDefault:
        return "Unexpected Default";
    case AbortReason_Abort:
    default:
        return "Abort";
    }
}

void DefaultPrinter(
    const AbortInfo &abortInfo) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    DWORD processId = GetCurrentProcessId();
    DWORD threadId  = GetCurrentThreadId();
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    Bit64 processId = 0, threadId = 0;
    nn::svc::GetProcessId(&processId, svc::PSEUDO_HANDLE_CURRENT_PROCESS);
    nn::svc::GetThreadId(&threadId, svc::PSEUDO_HANDLE_CURRENT_THREAD);
#endif
    const char* pThreadName = nullptr;
    if (nn::os::GetCurrentThread())
    {
        pThreadName = nn::os::GetThreadNamePointer(nn::os::GetCurrentThread());
    }

#if defined(NN_BUILD_CONFIG_OS_WIN)
    NN_SDK_LOG(
        "%s: \'%s\' in %s() at pid=%u, tid=%u(%s)\n%s(%d)\n",
        ToString(abortInfo.abortReason),
        abortInfo.condition,
        abortInfo.functionName,
        processId,
        threadId,
        pThreadName ? pThreadName : "unknown",
        abortInfo.fileName,
        abortInfo.lineNumber);
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_SDK_LOG(
        "%s: \'%s\' in %s() at pid=%lld, tid=%lld(%s)\n%s(%d)\n",
        ToString(abortInfo.abortReason),
        abortInfo.condition,
        abortInfo.functionName,
        processId,
        threadId,
        pThreadName ? pThreadName : "unknown",
        abortInfo.fileName,
        abortInfo.lineNumber);
#else
    NN_SDK_LOG(
        "%s: \'%s\' in %s()\n%s(%d)\n",
        ToString(abortInfo.abortReason),
        abortInfo.condition,
        abortInfo.functionName,
        abortInfo.fileName,
        abortInfo.lineNumber);
#endif
    NN_SDK_VLOG(abortInfo.message->format, *abortInfo.message->args);
    NN_SDK_LOG("\n");
    TentativeDumpStackTrace();
}

} // anonymous
#endif

void DefaultAbortObserver(
    const AbortInfo &abortInfo) NN_NOEXCEPT
{
#if defined(NN_SDK_BUILD_RELEASE)
    NN_UNUSED(abortInfo);
#else
    DefaultPrinter(abortInfo);
#endif
}

}}} // nn::diag::detail
