﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcpc_PrivateIncludes.h"

namespace nn { namespace bsdsocket { namespace dhcpc {

// Log Level
Util::LogLevel Util::m_LogLevel = Util::LogLevel_Info;

const char *Util::LogLevelNames[LogLevel_Max] =
{
    "verbose", "info ", "warn ", "error "
};

const char* Util::HwAddrToA(const unsigned char *hwaddr, size_t hwlen, ToABuffer *buf)
{
    char *p;
    size_t i;

    if (buf == NULL)
    {
        return "invalid";
    }

    if (hwlen * 3 > sizeof(buf->data))
    {
        return "invalid";
    }

    p = buf->data;
    for (i = 0; i < hwlen; i++)
    {
        if (i > 0)
        {
            *p++ = ':';
        }
        p += snprintf(p, 3, "%.2x", hwaddr[i]);
    }
    *p++ = '\0';
    return (const char *)(buf->data);
}

const char* Util::GetDhcpProtMsgTypeDescription(DhcpProtMsgType t)
{
    const char *pD = "UNKNOWN PROT MSG TYPE!";
    const char *descriptions[] =
    {
        "Discover", "Offer", "Request", "Decline", "Ack", "Nak", "Release", "Inform", "ForceRenew"
    };
    if (t < (sizeof(descriptions) / sizeof(char *)))
    {
        pD = descriptions[t];
    }
    return pD;
}

void Util::Log(Util::LogLevel level, const char *format, ...)
{
    int32_t hour, minute, second, millisecond;
    va_list ap;
    size_t offset;
    const size_t bufSize = 256;
    char buffer[bufSize];

    if (level < m_LogLevel)
    {
        return;
    }

    nn::os::Tick tick = nn::os::GetSystemTick();

    hour = nn::os::ConvertToTimeSpan(tick).GetHours();
    minute = nn::os::ConvertToTimeSpan(tick).GetMinutes() % 60;
    second = nn::os::ConvertToTimeSpan(tick).GetSeconds() % 60;
    millisecond = nn::os::ConvertToTimeSpan(tick).GetMilliSeconds() % 1000;

    va_start(ap, format);
    snprintf(buffer, bufSize, "[DHCPC] %s%d:%02d:%02d.%03d| ",
             LogLevelNames[level], hour, minute, second, millisecond);
    offset = strnlen(buffer, bufSize);
    vsnprintf(buffer + offset, bufSize - offset, format, ap);
    va_end(ap);

    NN_SDK_LOG(buffer);
}


void Util::GetSaturatedUpTimeSeconds(uint16_t *pRetVal)
{
    TimeSpan uptime = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
    *pRetVal = (uptime.GetSeconds() >= 0xFFFF) ? 0xFFFF : (uint16_t)uptime.GetSeconds();
}

void Util::GetUpTime(TimeSpan * pRetVal)
{
    *pRetVal = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
}

size_t Util::EncodeRfc1035(const char *src, uint8_t *dst)
{
    uint8_t *p;
    uint8_t *lp;
    size_t len;
    uint8_t has_dot;

    if (src == NULL || *src == '\0') return 0;

    if (dst)
    {
        p = dst;
        lp = p++;
    }
    /* Silence bogus GCC warnings */
    else p = lp = NULL;

    len = 1;
    has_dot = 0;
    for (; *src; src++)
    {
        if (*src == '\0') break;
        if (*src == '.')
        {
            /* Skip the trailing . */
            if (src[1] == '\0') break;
            has_dot = 1;
            if (dst)
            {
                *lp = (uint8_t)(p - lp - 1);
                if (*lp == '\0') return len;
                lp = p++;
            }
        }
        else if (dst)
        {
            *p++ = (uint8_t)(*src);
        }
        len++;
    }

    if (dst)
    {
        *lp = (uint8_t)(p - lp - 1);
        if (has_dot)
        {
            *p++ = '\0';
        }
    }

    if (has_dot) len++;

    return len;
}

uint16_t Util::Checksum(const uint8_t *data, uint16_t len)
{
    const uint8_t *addr = data;
    uint32_t sum = 0;
    uint16_t res;

    while (len > 1)
    {
        sum += addr[0] * 256 + addr[1];
        addr += 2;
        len -= 2;
    }

    if (len == 1) sum += *addr * 256;

    sum = (sum >> 16) + (sum & 0xffff);
    sum += (sum >> 16);

    res = nn::socket::InetHtons(sum);

    return ~res;
}

void Util::GetAddrs(int type, char *cp, struct sockaddr **sa)
{
    int i;
    for (i = 0; i < RTAX_MAX; i++)
    {
        if (type & (1 << i))
        {
            sa[i] = (struct sockaddr *)cp;
            RT_ADVANCE(cp, sa[i]);
        }
        else
        {
            sa[i] = NULL;
        }
    }
}

const uint8_t* Util::GetOption(uint8_t optionBuffer[DhcpSizes_Option + DhcpSizes_Bootfile + DhcpSizes_ServerName],
                               const DhcpMessage *dhcp, uint32_t opt, size_t *len)
{

    const uint8_t *p = dhcp->options;
    const uint8_t *e = p + sizeof(dhcp->options);
    uint8_t l, ol = 0;
    uint8_t o = 0;
    uint8_t overl = 0;
    uint8_t overlperm = 0;
    uint8_t *bp = NULL;
    const uint8_t *op = NULL;
    size_t bl = 0;
    size_t preOverloadBl = 0;
    uint8_t *preOverloadBp = NULL;
    const uint8_t *preOverloadOp = NULL;
    uint8_t overloadEndFound = 0;
    while (p < e)
    {
        o = *p++;
        if (o == opt)
        {
            if (op)
            {
                if (!bp) bp = optionBuffer;
                memcpy(bp, op, ol);
                bp += ol;
            }
            ol = *p;
            if (p + ol > e)
            {
                return NULL;
            }
            op = p + 1;
            bl += ol;
        }
        switch (o)
        {
        case DhcpProtOption_Pad:
            if (p - 1 == dhcp->bootfile || p - 1 == dhcp->servername)
            {
                goto exit;
            }
            continue;
        case DhcpProtOption_End:
            if (overl & 1)
            {
                /* bit 1 set means parse boot file */
                overl &= ~1;
                p = dhcp->bootfile;
                e = p + sizeof(dhcp->bootfile);
                preOverloadBl = bl;
                preOverloadBp = bp;
                preOverloadOp = op;
            }
            else if (overl & 2)
            {
                /* bit 2 set means parse server name */

                /* if the bootfile was just parsed for dhcp options... */
                if (overlperm & 1)
                {
                    /* ensure the end option was found and the area after the end is entirely pad options*/
                    uint8_t nullBlockBootFile[sizeof(dhcp->bootfile)];
                    memset(nullBlockBootFile, 0, sizeof(dhcp->bootfile));
                    if (!overloadEndFound || memcmp(p, nullBlockBootFile, e - p) != 0)
                    {
                        /* otherwise reset us back to before the overload options were parsed */
                        bl = preOverloadBl;
                        bp = preOverloadBp;
                        op = preOverloadOp;
                    }
                    /* reset overload end found flag */
                    overloadEndFound = 0;
                }


                overl &= ~2;
                p = dhcp->servername;
                e = p + sizeof(dhcp->servername);

                preOverloadBl = bl;
                preOverloadBp = bp;
                preOverloadOp = op;
            }
            else if (overl & 0x80)
            {
                overloadEndFound = 1;
                goto exit;
            }
            else  goto exit;
            continue;
        case DhcpProtOption_OptionsOverloaded:
            /* Ensure we only get this option once by setting
             * the last bit as well as the value.
             * This is valid because only the first two bits
             * actually mean anything in RFC2132 Section 9.3 */
            if (!overl) overl = overlperm = 0x80 | p[1];
            break;
        default:
            break;
        }
        l = *p++;
        p += l;
    }

exit:
    /* if the bootfile was parsed for dhcp options... */
    if (overl & 0x80)
    {
        /* ensure the end option was found */
        if (!overloadEndFound)
        {
            /* otherwise reset us back to before the overload options were parsed */
            bl = preOverloadBl;
            bp = preOverloadBp;
            op = preOverloadOp;
        }
        else
        {
            /* ensure the area after the end option is entirely pad options */
            if (overlperm & 1)
            {
                uint8_t nullBlockBootFile[sizeof(dhcp->bootfile)];
                memset(nullBlockBootFile, 0, sizeof(dhcp->bootfile));
                if (memcmp(p, nullBlockBootFile, e - p) != 0)
                {
                    /* otherwise reset us back to before the overload options were parsed */
                    bl = preOverloadBl;
                    bp = preOverloadBp;
                    op = preOverloadOp;
                }

            }
            /* ensure the area after the end option is entirely pad options */
            if (overlperm & 2)
            {
                uint8_t nullBlockServerName[sizeof(dhcp->servername)];
                memset(nullBlockServerName, 0, sizeof(dhcp->servername));
                if (memcmp(p, nullBlockServerName, e - p) != 0)
                {
                    /* otherwise reset us back to before the overload options were parsed */
                    bl = preOverloadBl;
                    bp = preOverloadBp;
                    op = preOverloadOp;
                }

            }
        }
    }

    if (len)
    {
        *len = bl;
    }
    if (bp)
    {
        memcpy(bp, op, ol);
        return (const uint8_t *)optionBuffer;
    }
    if (op) return op;
    return NULL;
} //NOLINT(impl/function_size)

int32_t Util::GetOptionAddr(uint8_t optionBuffer[DhcpSizes_Option + DhcpSizes_Bootfile + DhcpSizes_ServerName],
                            in_addr_t *a, const DhcpMessage *dhcp,
                            uint8_t option)
{
    const uint8_t *p;
    size_t len;
    in_addr_t netAddr;

    p = Util::GetOption(optionBuffer, dhcp, option, &len);
    if (!p || len < (ssize_t)sizeof(*a)) return -1;
    memcpy(&netAddr, p, sizeof(*a));
    *a = netAddr;

    return 0;
}

int32_t Util::GetOptionUint32(uint8_t optionBuffer[DhcpSizes_Option + DhcpSizes_Bootfile + DhcpSizes_ServerName],
                              uint32_t *i, const DhcpMessage *dhcp, uint8_t option)
{
    const uint8_t *p;
    size_t len;
    uint32_t d;

    p = Util::GetOption(optionBuffer, dhcp, option, &len);
    if (!p || len < (ssize_t)sizeof(d)) return -1;
    memcpy(&d, p, sizeof(d));
    if (i)
    {
        *i = d;
    }
    return 0;
}

int32_t Util::GetOptionUint8(uint8_t optionBuffer[DhcpSizes_Option + DhcpSizes_Bootfile + DhcpSizes_ServerName],
                             uint8_t *i, const DhcpMessage *dhcp, uint8_t option)
{
    const uint8_t *p;
    size_t len;

    p = Util::GetOption(optionBuffer, dhcp, option, &len);
    if (!p || len < (ssize_t)sizeof(*p)) return -1;
    if (i)
    {
        *i = *(p);
    }
    return 0;
}

void Util::GetLeaseFromDhcpMessage(uint8_t *pOptionBuffer, Lease *pReturnedLease, DhcpMessage *dhcp)
{
    uint32_t dataU32;

    pReturnedLease->cookie = dhcp->cookie;

    // BOOTP does not set yiaddr for replies when ciaddr is set
    if (dhcp->yiaddr)
    {
        pReturnedLease->addr.S_addr = dhcp->yiaddr;
    }
    else
    {
        pReturnedLease->addr.S_addr = dhcp->ciaddr;
    }

    // Get lease address
    if (GetOptionAddr(pOptionBuffer, &pReturnedLease->subnetMask.S_addr, dhcp, DhcpProtOption_SubnetMask) == -1)
    {
        pReturnedLease->subnetMask.S_addr = Util::Ipv4GetNetmask(pReturnedLease->addr.S_addr);
    }

    // Get broadcast address
    if (GetOptionAddr(pOptionBuffer, &pReturnedLease->broadcastAddr.S_addr, dhcp, DhcpProtOption_Broadcast) == -1)
    {
        pReturnedLease->broadcastAddr.S_addr = pReturnedLease->addr.S_addr | ~pReturnedLease->subnetMask.S_addr;
    }

    // Get lease time
    if (GetOptionUint32(pOptionBuffer, &dataU32, dhcp, DhcpProtOption_LeaseTime) != 0)
    {
        dataU32 = Config_InfiniteTime;
    }
    pReturnedLease->leaseTime = TimeSpan::FromSeconds(nn::socket::InetNtohl(dataU32));


    // Get renewal time
    if (GetOptionUint32(pOptionBuffer, &dataU32, dhcp, DhcpProtOption_RenewalTime) != 0)
    {
        dataU32 = 0;
    }
    pReturnedLease->renewalTime = TimeSpan::FromSeconds(nn::socket::InetNtohl(dataU32));

    // Get rebind time
    if (GetOptionUint32(pOptionBuffer, &dataU32, dhcp, DhcpProtOption_RebindTime) != 0)
    {
        dataU32 = 0;
    }
    pReturnedLease->rebindTime = TimeSpan::FromSeconds(nn::socket::InetNtohl(dataU32));

    // Get server address
    if (GetOptionAddr(pOptionBuffer, &pReturnedLease->serverAddr.S_addr, dhcp, DhcpProtOption_ServerId) != 0)
    {
        pReturnedLease->serverAddr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
    }

    // Get current timestamp
    pReturnedLease->leaseTimeStamp = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
}

bool Util::ApplyLeaseLimits(Lease *pLease)
{
    bool limitsApplied = false;
    if (pLease->leaseTime == TimeSpan::FromSeconds(Config_InfiniteTime))
    {
        pLease->rebindTime = pLease->renewalTime = pLease->leaseTime;
    }
    else
    {
        if (pLease->leaseTime < TimeSpan::FromSeconds(Config_MinLeaseTimeSeconds))
        {
            DHCPC_LOG_WARN("leaseTime of %ds is being limited to %d seconds.\n",
                           (int)pLease->leaseTime.GetSeconds(), (int)Config_MinLeaseTimeSeconds);
            pLease->leaseTime = TimeSpan::FromSeconds(Config_MinLeaseTimeSeconds);
            limitsApplied = true;
        }

        if (pLease->rebindTime.GetSeconds() == 0)
        {
            uint32_t rebindTime = (uint32_t)((pLease->leaseTime.GetSeconds() *
                                              (int64_t)Config_DefaultRebindRatio) / 1000LL);
            pLease->rebindTime = TimeSpan::FromSeconds(rebindTime);
        }
        else if (pLease->rebindTime >= pLease->leaseTime)
        {
            uint32_t rebindTime = (uint32_t)((pLease->leaseTime.GetSeconds() *
                                              (int64_t)Config_DefaultRebindRatio) / 1000LL);
            DHCPC_LOG_WARN("rebindTime greater than leaseTime, forcing rebindTime to %u seconds",
                           rebindTime);
            pLease->rebindTime = TimeSpan::FromSeconds(rebindTime);
            limitsApplied = true;
        }

        if (pLease->renewalTime.GetSeconds() == 0)
        {
            uint32_t renewalTime = (uint32_t)((pLease->leaseTime.GetSeconds() *
                                               (int64_t)Config_DefaultRenewalRatio) / 1000LL);
            pLease->renewalTime = TimeSpan::FromSeconds(renewalTime);
        }
        else if (pLease->renewalTime >= pLease->rebindTime)
        {
            uint32_t renewalTime = (uint32_t)((pLease->leaseTime.GetSeconds() *
                                               (int64_t)Config_DefaultRenewalRatio) / 1000LL);
            DHCPC_LOG_WARN("renewalTime greater than rebindTime, forcing renewalTime to %u seconds",
                           renewalTime);
            pLease->renewalTime = TimeSpan::FromSeconds(renewalTime);
            limitsApplied = true;
        }
    }
    return limitsApplied;
}

uint32_t Util::Ipv4GetNetmask(uint32_t addr)
{
    uint32_t dst;

    if (addr == 0) return 0;

    dst = addr;
    if (nn::socket::In_ClassA(dst)) return nn::socket::InetNtohl(static_cast<uint32_t>(nn::socket::InClassA::In_ClassA_Net));
    if (nn::socket::In_ClassB(dst)) return nn::socket::InetNtohl(static_cast<uint32_t>(nn::socket::InClassB::In_ClassB_Net));
    if (nn::socket::In_ClassC(dst)) return nn::socket::InetNtohl(static_cast<uint32_t>(nn::socket::InClassC::In_ClassC_Net));

    return 0;
}

} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn
