﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief DHCP client interface
 *
 * @details
 */
#include <random>

namespace nn { namespace bsdsocket { namespace dhcpc {

class DhcpcMain;
class FdEvent;

class Interface : public Fsm
{
public:
    enum State
    {
        State_Null = 0,
        State_WaitBeforeInit,
        State_Init,
        State_Selecting,
        State_Request,
        State_Bound,
        State_WaitBeforeInitReboot,
        State_InitReboot,
        State_Rebooting,
        State_Renew,
        State_Rebind,
        State_Inform,
        State_Probe,
        State_Failure,
        State_Maximum,
    };
    enum Event
    {
        Event_LinkStateChange = Fsm::Event_Base,
        Event_LinkUp,
        Event_Try,
        Event_Proceed,
        Event_Error,
        Event_ReTransmitTimer,
        Event_ReceivedBootP,
        Event_ReceivedOffer,
        Event_ReceivedAck,
        Event_ReceivedNack,
        Event_ReceivedForceRenew,
        Event_T1RenewTimer,
        Event_T2RebindTimer,
        Event_ExpireTimer,
        Event_Finalize,
        Event_UnrecoverableError,
        Event_Maximum,
    };

    struct WorkingDhcpMessage
    {
        bool                isValid;
        DhcpMessage message;
    };

    Interface();
    ~Interface()
    {
    }

    Result Initialize(DhcpcMain *pMain, const InterfaceConfigType *pIfConfig);
    Result Stop(DeferredRequest *pDr);
    Result Finalize();
    Result GetOptionData(DhcpProtOption option, void *pRetData,
                         size_t bufferSize, size_t *pRetSize);
    Result GetState(char *pReturnedStateName, size_t stateNameSizeLimit);
    Result ForceRenew();
    void GetHardwareAddress(uint8_t *pRetAddress);
    void GetInterfaceName(char *name, size_t maxSize);


private:
    static const nn::socket::InAddr        m_InAddrAny;
    static const nn::socket::InAddr        m_InAddrBroadcast;
    static const char           *EventNames[];
    static const char           *StateNames[];

    DhcpcMain                   *m_pMain;
    InterfaceConfigType         m_IfConfig;
    PacketManager               m_Packet;
    uint32_t                    m_Xid;
    Util::ToABuffer             m_ToABuffer;
    char                        m_ClientId[Config_MaxClientIdSize];

    PacketManager::DhcpPacket   m_ReceivePacket;
    uint8_t                     m_OptionBuffer[DhcpSizes_Option + DhcpSizes_Bootfile + DhcpSizes_ServerName];

    PacketManager::DhcpPacket   m_SendPacket;
    size_t                      m_SendPacketSize;

    WorkingDhcpMessage          m_NextOffer;
    Lease                       m_NextLease;
    WorkingDhcpMessage          m_Offer;
    Lease                       m_Lease;

    uint32_t                    m_ShutdownOptions;

    TimeSpan                    m_ReTransBackoffInterval;
    TimeSpan                    m_NakBackoffInterval;

    TimeSpan                    m_RequestSentTime;
    TimeSpan                    m_LeaseAquiredTime;

    TimeSpan                    m_InitWaitTime;

    uint16_t                    m_UpTimeSeconds;

    uint8_t                     m_RetryCount;

    LocalEventType              m_TryTimerEvent;
    LocalEventType              m_ReTransTimerEvent;
    LocalEventType              m_RenewTimerEvent;
    LocalEventType              m_RebindTimerEvent;
    LocalEventType              m_ExpireTimerEvent;
    Indication                  m_PendingClientIndication;
    Result                      m_FailureCause;
    DeferredRequest             *m_pDeferredFinalize;

    std::mt19937 m_Mt;

    void HandleFsmError(Result result);

    // State handlers
    Result FsmHandler(int32_t state, LocalEventDataType *pEvent);
    Result NullStateHandler(LocalEventDataType *pEvent);
    Result WaitBeforeInitStateHandler(LocalEventDataType *pEvent);
    Result InitStateHandler(LocalEventDataType *pEvent);
    Result WaitLinkUpStateHandler(LocalEventDataType *pEvent);
    Result SelectingStateHandler(LocalEventDataType *pEvent);
    Result RequestStateHandler(LocalEventDataType *pEvent);
    Result BoundStateHandler(LocalEventDataType *pEvent);
    Result WaitBeforeInitRebootStateHandler(LocalEventDataType *pEvent);
    Result InitRebootStateHandler(LocalEventDataType *pEvent);
    Result RebootingStateHandler(LocalEventDataType *pEvent);
    Result RenewStateHandler(LocalEventDataType *pEvent);
    Result RebindStateHandler(LocalEventDataType *pEvent);
    Result RebootStateHandler(LocalEventDataType *pEvent);
    Result InformStateHandler(LocalEventDataType *pEvent);
    Result ProbeStateHandler(LocalEventDataType *pEvent);
    Result FailureStateHandler(LocalEventDataType *pEvent);

    void DispatchReceivedDhcpMessage(DhcpMessage *dhcp, nn::socket::InAddr *pFrom);
    void HandleDeInit();
    void InitXid();
    void GenerateRebootLease();
    Result MakeMessage(DhcpMessage *dhcp, DhcpProtMsgType msgType,
                       const nn::socket::InAddr *ciaddr, size_t *pReturnedSize);
    Result MakeVivoMessage(DhcpProtMsgType msgType, uint8_t *m, uint8_t **pP);
    Result MakeParamReqMessage(DhcpProtMsgType msgType, uint8_t *m, uint8_t **pP);
    int DoMtu(int mtu);
    void UpdateReTransBackoffInterval(bool reset);
    void UpdateNakBackoffInterval(bool reset);
    void UpdateRenewReTransInterval();
    void UpdateRebindReTransInterval();
    void DoClientCallback(Indication indication);
    Result HandleFinalizeEvent();
    void HandleUnrecoverableErrorEvent(LocalEventDataType *pEvent);
    Result TakeReceiptOfNext(LocalEventDataType *pEvent);
    Result DeclareLeaseInvalid();
    void FdMonRawPacketCallback();
    static void FdMonRawPacketCallbackStatic(void *pContext)
    {
        Interface *pI = reinterpret_cast<Interface *>(pContext);
        pI->FdMonRawPacketCallback();
    }
    void LogContextual(Util::LogLevel severity, const char *format, ...);
};

} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn

