﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcpc_PrivateIncludes.h"

namespace nn { namespace bsdsocket { namespace dhcpc {

Result Fsm::Initialize(DhcpcMain *pMain, char *name,
                       const char **eventNames, int32_t numEvents,
                       const char **stateNames, int32_t numStates,
                       int32_t initialState)
{
    Result result;
    LocalEventDataType initialEntryEvent = LOCAL_EVENT_DATA_ZERO;
    m_pMain           = pMain;
    m_FsmTraceEnabled = true;
    m_EventNames      = eventNames;
    m_StateNames      = stateNames;
    m_NumEvents       = numEvents;
    m_NumStates       = numStates;
    m_State           = initialState;
    m_NextState       = initialState;
    strlcpy(m_FsmName, name, sizeof(m_FsmName));
    m_pMain->m_LocalEventManager.RegisterPool(this, &m_LocalEventPool, HandleEventStatic,
                                              &m_LocalEventStorage[0], sizeof(m_LocalEventStorage[0]),
                                              sizeof(m_LocalEventStorage) / sizeof(m_LocalEventStorage[0]));
    initialEntryEvent.eventId = Event_Entry;
    m_CurrentEvent = initialEntryEvent.eventId;
    result = FsmHandler(initialState, &initialEntryEvent);

    return result;
}

Result Fsm::Finalize()
{
    m_pMain->m_LocalEventManager.UnRegisterPool(&m_LocalEventPool);
    m_pMain = NULL;
    return ResultSuccess();
}

void Fsm::SetName(const char *name)
{
    strlcpy(m_FsmName, name, sizeof(m_FsmName));
}

const char* Fsm::GetName()
{
    return m_FsmName;
}

void Fsm::QueueFsmEvent(LocalEventDataType *pEvent)
{
    DHCPC_ABORT_UNLESS_SUCCESS(m_pMain->m_LocalEventManager.QueueEvent(&m_LocalEventPool, pEvent));
}

void Fsm::QueueFsmEvent(int32_t eventId)
{
    LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
    evtData.eventId       = eventId;
    evtData.stateOfOrigin = m_State;
    evtData.status        = ResultSuccess();
    QueueFsmEvent(&evtData);
}

void Fsm::QueueFsmEvent(int32_t eventId, Result status)
{
    LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
    evtData.eventId       = eventId;
    evtData.stateOfOrigin = m_State;
    evtData.status        = status;
    QueueFsmEvent(&evtData);
}

void Fsm::QueueFsmEvent(int32_t eventId, Result status, uint32_t data)
{
    LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
    evtData.eventId            = eventId;
    evtData.stateOfOrigin      = m_State;
    evtData.status             = status;
    evtData.args[0].dataUint32 = data;
    QueueFsmEvent(&evtData);
}

void Fsm::QueueFsmEvent(int32_t eventId, Result status, void* pData)
{
    LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
    evtData.eventId            = eventId;
    evtData.stateOfOrigin      = m_State;
    evtData.status             = status;
    evtData.args[0].pData      = pData;
    QueueFsmEvent(&evtData);
}

void Fsm::QueueFsmEvent(int32_t eventId, void *pData)
{
    LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
    evtData.eventId            = eventId;
    evtData.stateOfOrigin      = m_State;
    evtData.args[0].pData      = pData;
    QueueFsmEvent(&evtData);
}

Result Fsm::DispatchFsmEvent(int32_t eventId, void *pData)
{
    LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
    evtData.eventId            = eventId;
    evtData.stateOfOrigin      = m_State;
    evtData.args[0].pData      = pData;
    return HandleEvent(&evtData);
}

Result Fsm::DispatchFsmEvent(int32_t eventId, void *pData0, void *pData1)
{
    LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
    evtData.eventId            = eventId;
    evtData.stateOfOrigin      = m_State;
    evtData.args[0].pData      = pData0;
    evtData.args[1].pData      = pData1;
    return HandleEvent(&evtData);
}

Result Fsm::DispatchFsmEvent(int32_t eventId)
{
    LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
    evtData.eventId            = eventId;
    evtData.stateOfOrigin      = m_State;
    return HandleEvent(&evtData);
}

LocalEventType* Fsm::CreateTimedFsmEvent(int32_t eventId)
{
    LocalEventType *pEvent =  m_pMain->m_LocalEventManager.AllocEvent(&m_LocalEventPool, false);
    if (pEvent != NULL)
    {
        pEvent->data.eventId = eventId;
    }
    return pEvent;
}

void Fsm::DestroyTimedFsmEvent(LocalEventType *pEvent)
{
    m_pMain->m_LocalEventManager.FreeEvent(pEvent);
}

void Fsm::InitStaticTimedFsmEvent(LocalEventType *pEvent, int32_t eventId)
{
    m_pMain->m_LocalEventManager.InitStaticEvent(&m_LocalEventPool, pEvent);
    pEvent->data.eventId = eventId;
}

// This method will intentionally abort if timer was already running,
// you must know for sure it's not running.
void Fsm::StartTimedFsmEvent(LocalEventType *pEvent, TimeSpan relativeTime)
{
    pEvent->data.stateOfOrigin = m_State;
    m_pMain->m_LocalEventManager.StartTimedEvent(pEvent, relativeTime);
}

// This method can handle a timer that is already running, at which
// point it will be restarted with new specified timespan.
void Fsm::ReStartTimedFsmEvent(LocalEventType *pEvent, TimeSpan relativeTime)
{
    StopTimedFsmEvent(pEvent);
    pEvent->data.stateOfOrigin = m_State;
    m_pMain->m_LocalEventManager.StartTimedEvent(pEvent, relativeTime);
}

void Fsm::StopTimedFsmEvent(LocalEventType *pEvent)
{
    m_pMain->m_LocalEventManager.FreeEvent(pEvent);
}

LocalEventPoolType* Fsm::GetLocalEventPool()
{
    return &m_LocalEventPool;
}

void Fsm::SetState(int32_t nextState)
{
    if (nextState < m_NumStates)
    {
        m_NextState = nextState;
    }
    else
    {
        DHCPC_ABORT("%s FSM - invalid call to SetState(%d)",
                        m_FsmName, nextState);
    }
}

bool Fsm::IsEventStale(LocalEventDataType *pEventData)
{
    return (pEventData->stateOfOrigin == m_State) ? false : true;
}

int32_t Fsm::GetState()
{
    return m_State;
}

int32_t Fsm::GetCurrentEvent()
{
    return m_CurrentEvent - Event_Base;
}

void Fsm::SetFsmTraceEnable(bool isEnabled)
{
    m_FsmTraceEnabled = isEnabled;
}

Result Fsm::HandleEvent(LocalEventDataType *pEvent)
{
    Result result = ResultSuccess();
    int32_t eventNameIndex = pEvent->eventId - Event_Base;

    if (m_FsmTraceEnabled)
    {
        DHCPC_LOG_INFO("%s: state %s, event %s.\n",
                           m_FsmName, m_StateNames[m_State],
                           m_EventNames[eventNameIndex]);
    }

    // Execute current state event
    m_CurrentEvent = pEvent->eventId;
    if ( !(result = FsmHandler(m_State, pEvent)).IsSuccess())
    {
        DHCPC_LOG_WARN("%s: Error %d when processing state %s, event %s.\n",
                           m_FsmName, result, m_StateNames[m_State],
                           m_EventNames[eventNameIndex]);
    }

    // Look for state transition
    if (m_State != m_NextState)
    {
        LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;

        if (m_FsmTraceEnabled)
        {
            DHCPC_LOG_INFO("%s: state %s -> %s.\n",
                               m_FsmName, m_StateNames[m_State],
                               m_StateNames[m_NextState]);
        }

        // Exit from prior state
        evtData.eventId       = m_CurrentEvent = Event_Exit;
        evtData.stateOfOrigin = m_State;
        if ( !(result = FsmHandler(m_State, &evtData)).IsSuccess())
        {
            DHCPC_LOG_WARN("%s: Error %d when processing state %s exit.\n",
                               m_FsmName, result, m_StateNames[m_State]);
        }

        // Entry into next state
        m_State               = m_NextState;
        evtData.eventId       = m_CurrentEvent = Event_Entry;
        evtData.stateOfOrigin = m_State;
        if ( !(result = FsmHandler(m_State, &evtData)).IsSuccess())
        {
            DHCPC_LOG_WARN("%s: Error %d when processing state %s entry.\n",
                               m_FsmName, result, m_StateNames[m_State]);
        }
    }

    return result;
}

} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn


