﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "dhcpc_PrivateIncludes.h"

namespace nn { namespace bsdsocket { namespace dhcpc {

FdMonitor::FdMonitor()
    : m_pMain(nullptr)
    , m_pTimerEvent(nullptr)
{
    memset(m_Registrations, 0, sizeof(m_Registrations));
}

FdMonitor::~FdMonitor()
{

}

Result FdMonitor::Initialize(DhcpcMain *pMain)
{
    m_pMain       = pMain;
    m_pTimerEvent = nullptr;
    memset(m_Registrations, 0, sizeof(m_Registrations));
    pMain->m_LocalEventManager.RegisterPool(this, &m_LocalEventPool, LocalEventCallbackStatic,
                                            &m_LocalEventStorage[0], sizeof(m_LocalEventStorage[0]),
                                            sizeof(m_LocalEventStorage) / sizeof(m_LocalEventStorage[0]));
    DHCPC_ABORT_NULL(m_pTimerEvent =  m_pMain->m_LocalEventManager.AllocEvent(&m_LocalEventPool, false));

    return ResultSuccess();
}

Result FdMonitor::Finalize()
{
    m_pMain->m_LocalEventManager.UnRegisterPool(&m_LocalEventPool);
    memset(m_Registrations, 0, sizeof(m_Registrations));
    m_pMain       = nullptr;
    m_pTimerEvent = nullptr;
    return ResultSuccess();
}

Result FdMonitor::Register(int fd,
                           LocalEventPoolType *pLocalEventPool,
                           int32_t eventId)
{
    Result result = ResultMaximumExceeded();
    for (int32_t index = 0; index < MaxPolledEvent; index++)
    {
        UserRegistrationType *pUer = m_Registrations + index;
        if ( !pUer->active)
        {
            pUer->active                    = true;
            pUer->mode                      = UserRegistrationMode_Event;
            pUer->polledFd.fd               = fd;
            pUer->pollPeriodMs              = SlowPollPeriod;
            pUer->cfg.event.pLocalEventPool = pLocalEventPool;
            pUer->cfg.event.eventId         = eventId;
            result = ResultSuccess();
            UpdatePollTimer();
            break;
        }
    }
    return result;
}

Result FdMonitor::Register(int fd,
                           void (*callback)(void *),
                           void *pContext)
{
    Result result = ResultMaximumExceeded();
    for (int32_t index = 0; index < MaxPolledEvent; index++)
    {
        UserRegistrationType *pUer = m_Registrations + index;
        if ( !pUer->active)
        {
            pUer->active                    = true;
            pUer->mode                      = UserRegistrationMode_Callback;
            pUer->polledFd.fd               = fd;
            pUer->pollPeriodMs              = SlowPollPeriod;
            pUer->cfg.callback.callback     = callback;
            pUer->cfg.callback.pContext     = pContext;
            result = ResultSuccess();
            UpdatePollTimer();
            break;
        }
    }
    return result;
}

Result FdMonitor::UnRegister(int fd)
{
    Result result = ResultInvalidInternalState();
    for (int32_t index = 0; index < MaxPolledEvent; index++)
    {
        UserRegistrationType *pUer = m_Registrations + index;
        if (pUer->active &&
            pUer->polledFd.fd == fd)
        {
            pUer->polledFd.fd     = -1;
            pUer->active          = false;
            pUer->mode            = UserRegistrationMode_Invalid;
            result = ResultSuccess();
            UpdatePollTimer();
            break;
        }
    }
    return result;
}

Result FdMonitor::SetPollPeriod(int fd, int pollPeriodMs)
{
    Result result = ResultInvalidInternalState();
    for (int32_t index = 0; index < MaxPolledEvent; index++)
    {
        UserRegistrationType *pUer = m_Registrations + index;
        if (pUer->active &&
            pUer->polledFd.fd == fd)
        {
            pUer->pollPeriodMs = pollPeriodMs;
            result = ResultSuccess();
            UpdatePollTimer();
            break;
        }
    }
    return result;
}

void FdMonitor::Poll()
{
    int32_t numFds = 0;
    nn::socket::PollFd fdList[MaxPolledEvent];
    UserRegistrationType *uerList[MaxPolledEvent];
    memset(fdList,0,sizeof(fdList));
    memset(uerList,0,sizeof(uerList));
    for (int32_t index = 0; index < MaxPolledEvent; index++)
    {
        UserRegistrationType *pUer = m_Registrations + index;
        if (pUer->active)
        {
            fdList[numFds].fd      = pUer->polledFd.fd;
            fdList[numFds].events  = nn::socket::PollEvent::PollIn;
            fdList[numFds].revents = nn::socket::PollEvent::PollNone;
            uerList[numFds] = pUer;
            numFds++;
        }
    }

    if (numFds > 0)
    {
        bool keepTrying = false;
        do
        {
            int count = nn::socket::Poll(fdList, numFds, 0);
            keepTrying = ((count == -1) && ((nn::socket::GetLastError() == nn::socket::Errno::EAgain) || (nn::socket::GetLastError() == nn::socket::Errno::EIntr)));
        }while (keepTrying);

        for (int32_t index = 0; index < numFds; index++)
        {
            if ((fdList[index].revents & nn::socket::PollEvent::PollIn) != nn::socket::PollEvent::PollNone)
            {
                UserRegistrationType *pUer = uerList[index];
                if (pUer->active)
                {
                    switch (pUer->mode)
                    {
                    case UserRegistrationMode_Event:
                        {
                            LocalEventDataType evtData = LOCAL_EVENT_DATA_ZERO;
                            evtData.eventId = pUer->cfg.event.eventId;
                            m_pMain->m_LocalEventManager.QueueEvent(pUer->cfg.event.pLocalEventPool,
                                                                    &evtData);
                            break;
                        }
                    case UserRegistrationMode_Callback:
                        {
                            (*pUer->cfg.callback.callback)(pUer->cfg.callback.pContext);
                            break;
                        }
                    default:
                        break;
                    }
                }
            }
        }
    }
}

void FdMonitor::UpdatePollTimer()
{
    int activeFdCount = 0;
    nn::TimeSpan pollTimeSpan = TimeSpan::FromMilliSeconds(SlowPollPeriod);

    // Make sure timer is stopped
    m_pMain->m_LocalEventManager.StopTimedEvent(m_pTimerEvent);

    // For all fd registrations
    for (int32_t index = 0; index < MaxPolledEvent; index++)
    {
        UserRegistrationType *pUer = m_Registrations + index;
        if (pUer->active)
        {
            activeFdCount++;

            // Let the fastest requirement prevail
            if((pUer->pollPeriodMs > 0) &&
               (pUer->pollPeriodMs < static_cast<int>(pollTimeSpan.GetMilliSeconds())))
            {
                pollTimeSpan = TimeSpan::FromMilliSeconds(pUer->pollPeriodMs);
            }
        }
    }

    // Don't bother restarting timer if there are no active FDs
    if(activeFdCount > 0)
    {
        m_pMain->m_LocalEventManager.StartTimedEvent(m_pTimerEvent, pollTimeSpan);
    }
}

void FdMonitor::LocalEventCallback(LocalEventType *pEvent)
{
    NN_UNUSED(pEvent);
    Poll();
    UpdatePollTimer();
}




} // namespace dhcpc
} // namespace bsdsocket
} // namespace nn




