﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cfg_PrivateIncludes.h"

namespace nn { namespace bsdsocket { namespace cfg {

Result Ipv4::IfAddress(int s, const char *ifname, nn::socket::InAddr addr, nn::socket::InAddr mask,
                       nn::socket::InAddr bcast, int metric, Operation op)
{
    Result result = ResultSuccess();
    nn::socket::IfAliasReq ifra;
    nn::socket::IfReq ifr;
    nn::socket::SockAddrIn *sa;
    bool addOperation = (op == Operation_Add) ? true : false;

    memset(&ifra, 0, sizeof(ifra));
    nn::util::Strlcpy(ifra.ifra_name, ifname, sizeof(ifra.ifra_name));

    sa = (nn::socket::SockAddrIn *)(void *)&ifra.ifra_addr;
    sa->sin_family = nn::socket::Family::Af_Inet;
    sa->sin_len = sizeof(*sa);
    sa->sin_addr.S_addr = addr.S_addr;

    sa = (nn::socket::SockAddrIn *)(void *)&ifra.ifra_mask;
    sa->sin_family = nn::socket::Family::Af_Inet;
    sa->sin_len = sizeof(*sa);
    sa->sin_addr.S_addr = mask.S_addr;

    if ((bcast.S_addr != 0) && addOperation)
    {
        sa = (nn::socket::SockAddrIn *)(void *)&ifra.ifra_broadaddr;
        sa->sin_family = nn::socket::Family::Af_Inet;
        sa->sin_len = sizeof(*sa);
        sa->sin_addr.S_addr = bcast.S_addr;
    }
    if (nn::socket::Ioctl(s, (addOperation) ? nn::socket::IoctlCommandPrivate::SiocAIfAddr : nn::socket::IoctlCommandPrivate::SiocDIfAddr, (caddr_t)&ifra, sizeof(ifra)) == -1)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        if (addOperation)
        {
            CFG_LOG_WARN("Ioctl( nn::socket::SiocAIfAddr ) failed, errno=%d.\n", lastErrno);
            switch (lastErrno)
            {
            case nn::socket::Errno::ENxIo:
                result = ResultIfRemoved();
                break;
            case nn::socket::Errno::EInval:
                result = ResultInvalidIfAddress();
                break;
            case nn::socket::Errno::ENoBufs:
                result = ResultMemAllocStackError();
                break;
            default:
                result = ResultUnknownStackError();
                break;
            }
        }
        else
        {
            CFG_LOG_WARN("Ioctl( SiocDIfAddr ) failed, errno=%d.\n", lastErrno);
            switch (lastErrno)
            {
            case nn::socket::Errno::ENxIo:
                result = ResultIfRemoved();
                break;
            case nn::socket::Errno::EAddrNotAvail:
                // OK
                break;
            default:
                result = ResultUnknownStackError();
                break;
            }
        }
        return result;
    }

    memset(&ifr, 0, sizeof(ifr));
    nn::util::Strlcpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
    ifr.ifr_metric = metric;
    if (nn::socket::Ioctl(s, nn::socket::IoctlCommandPrivate::SiocSIfMetric, (caddr_t)&ifr, sizeof(ifr)) == -1)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        CFG_LOG_WARN("Ioctl( SiocSIfMetric ) failed, errno=%d.\n", lastErrno);
        switch (lastErrno)
        {
        case nn::socket::Errno::ENxIo:
            result = ResultIfRemoved();
            break;
        default:
            result = ResultUnknownStackError();
            break;
        }
        return result;
    }

    memset(&ifr, 0, sizeof(ifr));
    nn::util::Strlcpy(ifr.ifr_name, ifname, sizeof(ifr.ifr_name));
    ifr.ifr_addr  = ifra.ifra_addr;
    ifr.ifr_flags = nn::socket::IfrFlag::Iff_Up | nn::socket::IfrFlag::Iff_Running | ((bcast.S_addr != 0) ? nn::socket::IfrFlag::Iff_Broadcast : nn::socket::IfrFlag::Iff_None);
    if (nn::socket::Ioctl(s, nn::socket::IoctlCommandPrivate::SiocSIfFlags, (caddr_t)&ifr, sizeof(ifr)) == -1)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        CFG_LOG_WARN("Ioctl( SiocSIfFlags ) failed, errno=%d.\n", lastErrno);
        switch (lastErrno)
        {
        case nn::socket::Errno::ENxIo:
            result = ResultIfRemoved();
            break;
        default:
            result = ResultUnknownStackError();
            break;
        }
        return result;
    }

    return result;
}

Result Ipv4::VerifyIfAddress(int s, const char *name, nn::socket::InAddr inaddr, nn::socket::InAddr inmask, int metric)
{
    Result result = ResultSuccess();
    struct ifreq req;
    nn::socket::SockAddrIn *sa;
    memset(&req, 0, sizeof(req));
    nn::util::Strlcpy(req.ifr_name, name, sizeof(req.ifr_name));
    sa = (nn::socket::SockAddrIn *)(void *)&req.ifr_addr;
    sa->sin_family = nn::socket::Family::Af_Inet;
    sa->sin_len = sizeof(*sa);
    sa->sin_addr.S_addr = inaddr.S_addr;
    if (nn::socket::Ioctl(s, nn::socket::IoctlCommandPrivate::SiocGIfAddr, (caddr_t)&req, sizeof(req)) == -1)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        CFG_LOG_WARN("Ioctl( SiocGIfAddr ) failed, errno=%d.\n", lastErrno);
        switch (lastErrno)
        {
        case nn::socket::Errno::ENxIo:
            result = ResultIfRemoved();
            break;
        default:
            result = ResultUnknownStackError();
            break;
        }
        return result;
    }
    if (inaddr.S_addr != sa->sin_addr.S_addr)
    {
        CFG_LOG_WARN("Interface address verify error.\n");
        return ResultUnknownStackError();
    }
    if (nn::socket::Ioctl(s, nn::socket::IoctlCommandPrivate::SiocGIfNetmask, (caddr_t)&req, sizeof(req)) == -1)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        CFG_LOG_WARN("Ioctl( SiocGIfNetmask ) failed, errno=%d.\n", lastErrno);
        switch (lastErrno)
        {
        case nn::socket::Errno::ENxIo:
            result = ResultIfRemoved();
            break;
        default:
            result = ResultUnknownStackError();
            break;
        }
        return result;
    }
    if (inmask.S_addr != sa->sin_addr.S_addr)
    {
        CFG_LOG_WARN("Interface mask verify error.\n");
        return ResultUnknownStackError();
    }
    if (nn::socket::Ioctl(s, nn::socket::IoctlCommandPrivate::SiocGIfMetric, (caddr_t)&req, sizeof(req)) == -1)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        CFG_LOG_WARN("Ioctl( SiocGIfMetric ) failed, errno=%d.\n", lastErrno);
        switch (lastErrno)
        {
        case nn::socket::Errno::ENxIo:
            result = ResultIfRemoved();
            break;
        default:
            result = ResultUnknownStackError();
            break;
        }
        return result;
    }
    if (req.ifr_metric != metric)
    {
        CFG_LOG_WARN("Interface metric verify error.\n");
        return ResultUnknownStackError();
    }
    return result;
}

Result Ipv4::IfRoute(int s, nn::socket::InAddr ifNet, nn::socket::InAddr ifAddr, uint16_t sdlIndex,
                     Route *pRt, Ipv4::Operation op)
{
    Result result = ResultSuccess();
    union sockunion
    {
        nn::socket::SockAddr sa;
        nn::socket::SockAddrIn sin;
        nn::socket::SockAddrDl sdl;
        nn::socket::SockAddrStorage ss;
    } su;
    struct rtm
    {
        nn::socket::RtMsgHdr hdr;
        char buffer[sizeof(su) * 5];
    } rtm;
    char *bp = rtm.buffer;
    size_t l;

#define ADDSU {                                     \
        l = nn::socket::Rt_RoundUp(su.sa.sa_len);    \
        memcpy(bp, &su, l);                         \
        bp += l;                                    \
    }
#define ADDADDR(addr) {                             \
        memset(&su, 0, sizeof(su));                 \
        su.sin.sin_family = nn::socket::Family::Af_Inet;                \
        su.sin.sin_len = sizeof(su.sin);            \
        (&su.sin)->sin_addr.S_addr = *addr.S_addr;  \
        ADDSU;                                      \
    }

    memset(&rtm, 0, sizeof(rtm));
    rtm.hdr.rtm_version = nn::socket::Rtm_Version;
    rtm.hdr.rtm_seq = 1;
    rtm.hdr.rtm_addrs = nn::socket::RtaFlag::Rta_Dst;

    switch (op)
    {
    case Operation_Delete:
        rtm.hdr.rtm_type = nn::socket::RtMsgType::Rtm_Delete;
        break;
    case Operation_Add:
        rtm.hdr.rtm_type = nn::socket::RtMsgType::Rtm_Add;
        rtm.hdr.rtm_addrs |= nn::socket::RtaFlag::Rta_Gateway;
        break;
    case Operation_Change:
        rtm.hdr.rtm_type = nn::socket::RtMsgType::Rtm_Change;
        break;
    default:
        return ResultInvalidInternalState();
        break;
    }

    rtm.hdr.rtm_flags = nn::socket::RtFlag::Rtf_Up;
    if (rtm.hdr.rtm_type != nn::socket::RtMsgType::Rtm_Add)
    {
        rtm.hdr.rtm_flags |= nn::socket::RtFlag::Rtf_Pinned;
    }
#ifdef SIOCGIFPRIORITY
    rtm.hdr.rtm_priority = pRt->metric;
#endif

    /* None interface subnet routes are static. */
    if ((pRt->gate.S_addr != nn::socket::InetHtonl(nn::socket::InAddr_Any)) || (pRt->net.S_addr != ifNet.S_addr) ||
        (pRt->dest.S_addr != (ifAddr.S_addr & ifNet.S_addr)))
    {
        rtm.hdr.rtm_flags |= nn::socket::RtFlag::Rtf_Static;
    }

    if ((pRt->dest.S_addr == pRt->gate.S_addr) && (pRt->net.S_addr == nn::socket::InetHtonl(nn::socket::InAddr_Broadcast)))
    {
        rtm.hdr.rtm_flags |= nn::socket::RtFlag::Rtf_Host;
    }
    else if ((pRt->gate.S_addr == nn::socket::InetHtonl(nn::socket::InAddr_Loopback)) &&
             (pRt->net.S_addr == nn::socket::InetHtonl(nn::socket::InAddr_Broadcast)))
    {
        rtm.hdr.rtm_flags |= nn::socket::RtFlag::Rtf_Host | nn::socket::RtFlag::Rtf_Gateway;
    }
    else
    {
        rtm.hdr.rtm_addrs |= nn::socket::RtaFlag::Rta_NetMask;
        if ((rtm.hdr.rtm_flags & nn::socket::RtFlag::Rtf_Static) != nn::socket::RtFlag::Rtf_None)
        {
            rtm.hdr.rtm_flags |= nn::socket::RtFlag::Rtf_Gateway;
        }
        if (op != Operation_Delete)
        {
            rtm.hdr.rtm_addrs |= nn::socket::RtaFlag::Rta_Ifa;
        }
    }

    ADDADDR(&pRt->dest);
    if ((rtm.hdr.rtm_addrs & nn::socket::RtaFlag::Rta_Gateway) != nn::socket::RtaFlag::Rta_None)
    {
        if ((((rtm.hdr.rtm_flags & nn::socket::RtFlag::Rtf_Host) != nn::socket::RtFlag::Rtf_None) && (pRt->gate.S_addr != nn::socket::InetHtonl(nn::socket::InAddr_Loopback))) ||
            !(rtm.hdr.rtm_flags & nn::socket::RtFlag::Rtf_Static))
        {
            memset(&su.sdl, 0, sizeof(su.sdl));
            su.sdl.sdl_family = nn::socket::Family::Af_Link;
            su.sdl.sdl_len = sizeof(su.sdl);
            su.sdl.sdl_index = sdlIndex;
            su.sdl.sdl_nlen = su.sdl.sdl_alen = su.sdl.sdl_slen = 0;
            ADDSU;
        }
        else
        {
            ADDADDR(&pRt->gate);
        }
    }

    if ((rtm.hdr.rtm_addrs & nn::socket::RtaFlag::Rta_NetMask) != nn::socket::RtaFlag::Rta_None)
    {
        ADDADDR(&pRt->net);
    }

    if ((rtm.hdr.rtm_addrs & nn::socket::RtaFlag::Rta_Ifp) != nn::socket::RtaFlag::Rta_None)
    {
        memset(&su.sdl, 0, sizeof(su.sdl));
        su.sdl.sdl_family = nn::socket::Family::Af_Link;
        su.sdl.sdl_len = sizeof(su.sdl);
        su.sdl.sdl_index = sdlIndex;
        su.sdl.sdl_nlen = su.sdl.sdl_alen = su.sdl.sdl_slen = 0;
        ADDSU;
    }

    if ((rtm.hdr.rtm_addrs & nn::socket::RtaFlag::Rta_Ifa) != nn::socket::RtaFlag::Rta_None)
    {
        ADDADDR(&ifAddr);
    }

#undef ADDADDR
#undef ADDSU

    rtm.hdr.rtm_msglen = (unsigned short)(bp - (char *)&rtm);
    if (nn::socket::Write(s, &rtm, rtm.hdr.rtm_msglen) == -1)
    {
        nn::socket::Errno lastErrno = nn::socket::GetLastError();
        CFG_LOG_WARN("Route Write() failed, errno=%d.\n", lastErrno);
        switch (lastErrno)
        {
        case nn::socket::Errno::ENxIo:
            result = ResultIfRemoved();
            break;
        case nn::socket::Errno::ENoBufs:
            result = ResultMemAllocStackError();
            break;
        case nn::socket::Errno::EInval:
        case nn::socket::Errno::EExist:
            result = ResultRouteError();
            break;
        default:
            result = ResultUnknownStackError();
            break;
        }
    }

    return result;
} //NOLINT(impl/function_size)

void Ipv4::GetAddrs(int type, char *cp, nn::socket::SockAddr **sa)
{
    int i;
    for (i = 0; i < static_cast<int>(nn::socket::Rtax::Rtax_Max); i++)
    {
        if (type & (1 << i))
        {
            sa[i] = (nn::socket::SockAddr *)cp;
            nn::socket::Rt_Advance(&cp, sa[i]);
        }
        else
        {
            sa[i] = NULL;
        }
    }
}


} // namespace cfg
} // namespace bsdsocket
} // namespace nn




