﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Interface
 *
 * @details
 */

#pragma once

namespace nn { namespace bsdsocket { namespace cfg {

class Server;

class Interface
{
public:
    enum State
    {
        State_Null = 0,
        State_InitInProgress,
        State_CancelDhcp,
        State_Initialized,
        State_Error,
    };

    Interface()
        : m_IfName(nullptr)
        , m_IfIndex(-1)
        , m_FailureCause(ResultIfInvalid())
        , m_AccessMutex(true)
        , m_State(State_Null)
        , m_IsDhcpInit(false)
        , m_OperationEvent(nn::os::EventClearMode_AutoClear, true)
        , m_NewLeaseCount(0)
        , m_RenewedLeaseCount(0)
        , m_ReboundLeaseCount(0)
        , m_InvalidLeaseCount(0)
        , m_SdlIndex(0)
    {
        memset(&m_IfSettings, 0, sizeof(m_IfSettings));
        memset(&m_HardwareAddress, 0, sizeof(m_HardwareAddress));
        memset(&m_Addr, 0, sizeof(m_Addr));
        memset(&m_SubnetMask, 0, sizeof(m_SubnetMask));
        memset(&m_BroadcastAddr, 0, sizeof(m_BroadcastAddr));
        memset(&m_GatewayAddr, 0, sizeof(m_GatewayAddr));
        memset(m_DnsAddrs, 0, sizeof(m_DnsAddrs));
        memset(&m_ServerAddr, 0, sizeof(m_ServerAddr));
        memset(&m_CurrentDhcpLease, 0, sizeof(m_CurrentDhcpLease));
    }
    ~Interface()
    {
    }

    Result Initialize(int32_t ifIndex, const char *ifName,
                      const IfSettings *pOptions, nn::sf::Out<nn::sf::NativeHandle> *pSfHandle);
    Result Cancel();
    Result Finalize(uint32_t options = 0);
    Result GetState(IfState *pIfState);
    Result DhcpRenew();

    const char* GetName();
    nn::socket::InAddr GetAddr();
    nn::socket::InAddr GetNetMask();
    nn::socket::InAddr GetBcastAddr();
    nn::socket::InAddr GetGatewayAddr();
    nn::socket::InAddr GetServerAddr();
private:
    Result InitLink(int mibsock);
    Result InitMtu(int mibsock);
    Result SetDhcpLease(dhcpc::Lease *pLease);
    Result InvalidateDhcpLease();
    Result SetStaticIp();
    Result InvalidateStaticIp();
    Result ConfigureRoute(bool add);
    Result AddAddress();
    Result DeleteAddress();
    Result CheckForDuplicateIps(ShutdownOption & shutdownOption);
    void CurateDnsAddresses(size_t * validDnsAddrs);
    void DhcpCallback(dhcpc::IndicationDataType *pIndication);
    static void DhcpCallbackStatic(void *context, dhcpc::IndicationDataType *pIndication)
    {
        Interface *pI = reinterpret_cast<Interface *>(context);
        pI->DhcpCallback(pIndication);
    }

    Result StartDhcp(nn::socket::InAddr requestedAddr,nn::socket::InAddr serverAddr);
    Result StopDhcp(uint32_t options);
    Result ResolveHardwareAddress();
    Result SetLinkState(bool isLinkStateUp);
    Result CheckLinkState(bool *pReturnedLinkState);

    const char           *m_IfName;
    int32_t              m_IfIndex;
    IfSettings           m_IfSettings;
    uint8_t              m_HardwareAddress[dhcpc::Config_MaxHardwareAddressSize];
    nn::socket::InAddr   m_Addr;
    nn::socket::InAddr   m_SubnetMask;
    nn::socket::InAddr   m_BroadcastAddr;
    nn::socket::InAddr   m_GatewayAddr;
    nn::socket::InAddr   m_DnsAddrs[2];
    nn::socket::InAddr   m_ServerAddr;
    dhcpc::Lease         m_CurrentDhcpLease;
    Result               m_FailureCause;
    nn::os::Mutex        m_AccessMutex;
    State                m_State;
    bool                 m_IsDhcpInit;
    nn::os::EventType    m_InitializeEvent;
    nn::os::SystemEvent  m_OperationEvent;
    uint32_t             m_NewLeaseCount;
    uint32_t             m_RenewedLeaseCount;
    uint32_t             m_ReboundLeaseCount;
    uint32_t             m_InvalidLeaseCount;
    uint16_t             m_SdlIndex;
};



} // namespace cfg
} // namespace bsdsocket
} // namespace nn

