﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cfg_PrivateIncludes.h"

#include <nn/sf/sf_Types.h> // for nn::sf::SharedPointer
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h> // for nn::sf::ExpHeapStaticAllocator


namespace nn { namespace bsdsocket { namespace cfg {

struct CreateCfgByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, CreateCfgByHipcTag> MyAllocator;

class MyAllocatorInitializer
{
public:
    MyAllocatorInitializer() NN_NOEXCEPT
    {
        MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_CfgShimAllocatorInitializer;

static nn::sf::SharedPointer<ServerInterface> CreateSessionViaHipc() NN_NOEXCEPT
{
    nn::sf::SharedPointer<ServerInterface> ret;
    auto result = nn::sf::CreateHipcProxyByName<ServerInterface, MyAllocator::Policy>(&ret, "bsdcfg");
    NN_ABORT_UNLESS(result.IsSuccess());
    return ret;
}

Result SetIfUp(const char *ifName, IfSettings *pIfOptionData)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    nn::sf::InBuffer sfIfName(reinterpret_cast<const char *>(ifName),
                              strnlen(ifName, Limits_MaxIfNameSize - 1) + 1);
    result = pS->SetIfUp(sfIfName, *pIfOptionData);
    pS = NULL;
    return result;
}

Result SetIfUp(const char *ifName, IfSettings *pIfOptionData,
               nn::os::SystemEvent *pEvent, nn::os::EventClearMode eventClearMode)
{
    Result result;
    nn::sf::NativeHandle sfHandle;
    NN_ABORT_UNLESS_NOT_NULL(pEvent);
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    nn::sf::InBuffer sfIfName(reinterpret_cast<const char *>(ifName),
                              strnlen(ifName, Limits_MaxIfNameSize - 1) + 1);
    result = pS->SetIfUpWithEvent(sfIfName, *pIfOptionData, &sfHandle);
    if(result.IsSuccess())
    {
        pEvent->AttachReadableHandle(sfHandle.GetOsHandle(), sfHandle.IsManaged(), eventClearMode);
        sfHandle.Detach();
    }
    pS = NULL;
    return result;
}

Result CancelIf(const char *ifName)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    nn::sf::InBuffer sfIfName(reinterpret_cast<const char *>(ifName),
                              strnlen(ifName, Limits_MaxIfNameSize - 1) + 1);
    result = pS->CancelIf(sfIfName);
    pS = NULL;
    return result;
}

Result SetIfDown(const char *ifName)
{
    return SetIfDown(ifName,ShutdownOption_None);
}

Result SetIfDown(const char *ifName, uint32_t options)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    nn::sf::InBuffer sfIfName(reinterpret_cast<const char *>(ifName),
                              strnlen(ifName, Limits_MaxIfNameSize - 1) + 1);
    result = pS->SetIfDown(sfIfName,options);
    pS = NULL;
    return result;
}

Result GetIfState(const char *ifName, IfState *pIfState)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    nn::sf::InBuffer sfIfName(reinterpret_cast<const char *>(ifName),
                              strnlen(ifName, Limits_MaxIfNameSize - 1) + 1);
    nn::sf::OutBuffer sfIfState(reinterpret_cast<char *>(pIfState),
                                sizeof(IfState));
    result = pS->GetIfState(sfIfName, sfIfState);
    pS = NULL;
    return result;
}

Result DhcpRenew(const char *ifName)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    nn::sf::InBuffer sfIfName(reinterpret_cast<const char *>(ifName),
                              strnlen(ifName, Limits_MaxIfNameSize - 1) + 1);
    result = pS->DhcpRenew(sfIfName);
    pS = NULL;
    return result;
}

//Arp API

Result AddStaticArpEntry(nn::socket::InAddr ipAddress, const uint8_t* pInputHardwareAddress, size_t hardwareAddressSize)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_ABORT_UNLESS_EQUAL(hardwareAddressSize, nn::socket::Ether_Addr_Len);
    nn::sf::InBuffer sfInputHardwareAddress(reinterpret_cast<const char *>(pInputHardwareAddress), hardwareAddressSize);
    result = pS->AddStaticArpEntry(ipAddress.S_addr, sfInputHardwareAddress);
    pS = NULL;
    return result;
}

Result AddStaticArpEntry(in_addr ipAddress, const uint8_t* pInputHardwareAddress, size_t hardwareAddressSize)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_ABORT_UNLESS_EQUAL(hardwareAddressSize, nn::socket::Ether_Addr_Len);
    nn::sf::InBuffer sfInputHardwareAddress(reinterpret_cast<const char *>(pInputHardwareAddress), hardwareAddressSize);
    result = pS->AddStaticArpEntry(ipAddress.s_addr, sfInputHardwareAddress);
    pS = NULL;
    return result;
}

Result RemoveArpEntry(nn::socket::InAddr ipAddress)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    result = pS->RemoveArpEntry(ipAddress.S_addr);
    pS = NULL;
    return result;
}

Result RemoveArpEntry(in_addr ipAddress)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    result = pS->RemoveArpEntry(ipAddress.s_addr);
    pS = NULL;
    return result;
}

Result LookupArpEntry(uint8_t* pOutputHardwareAddress, size_t hardwareAddressSize, nn::socket::InAddr ipAddress)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_ABORT_UNLESS_EQUAL(hardwareAddressSize, nn::socket::Ether_Addr_Len);
    nn::sf::OutBuffer sfOutputHardwareAddress(reinterpret_cast<char *>(pOutputHardwareAddress), hardwareAddressSize);
    result = pS->LookupArpEntry(sfOutputHardwareAddress, ipAddress.S_addr);
    pS = NULL;
    return result;
}

Result LookupArpEntry(uint8_t* pOutputHardwareAddress, size_t hardwareAddressSize, in_addr ipAddress)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_ABORT_UNLESS_EQUAL(hardwareAddressSize, nn::socket::Ether_Addr_Len);
    nn::sf::OutBuffer sfOutputHardwareAddress(reinterpret_cast<char *>(pOutputHardwareAddress), hardwareAddressSize);
    result = pS->LookupArpEntry(sfOutputHardwareAddress, ipAddress.s_addr);
    pS = NULL;
    return result;
}

Result LookupArpEntry(nn::socket::InAddr* pOutputIpAddress, size_t ipAddressSize, const uint8_t* pHardwareAddress, size_t hardwareAddressSize)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_ABORT_UNLESS_EQUAL(hardwareAddressSize, nn::socket::Ether_Addr_Len);
    NN_ABORT_UNLESS_EQUAL(ipAddressSize, sizeof(nn::socket::InAddr));

    nn::sf::OutBuffer sfOutputIpAddress(reinterpret_cast<char *>(pOutputIpAddress), ipAddressSize);
    nn::sf::InBuffer sfHardwareAddress(reinterpret_cast<const char *>(pHardwareAddress), hardwareAddressSize);
    result = pS->LookupArpEntry2(sfOutputIpAddress, sfHardwareAddress);
    pS = NULL;
    return result;
}

Result LookupArpEntry(in_addr* pOutputIpAddress, size_t ipAddressSize, const uint8_t* pHardwareAddress, size_t hardwareAddressSize)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    NN_ABORT_UNLESS_EQUAL(hardwareAddressSize, nn::socket::Ether_Addr_Len);
    NN_ABORT_UNLESS_EQUAL(ipAddressSize, sizeof(nn::socket::InAddr));

    nn::sf::OutBuffer sfOutputIpAddress(reinterpret_cast<char *>(pOutputIpAddress), ipAddressSize);
    nn::sf::InBuffer sfHardwareAddress(reinterpret_cast<const char *>(pHardwareAddress), hardwareAddressSize);
    result = pS->LookupArpEntry2(sfOutputIpAddress, sfHardwareAddress);
    pS = NULL;
    return result;
}

Result ClearArpEntries()
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    result = pS->ClearArpEntries();
    pS = NULL;
    return result;
}

Result ClearArpEntries(const char * pIfName)
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    nn::sf::InBuffer sfIfName(reinterpret_cast<const char *>(pIfName),
        strnlen(pIfName, Limits_MaxIfNameSize - 1) + 1);
    result = pS->ClearArpEntries2(sfIfName);
    pS = NULL;
    return result;
}

Result PrintArpEntries()
{
    Result result;
    nn::sf::SharedPointer<ServerInterface> pS = CreateSessionViaHipc();
    NN_ABORT_UNLESS_NOT_NULL(pS);
    result = pS->PrintArpEntries();
    pS = NULL;
    return result;
}

} // namespace cfg
} // namespace bsdsocket
} // namespace nn
