﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_KHandleTable.h"
#include "kern_KAutoObject.h"
#include "kern_KThread.h"
#include "kern_KProcess.h"
#include "kern_Current.h"

namespace nn {
    namespace kern {
    template <typename T>
        inline T RoundDown(T value, size_t align)
        {
            return value & ~(align - 1);
        }

    template <typename T>
        inline T RoundUp(T value, size_t align)
        {
            return (value + (align - 1)) & ~(align - 1);
        }

        enum SlabType
        {
            SLAB_TYPE_EVENT,
            SLAB_TYPE_INTERRUPT_EVENT,
            SLAB_TYPE_PROCESS,
            SLAB_TYPE_THREAD,
            SLAB_TYPE_PORT,
            SLAB_TYPE_DMA_OBJECT,
            SLAB_TYPE_SHARED_MEMORY,
            SLAB_TYPE_TRANSFER_MEMORY,
            SLAB_TYPE_CODE_MEMORY,
            SLAB_TYPE_DEVICE_ADDRESS_SPACE,
            SLAB_TYPE_DEBUG,

            SLAB_TYPE_SESSION_REQUEST,
            SLAB_TYPE_SESSION,
            SLAB_TYPE_LIGHT_SESSION,
            SLAB_TYPE_LINKED_LIST_NODE,

            SLAB_TYPE_BLOCK_INFO,
            SLAB_TYPE_THREAD_LOCAL_PAGE,
            SLAB_TYPE_OBJECT_NAME,

            SLAB_TYPE_RESOURCE_LIMIT,
            SLAB_TYPE_SHARED_MEMORY_INFO,
            SLAB_TYPE_PAGE_BUFFER,
            SLAB_TYPE_INTERRUPT_EVENT_TASK,

            SLAB_TYPE_DEBUG_EVENT_OBJECT,
            NUM_SLAB_TYPE
        };
        struct SlabInfo
        {
            uintptr_t   addr;
            size_t      numUnit;
            size_t      unitSize;
            uintptr_t   headPointerAddr;
        };

        struct AnalyzeTable
        {
            // 0x00
            size_t      sizeParamsOnStack;
            size_t      sizeAutoObject;
            uintptr_t   addrKernelRegionP;
            size_t      sizeKernelRegion;

            // 0x10
            uintptr_t   addrPageManager;
            uintptr_t   addrHardwareTimer;
            uintptr_t   addrEventLog;
            size_t      sizeEventLog;

            // 0x20
            size_t      portOffset;
            uint8_t     padding0x20[0x0c];

            // 0x30
            uint8_t     padding0x30[0x10];

            // 0x40
            SlabInfo    slabInfo[NUM_SLAB_TYPE];
        };

        /*!
            @brief     ハンドルからオブジェクトを取得します。

            擬似ハンドルが与えられた場合、解釈して返します。

            @param[in]    handleTable     ハンドルテーブル
            @param[in]    handle          ハンドル

        */
        inline KAutoObject* GetObjectFromRealOrPseudoHandle(const KHandleTable& handleTable, nn::svc::Handle handle)
        {
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS)
            {
                // カレントプロセスを示す擬似ハンドルだった場合、カレントプロセスを返します
                KProcess& p = GetCurrentProcess();
                p.Open();
                return &p;
            }
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_THREAD)
            {
                // カレントスレッドを示す擬似ハンドルだった場合、カレントプロセスを返します
                KThread& p = GetCurrentThread();
                p.Open();
                return &p;
            }

            // テーブル引きでオブジェクトを返します
            return handleTable.GetObject(handle);
        }

        inline KAutoObject* GetObjectFromRealOrPseudoHandleForIpc(const KHandleTable& handleTable, nn::svc::Handle handle)
        {
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS)
            {
                // カレントプロセスを示す擬似ハンドルだった場合、カレントプロセスを返します
                KProcess& p = GetCurrentProcess();
                p.Open();
                return &p;
            }
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_THREAD)
            {
                // カレントスレッドを示す擬似ハンドルだった場合、カレントプロセスを返します
                KThread& p = GetCurrentThread();
                p.Open();
                return &p;
            }

            // テーブル引きでオブジェクトを返します
            return handleTable.GetObjectForIpc(handle);
        }

        /*!
            @brief     ハンドルからオブジェクトを取得します。

            擬似ハンドルが与えられた場合、解釈して返します。

            @param[in]    handleTable     ハンドルテーブル
            @param[in]    handle          ハンドル
            @param[in]    pCurrent        カレントスレッド

        */
        inline KAutoObject* GetObjectFromRealOrPseudoHandle(const KHandleTable& handleTable, nn::svc::Handle handle, KThread* pCurrent)
        {
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS)
            {
                // カレントプロセスを示す擬似ハンドルだった場合、カレントプロセスを返します
                KProcess& p = pCurrent->GetParent();
                p.Open();
                return &p;
            }
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_THREAD)
            {
                // カレントスレッドを示す擬似ハンドルだった場合、カレントプロセスを返します
                KThread* p = pCurrent;
                p->Open();
                return p;
            }

            // テーブル引きでオブジェクトを返します
            return handleTable.GetObject(handle);
        }

        inline KAutoObject* GetObjectFromRealOrPseudoHandleForIpc(const KHandleTable& handleTable, nn::svc::Handle handle, KThread* pCurrent)
        {
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS)
            {
                // カレントプロセスを示す擬似ハンドルだった場合、カレントプロセスを返します
                KProcess& p = pCurrent->GetParent();
                p.Open();
                return &p;
            }
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_THREAD)
            {
                // カレントスレッドを示す擬似ハンドルだった場合、カレントプロセスを返します
                KThread* p = pCurrent;
                p->Open();
                return p;
            }

            // テーブル引きでオブジェクトを返します
            return handleTable.GetObjectForIpc(handle);
        }


        /*!
            @brief     ハンドルからオブジェクトを取得します。

            @param[in]    handleTable     ハンドルテーブル
            @param[in]    handle          ハンドル

        */
        template <class T>
        inline T* GetObjectFromRealOrPseudoHandle(const KHandleTable& handleTable, nn::svc::Handle handle)
        {
            // テンプレート特殊化でオブジェクトを取得する処理の対となります。
            // KProcess 型 / KThread 型オブジェクトの問い合わせはありえないため、擬似ハンドル判定を省略できます。
            KAutoObject* pObject = handleTable.GetObject(handle);
            return (pObject == NULL) ? NULL: pObject->DynamicCast<T*>();
        }

        /*!
            @brief     ハンドルからKProcessオブジェクトを取得します。

            擬似ハンドルが与えられた場合、解釈して返します。

            @param[in]    handleTable     ハンドルテーブル
            @param[in]    handle          ハンドル

        */
        template <>
        inline KProcess* GetObjectFromRealOrPseudoHandle<KProcess>(const KHandleTable& handleTable, nn::svc::Handle handle)
        {
            // テンプレートの特殊化でプロセス型のオブジェクトを取得します。
            // カレントスレッドを示す擬似ハンドルの判定を省略できます。
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS)
            {
                KProcess& p = GetCurrentProcess();
                p.Open();
                return &p;
            }
            else
            {
                // ハンドルがKProcess型であった場合のみ成功します
                return handleTable.GetObject<KProcess>(handle);
            }
        }

        /*!
            @brief     ハンドルから KThread オブジェクトを取得します。

            擬似ハンドルが与えられた場合、解釈して返します。

            @param[in]    handleTable     ハンドルテーブル
            @param[in]    handle          ハンドル

        */
        template <>
        inline KThread* GetObjectFromRealOrPseudoHandle<KThread>(const KHandleTable& handleTable, nn::svc::Handle handle)
        {
            // テンプレートの特殊化でスレッド型のオブジェクトを取得します。
            // カレントプロセスを示す擬似ハンドルの判定を省略できます。
            if(handle == nn::svc::PSEUDO_HANDLE_CURRENT_THREAD)
            {
                KThread& p = GetCurrentThread();
                p.Open();
                return &p;
            }
            else
            {
                // ハンドルが KThread 型であった場合のみ成功します
                return handleTable.GetObject<KThread>(handle);
            }
        }


        /*!
            @brief     任意優先度の ユーザー属性スレッドを生成します。
            @param[in]    pThread     KThread::Create()により得られるスレッドオブジェクト
            @param[in]    f           スレッドのエントリーポイント
            @param[in]    param       任意のパラメーター
            @param[in]    userStackBottom     スタックの初期位置
            @param[in]    prio        スレッドの優先度
            @param[in]    coreNo      割り当てるCPUコア
            @param[in]    pParent     所属するプロセス

        */
        Result InitializeThreadForUser(
            KThread*            pThread,
            ThreadFunc          f,
            uintptr_t           param,
            KProcessAddress     userStackBottom,
            int32_t             prio,
            int32_t             coreNo,
            KProcess*           pParent );

        /*!
            @brief     カーネルスレッドのスタックを開放します。
            @param[in]    svStackBottom スタックの初期アドレス

        */
        void CleanupSuperVisorStack(void* svStackBottom);

        /*!
            @brief     きわめて優先度の高い リアルタイム属性のスレッド(優先度:0)を生成します。

            現在はハードウェア割り込みや、スケジューラー用ソフトウェア割り込みの
            後処理スレッド( DPC タスク )にのみ使用されています。

            @param[in]    pThread   KThread::Create()により得られるスレッドオブジェクト
            @param[in]    f         スレッドのエントリーポイント
            @param[in]    param     スレッドに渡す任意のパラメーター

        */
        Result InitializeThreadForRealTime(
            KThread*            pThread,
            ThreadFunc          f,
            uintptr_t           param);

        Result InitializeThreadForKernel(
            KThread*            pThread,
            ThreadFunc          f,
            uintptr_t           param,
            int32_t             prio,
            int32_t             coreNo);

        /*!
            @brief     LCDやデバッグLEDにインジケートします。
            @param[in]    no     インジケートする内容

        */
        inline void IndicateDebugPoint(uint8_t no) { NN_UNUSED(no); }
    }
}

