﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/diag/diag_LogTypes.h>
#include <nn/diag/detail/diag_DetailAssert.h>
#include <nn/diag/diag_AssertionFailureHandler.h>
#include <nn/svc/svc_Kernel.h>
#include <cstring>
#include <cstdarg>
#include "kern_Platform.h"
#include "kern_Panic.h"
#include "kern_DebugString.h"
#include "kern_Result.h"
#include "kern_KProcess.h"
#include "kern_KThread.h"
#include "kern_DebugSelect.h"
#include "kern_KTrace.h"
#include "kern_SystemControl.h"

#if defined NN_KERN_FOR_DEVELOPMENT && \
    defined NN_BUILD_CONFIG_COMPILER_GCC && \
    defined NN_BUILD_CONFIG_CPU_ARM
#include <unwind.h>
#endif

namespace nn { namespace dbg {

    Result Break(nn::svc::BreakReason reason)
    {
        NN_UNUSED(reason);
        NN_KERNEL_PANIC("nn::dbg::Break(%d) called", reason);
        return nn::svc::ResultNotImplemented();
    }

}}

namespace nn { namespace result { namespace detail {

    NN_NORETURN void OnUnhandledResult(Result result) NN_NOEXCEPT
    {
        NN_UNUSED(result);
        NN_KERNEL_PANIC("UnhandledResult");
        while (1) {}
    }

}}}


namespace nn { namespace kern {
    namespace
    {
#ifdef NN_KERN_FOR_DEVELOPMENT
        int s_PanicCount = 0;

#if defined NN_BUILD_CONFIG_COMPILER_GCC && defined NN_BUILD_CONFIG_CPU_ARM
        _Unwind_Reason_Code TraceFunction(struct _Unwind_Context* pContext, void* pArg)
        {
            void *ip = reinterpret_cast<void *>(_Unwind_GetIP(pContext));
            NN_KERN_RELEASE_LOG("   %p\n", ip);
            if (ip == nullptr)
            {
                return _URC_END_OF_STACK;
            }
            else
            {
                return _URC_NO_REASON;
            }
        }
#endif

        /*!
            @brief     現在のプロセスやスレッドに関する情報を表示します。

        */
        void PrintCurrentInfo();
        void PrintCurrentInfo()
        {
            // 多重にPanicしないようにガードします
            //
            // CHECK: 以下のスコープへ入る前処理として、割り込み禁止+SpinLockの取得を推奨します
            if( s_PanicCount <= 1 )
            {
                s_PanicCount++;

                KThread& currentThread       = GetCurrentThread();
                KProcess* pCurrentProcess    = GetCurrentProcessPointer();

                NN_KERN_RELEASE_LOG("cur thread               = %p (id = %lld)\n", &currentThread, currentThread.GetId());
                if( pCurrentProcess != NULL )
                {
                    char name[16];
                    std::memcpy(name, pCurrentProcess->GetName(), sizeof(name));
                    name[sizeof(name) - 1] = '\0';

                    NN_KERN_RELEASE_LOG("cur process              = %p (id = %lld, name = \"%s\")\n", pCurrentProcess, pCurrentProcess->GetId(), name);
                }
                else
                {
                    NN_KERN_RELEASE_LOG("cur process              = NULL (id = 0)\n");
                }
                if( currentThread.GetParentPointer() != NULL )
                {
                    NN_KERN_RELEASE_LOG("process of cur thread    = %p (id = %lld)\n", currentThread.GetParentPointer(), currentThread.GetParentPointer()->GetId());
                }
                else
                {
                    NN_KERN_RELEASE_LOG("process of cur thread    = kernel\n");
                }
                KDebug::PrintRegister();
                KDebug::PrintBacktrace();

#if defined NN_BUILD_CONFIG_COMPILER_GCC && defined NN_BUILD_CONFIG_CPU_ARM
                NN_KERN_RELEASE_LOG("Backtrace\n");
                _Unwind_Backtrace(&TraceFunction, nullptr);
#elif defined NN_BUILD_CONFIG_CPU_ARM64
                NN_KERN_RELEASE_LOG("Backtrace\n");
                uintptr_t frameAddress = reinterpret_cast<uintptr_t>(__builtin_frame_address(0));
                for (int i = 0; i < 32; i++)
                {
                    if (frameAddress & 0xF)
                    {
                        break;
                    }
                    KPhysicalAddress phys;
                    if (!KCPU::GetPhysicalAddressOfCurrentSpace(&phys, frameAddress, true))
                    {
                        break;
                    }

                    struct frame
                    {
                        uintptr_t frame;
                        uintptr_t lr;
                    };
                    frame* pFrame = reinterpret_cast<frame*>(frameAddress);
                    NN_KERN_RELEASE_LOG("   %p\n", pFrame->lr);
                    frameAddress = pFrame->frame;
                }
#endif
            }
        }
#endif  // ifdef NN_KERN_FOR_DEVELOPMENT

        /*!
            @brief     CPUを停止状態にします。

        */
        NN_NORETURN void Stop()
        {
#ifdef NN_KERN_FOR_DEVELOPMENT
            PrintCurrentInfo();
#endif  // ifdef NN_KERN_FOR_DEVELOPMENT

            NN_KERN_KTRACE_STOP();
            NN_KERN_KTRACE_DUMP();

            // プログラムの実行を停止します。
            //KernelControl(nn::svc::SYSTEM_OPERATION_STOP, 0, 0);
            KSystemControl::StopSystem();
        }
    }

    /*!
        @brief     プログラムの実行を停止します。
        @param[in]    filename     ファイル名
        @param[in]    lineno       行番号
        @param[in]    fmt          出力書式文字列(フォーマット文字列)`

    */
    void Panic(const char* filename, int lineno, const char* fmt, ...)
    {
        NN_KERN_KTRACE_STOP();
#ifdef NN_KERN_FOR_DEVELOPMENT
        // 多重にPanicしないようにガードします
        //
        // CHECK: 以下のスコープへ入る前処理として、割り込み禁止+SpinLockの取得を推奨します
        if( s_PanicCount <= 0 )
        {
            s_PanicCount++;

            ::std::va_list vlist;
            va_start(vlist, fmt);
            NN_KERN_RELEASE_LOG("KernelPanic[core%d]: %s:%d\n", GetCurrentCpuNo(), filename, lineno);
            NN_KERN_RELEASE_VLOG(fmt, vlist);
            NN_KERN_RELEASE_LOG("\n");
            va_end(vlist);

        }
#else   // ifdef NN_KERN_FOR_DEVELOPMENT
        NN_UNUSED(filename);
        NN_UNUSED(lineno);
        NN_UNUSED(fmt);
#endif  // ifdef NN_KERN_FOR_DEVELOPMENT else

        Stop();
    }

    void PanicQuietly()
    {
        Stop();
    }
}}

namespace nn { namespace diag {
namespace {
const char* GetAssertionFailureMessageByType(
        AssertionType assertionType) NN_NOEXCEPT
{
    switch( assertionType )
    {
    case AssertionType_SdkAssert:
        return "SDK Assertion Failure";
    case AssertionType_SdkRequires:
        return "Precondition not met";
    case AssertionType_UserAssert:
        return "Assertion Failure";
    default:
        // 不明な AssertionType が渡された場合であっても、AssertionType が正しいことの Assert で停止するのではなく、
        // Assertion 失敗の情報を表示するべきです。
        return "Assertion Failure (Unknown Type)";
    }
}
void DefaultPrinter(
        const char* failureMessage,
        const char* condition,
        const char* functionName,
        const char* fileName,
        int lineNumber,
        const LogMessage& message) NN_NOEXCEPT
{
    // TORIAEZU: LogMessage オブジェクトのフォーマットがまだできないため、NN_SDK_LOG を使用せず直接デバッグ出力を行う
    NN_UNUSED(failureMessage);
    NN_UNUSED(condition);
    NN_UNUSED(functionName);
    NN_UNUSED(fileName);
    NN_UNUSED(lineNumber);
    NN_UNUSED(message);
    NN_KERN_RELEASE_LOG(
            "%s: \'%s\' at '%s':%d in %s\n",
            failureMessage, condition, fileName, lineNumber, functionName);
    NN_KERN_RELEASE_VLOG(message.format, *message.args);
    NN_KERN_RELEASE_LOG("\n");
}

AssertionFailureOperation DefaultAssertionFailureHandler(
        const AssertionInfo &assertionInfo)
{
    DefaultPrinter(
            GetAssertionFailureMessageByType(assertionInfo.assertionType),
            assertionInfo.condition,
            assertionInfo.functionName,
            assertionInfo.fileName,
            assertionInfo.lineNumber,
            *assertionInfo.message);

    return AssertionFailureOperation_Abort;
}

}

void detail::OnAssertionFailure(
    AssertionType assertionType,
    const char* condition,
    const char* functionName,
    const char* fileName,
    int lineNumber,
    const char* format,
    ...)
{
    ::std::va_list list;
    va_start(list, format);

    const nn::diag::LogMessage message = {format, &list};
    const AssertionInfo assertionInfo = {assertionType, &message, condition, functionName, fileName, lineNumber};
    DefaultAssertionFailureHandler(assertionInfo);

    va_end(list);
    nn::kern::Stop();
}

void detail::OnAssertionFailure(
    AssertionType assertionType,
    const char* condition,
    const char* functionName,
    const char* fileName,
    int lineNumber)
{
    NN_KERN_KTRACE_STOP();
    detail::OnAssertionFailure(assertionType, condition, functionName, fileName, lineNumber, "");
}
}}
