﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "kern_InterlockedSelect.h"
#include "kern_CPUSelect.h"
#include "kern_KScheduler.h"
#include "kern_KThread.h"
#include "kern_PageTableSelect.h"
#include "kern_KInterruptTaskManager.h"
#include "kern_InterruptManagerSelect.h"
#include "kern_Current.h"
#include "kern_WorkThread.h"
#include "kern_Assert.h"
#include "kern_KMemoryLayout.h"
#include <nn/nn_BitTypes.h>

namespace nn { namespace kern {

// forward宣言
class KMemoryManager;
class KPageHeap;
class KScheduler;
class KSynchronization;
class KInterruptTaskManager;
class KDmaManager;
class KPageTableManager;
class KMemoryBlockResourceManager;
class KBlockInfoManager;
class KResourceLimit;
class KUnsafeMemory;

// TORIAEZU
#if defined(NN_BUILD_CONFIG_CPU_CORTEX_A9)

namespace ARMv7A {
    class KInterruptManager;
    class KKernelPageTable;
}
namespace CortexA9 {
    class KHardwareTimer;
}

using nn::kern::ARMv7A::KInterruptManager;
using nn::kern::ARMv7A::KKernelPageTable;
using nn::kern::CortexA9::KHardwareTimer;

#elif defined(NN_BUILD_CONFIG_CPU_CORTEX_A15) || defined(NN_BUILD_CONFIG_CPU_CORTEX_A7)

namespace ARMv7A {
    class KInterruptManager;
    class KKernelPageTable;
    class KHardwareTimer;
}

using nn::kern::ARMv7A::KInterruptManager;
using nn::kern::ARMv7A::KKernelPageTable;
using nn::kern::ARMv7A::KHardwareTimer;

#elif defined(NN_BUILD_CONFIG_CPU_ARM_V8A)

namespace ARMv8A {
    class KInterruptManager;
    class KKernelPageTable;
    class KHardwareTimer;
}

using nn::kern::ARMv8A::KInterruptManager;
using nn::kern::ARMv8A::KKernelPageTable;
using nn::kern::ARMv8A::KHardwareTimer;

#else
#error No processor is selected
#endif

class Kernel
{
public:
    enum State: uint8_t
    {
        STATE_NULL,
        STATE_INITIALIZING,
        STATE_NORMAL,
        STATE_FINALIZING,
        STATE_FINALIZED
    };
#ifndef NN_BUILD_FOR_ASSEMBLY_OFFSET
private:
#endif
    struct CoreLocalRegion
    {
        CurrentSet              currentSet;
        KCPU                    core;
        KScheduler              scheduler;
        KInterruptTaskManager   interruptTaskManager;
        KInterruptManager       interruptManager;
        KHardwareTimer          hardwareTimer;

        int64_t     numSoftwareInterrupts;
        int64_t     numHardwareInterrupts;
        InterlockedVariable<int64_t> numSystemCall;
        int64_t     numProcessSwitch;
        int64_t     numThreadSwitch;
        int64_t     numFpuSwitch;
        int64_t     numSchedulerUpdate;
        int64_t     numCallSchedulerUpdate;

        InterlockedVariable<int64_t> numSvc[128];

        uint32_t    prevCounter[KCPU::NUM_PERFORMANCE_COUNTER];
    };
    static_assert( sizeof(CoreLocalRegion) <= NN_KERN_V_ADDR_CORE_LOCAL_REGION_SIZE, "" );

public:
    static void Initialize(int32_t coreNo);
    static void InitializeResourceManager(KVirtualAddress buffer, size_t bufferSize);

    // システムで 1 つ
    static State GetState() { return s_State; }
    static void SetState(State s) { s_State = s; }
    static KMemoryManager&      GetKernelHeapManager()  { return s_KernelHeapManager; }
    static KPageTableManager&   GetPageTableManager()   { return s_PageTableManager; }
    static KMemoryBlockResourceManager& GetAppMemoryBlockManager() { return s_AppMemoryBlockManager; }
    static KMemoryBlockResourceManager& GetSysMemoryBlockManager() { return s_SysMemoryBlockManager; }
    static KBlockInfoManager&   GetBlockInfoManager()   { return s_BlockInfoManager; }
    static KSynchronization&    GetSynchronization()    { return s_Synchronization; }
    static KDmaManager&         GetDmaManager()         { return s_DmaManager; }
    static KKernelPageTable&    GetKernelPageTable()    { return s_KernelPageTable; }
    static KResourceLimit&      GetSystemResourceLimit(){ return s_SystemResourceLimit; }
    static KUnsafeMemory&       GetUnsafeMemory()       { return s_UnsafeMemory; }
    static KWorkThreadManager&  GetWorkThreadManager(Bit32 id)
    {
        NN_KERN_MAX_ASSERT(id, sizeof(s_WorkThreadManager) / sizeof(*s_WorkThreadManager));
        return s_WorkThreadManager[id];
    }

    // コアごとに 1 つ
    static CurrentSet& GetCurrentSet(int32_t coreNo)
    {
        return GetCoreLocalRegion(coreNo).currentSet;
    }
    static KScheduler& GetScheduler(int32_t coreNo)
    {
        return GetCoreLocalRegion(coreNo).scheduler;
    }
    static KInterruptTaskManager& GetInterruptTaskManager()
    {
        return GetCoreLocalRegion().interruptTaskManager;
    }
    static KInterruptManager& GetInterruptManager()
    {
        return GetCoreLocalRegion().interruptManager;
    }
    static KThread& GetMainThread(int32_t coreNo)
    {
        // KThread* の一意性を保つため
        // GetCoreLocalRegion() を使用してはいけない
        return s_MainThread[coreNo];
    }
    static KThread& GetIdleThread(int32_t coreNo)
    {
        // KThread* の一意性を保つため
        // GetCoreLocalRegion() を使用してはいけない
        return s_IdleThread[coreNo];
    }
    static KHardwareTimer& GetHardwareTimer()
    {
        // GetCoreLocalRegion() を使用してはいけない
        return GetCoreLocalRegion(GetCurrentCpuNo()).hardwareTimer;
    }

    // パフォーマンスカウンタ
#ifdef NN_KERN_ENABLE_SVC_PROFILE
    static void IncrementNumSvc(int32_t svcNo)
    {
        ++GetCoreLocalRegion().numSvc[svcNo];
    }
#endif

#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
    static uint32_t UpdatePerformanceCounter(int n)
    {
        uint32_t pmc = KCPU::GetPerformanceCounter(n);
        uint32_t delta = pmc - GetCoreLocalRegion().prevCounter[n];
        GetCoreLocalRegion().prevCounter[n] = pmc;
        return delta;
    }

    static uint32_t GetPrevPerformanceCounter(int n)
    {
            return GetCoreLocalRegion().prevCounter[n];
    }
#endif

    static int64_t GetNumSoftwareInterrupts(int32_t coreNo) { return GetCoreLocalRegion(coreNo).numSoftwareInterrupts; }
    static int64_t GetNumHardwareInterrupts(int32_t coreNo) { return GetCoreLocalRegion(coreNo).numHardwareInterrupts; }
    static int64_t GetNumSystemCall(int32_t coreNo)         { return GetCoreLocalRegion(coreNo).numSystemCall; }
    static int64_t GetNumProcessSwitch(int32_t coreNo)      { return GetCoreLocalRegion(coreNo).numProcessSwitch; }
    static int64_t GetNumThreadSwitch(int32_t coreNo)       { return GetCoreLocalRegion(coreNo).numThreadSwitch; }
    static int64_t GetNumFpuSwitch(int32_t coreNo)          { return GetCoreLocalRegion(coreNo).numFpuSwitch; }
    static int64_t GetNumSchedulerUpdate(int32_t coreNo)    { return GetCoreLocalRegion(coreNo).numSchedulerUpdate; }
    static int64_t GetNumCallSchedulerUpdate(int32_t coreNo){ return GetCoreLocalRegion(coreNo).numCallSchedulerUpdate; }
    static int64_t GetNumPageClear()                    { return s_NumPageClear; }
    static int64_t GetNumIpc()                          { return s_NumIpc; }

    static int64_t GetNumSvc(int32_t svcNo, int32_t coreNo)
    {
        return GetCoreLocalRegion(coreNo).numSvc[svcNo];
    }
    static void IncrementNumSchedulerUpdate(int32_t coreNo) { ++GetCoreLocalRegion(coreNo).numSchedulerUpdate; }

    static void IncrementNumSoftwareInterrupts()    { ++GetCoreLocalRegion().numSoftwareInterrupts; }
    static void IncrementNumHardwareInterrupts()    { ++GetCoreLocalRegion().numHardwareInterrupts; }
    static void IncrementNumSystemCall()            { ++GetCoreLocalRegion().numSystemCall; }
    static void IncrementNumProcessSwitch()         { ++GetCoreLocalRegion().numProcessSwitch; }
    static void IncrementNumThreadSwitch()          { ++GetCoreLocalRegion().numThreadSwitch; }
    static void IncrementNumFpuSwitch()             { ++GetCoreLocalRegion().numFpuSwitch; }
    static void IncrementNumCallSchedulerUpdate()   { ++GetCoreLocalRegion().numCallSchedulerUpdate; }
    static void IncrementNumPageClear()             { ++s_NumPageClear; }
    static void IncrementNumIpc()                   { ++s_NumIpc; }

private:
    static CoreLocalRegion& GetCoreLocalRegion()
    {
        return *reinterpret_cast<CoreLocalRegion*>(KCPU::GetCurrentValue());
    }
    static CoreLocalRegion& GetCoreLocalRegion(int32_t coreNo)
    {
        return *GetTypedPointer<CoreLocalRegion>(KMemoryLayout::GetCoreLocalRegionBegin() + (1 + coreNo) * NN_KERN_V_ADDR_CORE_LOCAL_REGION_SIZE);
    }

private:
    static State                                s_State;
    static KKernelPageTable                     s_KernelPageTable;
    static KMemoryManager                       s_KernelHeapManager;
    static KPageTableManager                    s_PageTableManager;
    static KMemoryBlockResourceManager          s_AppMemoryBlockManager;
    static KMemoryBlockResourceManager          s_SysMemoryBlockManager;
    static KBlockInfoManager                    s_BlockInfoManager;
    static KSynchronization                     s_Synchronization;
    static KResourceLimit                       s_SystemResourceLimit;
    static KUnsafeMemory                        s_UnsafeMemory;
    static KDmaManager                          s_DmaManager;
    static KWorkThreadManager                   s_WorkThreadManager[KWorkThreadManager::WORK_THREAD_COUNT];
    static InterlockedVariable<int64_t>         s_NumPageClear;
    static InterlockedVariable<int64_t>         s_NumIpc;
    static KThread                              s_MainThread[KCPU::NUM_CORE];
    static KThread                              s_IdleThread[KCPU::NUM_CORE];
};

}}
