﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KWaitObject.h"
#include "kern_KScopedSchedulingLock.h"

namespace nn { namespace kern {

void KWaitObject::OnTimer()
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    KThread::QueueEntry* pEntry = &m_Root;
    for (;;)
    {
        KThread* pThread = pEntry->pNext;
        if (pThread == nullptr)
        {
            break;
        }
        pThread->WakeUp();
        pEntry = &pThread->m_QueueEntry;
    }
}

Result KWaitObject::Synchronize(int64_t timeoutTick)
{
    KHardwareTimer* pTimer = nullptr;
    KThread* pThread = &GetCurrentThread();

    {
        KScopedSchedulingLock lock;

        if (pThread->IsTerminateRequested())
        {
            return nn::svc::ResultTerminateRequested();
        }

        if (timeoutTick >= 0 && m_TimerUsed)
        {
            return nn::svc::ResultBusy();
        }

        if (timeoutTick > 0)
        {
            m_TimerUsed = true;
            pTimer = &Kernel::GetHardwareTimer();
            pTimer->RegisterTaskAbsolute(this, timeoutTick);
        }

        if (timeoutTick == 0)
        {
            OnTimer();
        }
        else
        {
            Enqueue(pThread);
            pThread->SetState(KThread::STATE_WAIT);
            pThread->SetSyncedObject(NULL, nn::svc::ResultTimeout());
        }
    }

    // プロセス終了時などで、タイムアウトしていないのに待ちが解除されることがある。
    {
        KScopedSchedulingLock lock;

        if (timeoutTick > 0)
        {
            NN_KERN_ASSERT(m_TimerUsed);
            NN_KERN_ASSERT(pTimer != nullptr);
            pTimer->CancelTask(this);
            m_TimerUsed = false;
        }

        if (timeoutTick != 0)
        {
            Remove(pThread);
        }
    }

    return ResultSuccess();
}

}}

