﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_Assert.h"
#include "kern_Result.h"
#include "kern_KLightMutex.h"

namespace nn { namespace kern {
class KUnsafeMemory
{
public:
    KUnsafeMemory():
        m_LimitSize(0),
        m_CurrentSize(0)
    {
        NN_KERN_THIS_ASSERT();
    }

    size_t GetLimitSize() const
    {
        NN_KERN_THIS_ASSERT();
        KScopedLightLock locker(&m_Lock);
        return m_LimitSize;
    }

    size_t GetCurrentSize() const
    {
        NN_KERN_THIS_ASSERT();
        KScopedLightLock locker(&m_Lock);
        return m_CurrentSize;
    }

    Result SetLimitSize(size_t size)
    {
        NN_KERN_THIS_ASSERT();
        KScopedLightLock locker(&m_Lock);
        if (size < m_CurrentSize)
        {
            return nn::svc::ResultLimit();
        }
        m_LimitSize = size;
        return ResultSuccess();
    }

    bool TestLimit(size_t size)
    {
        NN_KERN_THIS_ASSERT();
        KScopedLightLock locker(&m_Lock);
        if (m_CurrentSize > m_CurrentSize + size)
        {
            return false;
        }
        if (m_CurrentSize + size > m_LimitSize)
        {
            return false;
        }
        m_CurrentSize += size;
        return true;
    }

    void ReleaseLimit(size_t size)
    {
        NN_KERN_THIS_ASSERT();
        KScopedLightLock locker(&m_Lock);
        NN_KERN_ABORT_UNLESS(m_CurrentSize >= size);
        m_CurrentSize -= size;
    }

private:
    mutable KLightMutex m_Lock;
    size_t m_LimitSize;
    size_t m_CurrentSize;
};

}}

