﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/util/util_IntrusiveList.h>
#include "kern_Assert.h"
#include "kern_CPUSelect.h"
#include "kern_ContextSelect.h"
#include "kern_KAffinityMask.h"
#include "kern_KLightMutex.h"
#include "kern_KCapability.h"
#include "kern_KSynchronizationObject.h"
#include "kern_KObjectAdaptor.h"
#include "kern_KTimerTask.h"
#include "kern_KWorkerTask.h"
#include "kern_KRbTree.h"

/*! @file

    @brief      KThread クラスの定義です。

*/

namespace nn { namespace kern {

// 前方宣言
class KScheduler;
class KThreadQueue;
class KPriorityQueue;
class KProcess;
class KLightMutex;
class KLockWithPriorityInheritance;
class KWaitObject;

typedef void (*ThreadFunc)(uintptr_t);

class KThread :
    public KObjectAdaptor<KThread, KSynchronizationObject>,
    public KTimerTask,
    public KWorkerTask
{
private:
    static const int32_t PROCESSOR_NOT_REGISTERED   = -1;                           //!< 実行プロセッサが未確定状態

public:
    friend class KLightMutex;
    friend class KProcess;
    friend class KLockWithPriorityInheritance;
    friend class KAddressArbiter;
    friend class KThreadQueue;
    friend class KWaitObject;
    friend class KPriorityQueue;

    static const int32_t IdleThreadPriority = nn::svc::LowestThreadPriority + 1;  //!< アイドルスレッドの優先度。通常のスレッドがとりうる範囲より低い優先度です。
    static_assert(((nn::svc::LowestThreadPriority - nn::svc::HighestThreadPriority + 1) % 32) == 0,"");

    enum SuspendType
    {
        SuspendType_Process = 0,
        SuspendType_Thread,
        SuspendType_Debug,
        SuspendType_BackTrace,
    };

    /*!
        @enum      ThreadState

        @brief     スレッド状態の列挙体です。

    */
    enum ThreadState: uint8_t
    {
        STATE_INITIALIZED,                      //!< スレッドは実行開始前です。
        STATE_WAIT,                             //!< スレッドがスリープ、同期待ち中です
        STATE_RUNNABLE,                         //!< スレッドは実行可能状態です
        STATE_TERMINATED,                       //!< スレッドは終了処理中です。

        STATE_MASK                  = 0x0F,     //!< スレッド実行状態のマスク

        STATE_SUSPENDED_SHIFT       = 4,

        STATE_PROCESS_SUSPENDED     = 0x1 << (STATE_SUSPENDED_SHIFT + SuspendType_Process),
        STATE_THREAD_SUSPENDED      = 0x1 << (STATE_SUSPENDED_SHIFT + SuspendType_Thread),
        STATE_DEBUG_SUSPENDED       = 0x1 << (STATE_SUSPENDED_SHIFT + SuspendType_Debug),

        STATE_SUSPENDED_MASK        = STATE_PROCESS_SUSPENDED | STATE_THREAD_SUSPENDED | STATE_DEBUG_SUSPENDED,
    };

    /*!
        @enum      Type

        @brief     スレッドタイプの列挙体です。

    */
    enum Type
    {
        TYPE_MAIN,                          //!< メインスレッド (最初のスレッド)
        TYPE_KERNEL,                        //!< カーネルスレッド
        TYPE_REALTIME,                      //!< 割り込み処理スレッドなど優先度の非常に高いスレッド
        TYPE_USER                           //!< ユーザーのスレッド
    };
    enum DpcFunc
    {
        DPC_FUNC_TERMINATE_SHIFT = 0,
        DPC_FUNC_TERMINATED_SHIFT,

        DPC_FUNC_TERMINATE              = (1u << DPC_FUNC_TERMINATE_SHIFT),
        DPC_FUNC_TERMINATED             = (1u << DPC_FUNC_TERMINATED_SHIFT),
    };
    struct QueueEntry
    {
    public:
        void Initialize()
        {
            pPrev = NULL;
            pNext = NULL;
        }
    private:
        friend class KThreadQueue;
        friend class KWaitObject;
        friend class KPriorityQueue;
        KThread*    pPrev;
        KThread*    pNext;
    };

private:
    enum
    {
        PriorityInheritanceNestMax = 10
    };

public:
    struct ParamsOnStack
    {
        uint8_t                         svcPermission[KCapability::SVC_CAPABILITY_BYTES] __attribute__ ((aligned(16)));
        InterlockedVariable<uint8_t>    dpcFlags;
        uint8_t                         svcNo;
        bool                            isCallingSvc;
        bool                            inExceptionHandler;
        int32_t                         disableCount;
        KContext*                       pContext;
    };

private:
    typedef ThreadState                            ThreadStateSize1;
    typedef InterlockedVariable<bool>              InterlockedBool;
    typedef InterlockedVariable<uint8_t>           InterlockedBit8;

private:
    KContext                m_Context __attribute__ ((aligned(16)));
    KAffinityMask           m_AffinityMask;                 //!< スレッドの所属可能なプロセッサマスク
    Bit64                   m_ThreadId;                     //!< スレッドID
    InterlockedVariable<int64_t> m_CpuTime;
#if defined(NN_KERN_CORE_UTILIZATION)
    InterlockedVariable<int64_t> m_CoreTime[KCPU::NUM_CORE];
#endif
    KSynchronizationObject* m_SyncedObj;                    //!< 最後に待ち合わせした同期オブジェクト
    KLightMutex*            m_pWaitingLock;                 //!< このスレッドが Unlock を待っている KLightMutex。KLightMutex がこのメンバーを直接操作します。
    uintptr_t               m_CvKey;
    uintptr_t               m_EntryAddress;
    KProcessAddress         m_AddressKey;                   //!< AddressArbiter がアドレスベースでの同期待ちに使用
    KProcess*               m_pParent;                      //!< このスレッドを持つプロセス(親プロセス)のポインタ。親がいない場合 NULL。
    void*                   m_pKernelStackEnd;              //!< SVCモード用スタック
    Bit32*                  m_pLightSessionData;
    KProcessAddress         m_ThreadLocalRegionAddr;        //!< スレッドローカルな領域へのポインタ
    void*                   m_pThreadLocalRegionHeapAddr;
    union SyncObjectBuffer
    {
        SyncObjectBuffer() {}
        KSynchronizationObject* pSync[nn::svc::ArgumentHandleCountMax];
        nn::svc::Handle         handle[sizeof(KSynchronizationObject*) * nn::svc::ArgumentHandleCountMax / sizeof(nn::svc::Handle)];
    };
    SyncObjectBuffer        m_SyncObjectBuffer;

    int64_t                         m_SchedCount;
    int64_t                         m_LeaveTime;
    QueueEntry                      m_PriorityQueueEntry[KCPU::NUM_CORE];
    QueueEntry                      m_QueueEntry;               //!< KThreadQueueで使用
    KThreadQueue*                   m_SleepingQueue;            //!< KThreadQueueで使用
    nn::util::IntrusiveListNode     m_WaiterNode;
    typedef nn::util::IntrusiveList<KThread, nn::util::IntrusiveListMemberNodeTraits<KThread, &KThread::m_WaiterNode>> WaiterList;
    IntrusiveRbTreeNode             m_ConditionVariableNode;
    IntrusiveRbTreeNode             m_AddressArbiterNode;
    nn::util::IntrusiveListNode     m_Node;
    WaiterList                      m_WaiterList;
    KThread*                        m_pLockOwner;
    KLockWithPriorityInheritance*   m_pConditionVariable;
    uintptr_t                       m_DebugParam1;
    uintptr_t                       m_DebugParam2;
    uintptr_t                       m_DebugParam3;

    Bit32                   m_AddressKeyValue;
    Bit32                   m_SuspendRequested;
    Result                  m_WaitResult;                   //!< 最後に待ち合わせした同期オブジェクトのブロック解除理由
    Result                  m_DebugExceptionResult;
    int32_t                 m_Priority;                     //!< 継承を含むスレッド優先度
    int32_t                 m_RunningProcessor;
    int32_t                 m_BasePriority;                 //!< 基本の(継承前の)スレッド優先度
    int32_t                 m_IdealProcessor;               //!< 割り当てが好ましいプロセッサ
    int32_t                 m_NumKernelWaiter;


    KAffinityMask           m_OrgAffinityMask;
    int32_t                 m_OrgIdealProcessor;
    int32_t                 m_CoreMigrarionDisableCount;

    ThreadStateSize1        m_State;                        //!< スレッドの実行、待機状態
    InterlockedBool         m_TerminateRequested;           //!< スレッドが強制シャットダウン中 true
    bool                    m_IpcCancelled;                 //!< セッションの強制切断フラグ用途に使用されます。
    bool                    m_WaitCanceled;
    bool                    m_CanCancel;
    bool                    m_Registered;                   //!< プロセッサを登録したら true
    bool                    m_IsSignaled;                   //!< 同期イベント。シグナル状態 = スレッドシャットダウン中を意味します
    bool                    m_IsInitialized;
    bool                    m_DebugCreateThread;
    int8_t                  m_PriorityInheritanceNest;
    bool                    m_ReleaseHint;
    bool                    m_WaitAddressArbiter;

private:
#ifdef NN_KERN_ENABLE_SVC_PROFILE
    InterlockedVariable<uint32_t> m_NumSvc[128];
#endif
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
    int64_t                 m_RunnableStart;
    int64_t                 m_RunnableTime;
    int64_t                 m_NumSwitch;                                //!< (統計情報) スレッドの切り替わり回数
    int64_t                 m_NumFpuSwitch;                             //!< (統計情報) FPUレジスタセットの切り替わり回数
    int32_t                 m_CurrentProcessor;
    int64_t                 m_NumCoreSwitch;
#endif
#ifdef NN_KERN_ENABLE_IPC_PROFILE
    int64_t                 m_NumIpc;                                   //!< (統計情報) IPCの呼び出し回数
    int64_t                 m_NumIpcRecv;                               //!< (統計情報) IPCの呼び出し回数
    int64_t                 m_NumIpcReply;                              //!< (統計情報) IPCの呼び出し回数
#endif
#ifdef NN_KERN_ENABLE_PERFORMANCE_COUNTER
    InterlockedVariable<uint64_t> m_PerformanceCounter[KCPU::NUM_PERFORMANCE_COUNTER];
#endif

    static InterlockedVariable<int64_t>    s_NumCreated;           //!< スレッドの生成回数。スレッドIDの生成に利用します

public:
    explicit KThread() : m_pWaitingLock(NULL), m_pParent(NULL), m_IsInitialized(false) {}
    virtual  ~KThread(){}

    /*!
        @brief      スレッドを初期化します

        @param[in]  f               スレッドのエントリーポイント
        @param[in]  param           ユーザー定義の任意ポインタ
        @param[in]  svStackBottom   SVC モードでのスレッドスタック下限
        @param[in]  userStackBottom USR モードでのスレッドスタック下限
        @param[in]  prio            スレッドの優先度 (KThread:PRIORITY_*)
        @param[in]  coreNo          割り当てる CPU コア
        @param[in]  pParent         スレッドの所有プロセス
        @param[in]  type            スレッドのタイプ

        @return     スレッドが正しく生成できたら成功を返します

    */
    Result Initialize(
            ThreadFunc              f,
            uintptr_t           param,
            void*       svStackBottom,
            KProcessAddress userStackBottom,
            int32_t              prio,
            int32_t            coreNo,
            KProcess*         pParent,
            Type                type );

    //-----------------------------------------------------------------
    // normal

    /*!
        @brief      スレッドの終了通知

        カレントスレッドを終了します。
        スケジューラーがプリエンプション許可状態で呼び出す必要があります。
        GetCurrentThread().Exit() と呼び出す必要があります。

    */
    void    Exit();

    /*!
        @brief      スレッドの強制停止

        スレッドを停止します。カレントスレッドは強制停止できません。
        pThread->Terminate() と呼び出す必要があります。

    */
    void    Terminate();

    /*!
        @brief      Requests thread termination

        Guarantees that the thread will terminate at
        some point after the call.

        @return     Thread state after request.

    */
    ThreadState RequestTerminate();

    /*!
        @brief      スレッドを"実行"します。

        スケジューラー側からスレッドの初期化処理をします。
        STATE_RUNNABLE であればスケジューラーの実行待機キュー入りします。

    */
    Result Run();

    /*!
        @brief      スレッドを指定時間休眠させます

        @param[in]  time    休眠時間(ナノ秒)

    */
    Result  Sleep(int64_t time);

    //! 使用されていません
    Result  EnableFpu();

    //! 使用されていません
    Result  GetProcessor();

    //! 使用されていません
    Result  OnFirstSchedule();

    /*!
        @brief      スレッドの優先度を設定します

        @param[in]  priority     スレッド優先度 (KThread:PRIORITY_*)

    */
    void SetPriority(int32_t priority);

    /*!
        @brief      カレントスレッドの優先度をIDLEに設定します

        カーネル起動スレッドをIdleスレッド化するために呼ばれます。

    */
    Result  SetPriorityForIdleThread();

    /*!
        @brief      スレッドの実行状態を設定します

        "STATE_RUNNABLE" に変更すると、このスレッドはスケジューラーのランキューに格納されます。
        "STATE_RUNNABLE" 以外に変更すると、このスレッドはスケジューラーの管理から手放されます。
        (ただしデバッガにAttachされていない場合に限る)

        @param[in]  state     スレッド実行状態

    */
    void    SetState(ThreadState state);

    Bit64 GetProcessId() const;

    void AddWaiter(KThread* pThread);
    void RemoveWaiter(KThread* pThread);
    KThread* RemoveWaiterByKey(int32_t* pNumWaiter, KProcessAddress key);

    //-----------------------------------------------------------------
    // inline
    //! スレッド ID を取得します
    virtual Bit64 GetId() const
    {
        NN_KERN_THIS_ASSERT();

        return m_ThreadId;
    }
    //! スレッド優先度を取得します。この値は優先度継承を考慮しています。
    int32_t GetPriority() const
    {
        NN_KERN_THIS_ASSERT();

        return m_Priority;
    }
    //! スレッド優先度を取得します。この値は優先度継承を考慮していません。
    int32_t GetBasePriority() const
    {
        NN_KERN_THIS_ASSERT();
        return m_BasePriority;
    }
    //! このスレッドが Unlock を待っている KLightMutex を取得します。
    KLightMutex* GetWaitingLightMutex()
    {
        NN_KERN_THIS_ASSERT();
        return m_pWaitingLock;
    }
    //! このスレッドの親プロセスを取得します。
    KProcess& GetParent() const
    {
        NN_KERN_THIS_ASSERT();

        return *m_pParent;
    }
    KProcess* GetParentPointer() const
    {
        NN_KERN_THIS_ASSERT();

        return m_pParent;
    }
    //! このスレッドはユーザースレッドか判定します。
    bool IsUserThread() const
    {
        return m_pParent != NULL;
    }

    bool IsWaitCanceled() const { return m_WaitCanceled; }
    void ClearWaitCanceled() { m_WaitCanceled = false; }

    void ClearCanCancel() { m_CanCancel = false; }
    void SetCanCancel()   { m_CanCancel = true; }

    void WaitCancel();
    Bit32* GetLightSessionData() { return m_pLightSessionData; }
    void SetLightSessionData(Bit32* pData) { m_pLightSessionData = pData; }

    //! スレッドローカルな領域を取得します。
    KProcessAddress GetThreadLocalRegionAddr() const
    {
        NN_KERN_THIS_ASSERT();
        return m_ThreadLocalRegionAddr;
    }
    void* GetThreadLocalRegionHeapAddr() const
    {
        NN_KERN_THIS_ASSERT();
        return m_pThreadLocalRegionHeapAddr;
    }

    KSynchronizationObject** GetSynchronizationObjectBuffer() { return &m_SyncObjectBuffer.pSync[0]; }
    nn::svc::Handle*         GetHandleBuffer() { return &m_SyncObjectBuffer.handle[sizeof(m_SyncObjectBuffer.pSync) / sizeof(nn::svc::Handle) - nn::svc::ArgumentHandleCountMax]; }

    void* GetKernelStackBottom() const
    {
        NN_KERN_THIS_ASSERT();
        return m_pKernelStackEnd;
    }

    void* GetStackBottom() const
    {
        return reinterpret_cast<ParamsOnStack*>(m_pKernelStackEnd) - 1;
    }

    //! 優先プロセッサを設定します
    void SetIdealProcessor(int32_t number)
    {
        NN_KERN_THIS_ASSERT();
        m_IdealProcessor = number;
    }

    //! 優先プロセッサを取得します
    int32_t GetIdealProcessor() const
    {
        NN_KERN_THIS_ASSERT();
        return m_IdealProcessor;
    }

    int32_t GetRunningProcessor() const
    {
        NN_KERN_THIS_ASSERT();
        return m_RunningProcessor;
    }

    int64_t GetLeaveTime() const
    {
        NN_KERN_THIS_ASSERT();
        return m_LeaveTime;
    }

    void SetLeaveTime(int64_t tick)
    {
        NN_KERN_THIS_ASSERT();
        m_LeaveTime = tick;
    }

    int64_t GetYieldSchedCount() const
    {
        NN_KERN_THIS_ASSERT();
        return m_SchedCount;
    }

    void SetYieldSchedCount(int64_t count)
    {
        NN_KERN_THIS_ASSERT();
        m_SchedCount = count;
    }

#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
    void SwitchCurrentProcessor(int32_t nextProcessor)
    {
        NN_KERN_THIS_ASSERT();
        if (nextProcessor != m_CurrentProcessor)
        {
            m_NumCoreSwitch++;
            m_CurrentProcessor = nextProcessor;
        }
    }
    int64_t GetNumCoreSwitch() const
    {
        NN_KERN_THIS_ASSERT();
        return m_NumCoreSwitch;
    }
#endif

    void SetRunningProcessor(int32_t runningProcessor)
    {
        NN_KERN_THIS_ASSERT();
        m_RunningProcessor = runningProcessor;
    }

    //! スレッドの実行状態を取得します。
    // CHECK: 関数にconstの追加を推奨します。
    ThreadState GetState() const
    {
        NN_KERN_THIS_ASSERT();

        return static_cast<ThreadState>(m_State & STATE_MASK);
    }

    //! スレッドの実行状態を取得します。(デバッガマスク込み)
    // CHECK: 関数にconstの追加を推奨します。
    ThreadState GetStateRaw() const
    {
        NN_KERN_THIS_ASSERT();

        return m_State;
    }

    //! 最終的にブロッキングしていた同期オブジェクトと、ブロッキング解除要因を設定します。
    Result SetSyncedObject(KSynchronizationObject *pobj, Result result)
    {
        NN_KERN_THIS_ASSERT();

        m_SyncedObj  = pobj;
        m_WaitResult = result;
        return ResultSuccess();
    }

    //! 最終的にブロッキングしていた同期オブジェクトと、ブロッキング解除要因を取得します。
    // CHECK: 関数にconstの追加を推奨します。
    Result GetWaitResult(KSynchronizationObject** ppObj)
    {
        NN_KERN_THIS_ASSERT();

        *ppObj = m_SyncedObj;
        return m_WaitResult;
    }

    void SetDebugExceptionResult(Result result)
    {
        NN_KERN_THIS_ASSERT();
        m_DebugExceptionResult = result;
    }

    Result GetDebugExceptionResult() const
    {
        NN_KERN_THIS_ASSERT();
        return m_DebugExceptionResult;
    }

    //! Affinity Mask を取得します。
    const KAffinityMask& GetAffinityMask() const { return m_AffinityMask; }

    Result GetCoreMask(int32_t* pIdealCore, Bit64* pAffinityMask);
    Result SetCoreMask(int32_t idealCore, Bit64 affinityMask);

    //! このスレッドの汎用レジスターセットを取得します。
    KContext* GetContext()
    {
        NN_KERN_THIS_ASSERT();
        return &m_Context;
    }
    //! このスレッドの汎用レジスターセットを取得します。
    const KContext* GetContext() const
    {
        NN_KERN_THIS_ASSERT();
        return &m_Context;
    }
    KContext* GetContextPointer();
    //! KAddressArbiterが内部的に使用します
    KProcessAddress GetAddressKey() const
    {
        NN_KERN_THIS_ASSERT();
        return m_AddressKey;
    }
    Bit32 GetAddressKeyValue() const
    {
        return m_AddressKeyValue;
    }
    //! KAddressArbiterが内部的に使用します
    void SetAddressKey(KProcessAddress addr)
    {
        NN_KERN_THIS_ASSERT();
        m_AddressKey = addr;
    }
    void SetAddressKey(KProcessAddress addr, Bit32 val)
    {
        NN_KERN_THIS_ASSERT();
        m_AddressKey = addr;
        m_AddressKeyValue = val;
    }
    KThread* GetLockOwner() const
    {
        return m_pLockOwner;
    }

    void SetAddressArbiter(KProcessAddress addr)
    {
        NN_KERN_THIS_ASSERT();
        m_AddressKey = addr;
        m_WaitAddressArbiter = true;
    }
    void ClearAddressArbiter()
    {
        NN_KERN_THIS_ASSERT();
        m_WaitAddressArbiter = false;
    }
    bool InWaitingForAddressArbiter() const
    {
        NN_KERN_THIS_ASSERT();
        return m_WaitAddressArbiter;
    }
    void SetupForAddressArbiterCompare(KProcessAddress addr, int priority)
    {
        NN_KERN_THIS_ASSERT();
        m_AddressKey = addr;
        m_Priority = priority;
    }

    uintptr_t GetConditionVariableKeyValue() const
    {
        NN_KERN_THIS_ASSERT();
        return m_CvKey;
    }
    // KConditionVariableCompare から以外は使用しないこと。
    void SetupForConditionVariableCompare(uintptr_t cvKey, int priority)
    {
        NN_KERN_THIS_ASSERT();
        m_CvKey = cvKey;
        m_Priority = priority;
    }
    void SetConditionVariable(KLockWithPriorityInheritance* pCv, KProcessAddress addr, uintptr_t cvKey, Bit32 ownValue)
    {
        NN_KERN_THIS_ASSERT();
        m_pConditionVariable = pCv;
        m_AddressKey = addr;
        m_AddressKeyValue = ownValue;
        m_CvKey = cvKey;
    }
    void ClearConditionVariable()
    {
        NN_KERN_THIS_ASSERT();
        m_pConditionVariable = NULL;
    }
    bool InWaitingForConditionVariable() const
    {
        NN_KERN_THIS_ASSERT();
        return m_pConditionVariable != NULL;
    }
    bool HasWaiterThread() const
    {
        NN_KERN_THIS_ASSERT();
        return !m_WaiterList.empty();
    }
    bool IsTerminateRequested() const
    {
        NN_KERN_THIS_ASSERT();
        return m_TerminateRequested || m_State == STATE_TERMINATED;
    }
    void RegisterDpc(DpcFunc f)
    {
        GetParamsOnStack().dpcFlags |= f;
    }
    void UnregisterDpc(DpcFunc f)
    {
        GetParamsOnStack().dpcFlags &= ~f;
    }
    bool IsDpcRegistered() const
    {
        return GetParamsOnStack().dpcFlags != 0;
    }
    uint8_t GetDpc() const
    {
        return GetParamsOnStack().dpcFlags;
    }
    void ClearDpc()
    {
        GetParamsOnStack().dpcFlags = 0;
    }
    void ClearSvc()
    {
        GetParamsOnStack().isCallingSvc = false;
    }
    void ClearInExceptionHandler()
    {
        GetParamsOnStack().inExceptionHandler = false;
    }
    bool IsCallingSvc() const
    {
        return GetParamsOnStack().isCallingSvc;
    }
    uint8_t GetSvcNo() const
    {
        return GetParamsOnStack().svcNo;
    }
    void SetInExceptionHandler()
    {
        GetParamsOnStack().inExceptionHandler = true;
    }
    bool IsInExceptionHandler()
    {
        return GetParamsOnStack().inExceptionHandler;
    }

    uintptr_t GetEntryAddress() const
    {
        NN_KERN_THIS_ASSERT();

        return m_EntryAddress;
    }

    void SaveDebugParams(uintptr_t param1, uintptr_t param2, uintptr_t param3)
    {
        m_DebugParam1 = param1;
        m_DebugParam2 = param2;
        m_DebugParam3 = param3;
    }

    void RestoreDebugParams(uintptr_t* pParam1, uintptr_t* pParam2, uintptr_t* pParam3) const
    {
        *pParam1 = m_DebugParam1;
        *pParam2 = m_DebugParam2;
        *pParam3 = m_DebugParam3;
    }

    bool IsSuspended() const { return !!m_SuspendRequested; }
    Bit32 GetSuspendedRaw() const { return m_SuspendRequested; }

    void SuspendRequest(SuspendType type);
    void Resume(SuspendType type);
    bool IsSuspendRequested(SuspendType type) const { return !!(m_SuspendRequested & (1u << (STATE_SUSPENDED_SHIFT + type))); }
    void TrySuspend();
    void Continue();

    Result SetActivity(nn::svc::ThreadActivity activity);
    Result GetThreadContext3(nn::svc::ThreadContext* pContext);

    void DisableCoreMigration();
    void EnableCoreMigration();
    void DisableDispatch()
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_MIN_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 0);
        GetParamsOnStack().disableCount++;
    }
    void EnableDispatch()
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_MIN_ASSERT(GetCurrentThread().GetDisableDispatchCount(), 1);
        GetParamsOnStack().disableCount--;
    }
    int32_t GetDisableDispatchCount() const
    {
        NN_KERN_THIS_ASSERT();
        return GetParamsOnStack().disableCount;
    }

private:
    void Suspend();
public:
    bool GetDebugCreateThread() const { return m_DebugCreateThread; }
    void SetDebugCreateThread() { m_DebugCreateThread = true; }

#ifdef NN_KERN_ENABLE_SVC_PROFILE
    uint32_t GetNumSystemCall(int id)
    {
        uint32_t old = m_NumSvc[id].Read();
        for (;;)
        {
            uint32_t v = m_NumSvc[id].CompareAndSwap(old, 0);
            if (v == old)
            {
                break;
            }
            old = v;
        }
        return old;
    }
    void IncrementNumSvc(int id);
#endif
#ifdef NN_KERN_ENABLE_SWITCH_PROFILE
    int64_t GetRunnableTime()  const;
    int64_t GetNumSwitch()     const { return m_NumSwitch; }
    int64_t GetNumFpuSwitch()  const { return m_NumFpuSwitch; }
    void IncrementNumSwitch();
    void IncrementNumFpuSwitch();
#endif
#ifdef NN_KERN_ENABLE_IPC_PROFILE
    int64_t GetNumIpc()        const { return m_NumIpc; }
    int64_t GetNumIpcRecv()    const { return m_NumIpcRecv; }
    int64_t GetNumIpcReply()   const { return m_NumIpcReply; }
    void IncrementNumIpc();
    void IncrementNumIpcRecv();
    void IncrementNumIpcReply();
#endif
#if defined(NN_KERN_ENABLE_PERFORMANCE_COUNTER)
    void AddPerformanceCounter(int n, uint64_t delta)
    {
        m_PerformanceCounter[n] += delta;
    }
    uint64_t GetPerformanceCounter(int n) const
    {
        return m_PerformanceCounter[n];
    }
#endif

    int64_t GetCpuTime()   const
    {
        NN_KERN_THIS_ASSERT();
        return m_CpuTime;
    }
    int64_t GetCpuTime(int coreNo)   const
    {
        NN_KERN_THIS_ASSERT();
        NN_KERN_ABORT_UNLESS(0 <= coreNo && coreNo < KCPU::NUM_CORE);
#if defined(NN_KERN_CORE_UTILIZATION)
        return m_CoreTime[coreNo];
#else
        return 0;
#endif
    }
    void AddCpuTime(int coreNo, int64_t add)
    {
        NN_KERN_THIS_ASSERT();
        NN_UNUSED(coreNo);
#if defined(NN_KERN_CORE_UTILIZATION)
        m_CoreTime[coreNo] += add;
#endif
        m_CpuTime += add;
    }
    //-----------------------------------------------------------------
    // override

    virtual bool IsInitialized() const { return m_IsInitialized; }
    virtual uintptr_t GetPostFinalizeArgument() const { return (reinterpret_cast<uintptr_t>(m_pParent) | (m_ReleaseHint? 1: 0)); }
    static void PostFinalize(uintptr_t arg);


    // override KAutoObject
    virtual void    Finalize();

    // override KSynchronizationObject
    virtual bool    IsSignaled() const;

    // override KTimerTask
    virtual void OnTimer();

    /*!
        @brief     ワーカータスクのエントリーポイント

    */
    // override KWorkerTask
    virtual void DoTask();

    /*!
        @brief  割り込みハンドラ中での処理です。

                割り込みベクタから割り込み禁止状態で呼ばれます。
                ハンドラ中で行うことのみを記述します。

        @param[in]  interruptNo 発生した割り込みの割り込み番号

        @return DPC スレッドから起動されるべきタスクを返します。NULLの時は起動しません。

    */

    //-----------------------------------------------------------------
    // static
    static KThread* GetThreadById(Bit64 id);
    static Result GetThreadList(int32_t* pNumThreads, svc::KUserPointer<Bit64*> pThreadIds, int32_t arraySize);


    static int64_t GetNumCreated()          { return s_NumCreated; }

    size_t GetKernelStackUsage() const;
    Result GetUserStackInfo(size_t* pUsageSize, size_t* pSize) const;

private:
    void ShutdownStep1();
    void ShutdownStep2();
    NN_FORCEINLINE void AddWaiterImpl(KThread* pThread);
    NN_FORCEINLINE void RemoveWaiterImpl(KThread* pThread);
    //! スレッドの継承つき優先度を更新します
    NN_FORCEINLINE static void RestorePriority(KThread* pThread);

    void WakeUp();

    ParamsOnStack& GetParamsOnStack()
    {
        return *( reinterpret_cast<ParamsOnStack*>(m_pKernelStackEnd) - 1 );
    }
    const ParamsOnStack& GetParamsOnStack() const
    {
        return *( reinterpret_cast<const ParamsOnStack*>(m_pKernelStackEnd) - 1 );
    }
    void SetLockOwner(KThread* pThread)
    {
        m_pLockOwner = pThread;
    }
    void FillKernelStack();

    //! KAutoObjectのプリセット関数セット定義です。クラスの末尾に記述する必要があります
    NN_AUTOOBJECT_DERIVED_FUNCSET(KThread, KSynchronizationObject)
};


}}

