﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KScheduler.h"
#include "kern_KSynchronization.h"
#include "kern_InterruptManagerSelect.h"
#include "kern_HardwareTimerSelect.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KScopedSchedulingLockAndSleep.h"

namespace nn { namespace kern {


KSynchronization::KSynchronization()
{
    NN_KERN_THIS_ASSERT();
}

KSynchronization::~KSynchronization()
{
    NN_KERN_THIS_ASSERT();
}

/*!
    @brief      一つ以上の同期オブジェクトに対してシグナル待ちをします。

    オブジェクトがシグナル状態になるか、指定時間経過するまでブロッキングします。
    スレッド pIndex は実行可能状態である必要があります。

    @param[out] pIndex          何番目の同期オブジェクトがシグナル状態になったかを取得
    @param[in]  pThread         同期待ち対象のスレッド
    @param[in]  pObjs           同期待ちオブジェクトポインタが並んだ配列
    @param[in]  num             同期待ちオブジェクトポインタ数
    @param[in]  timeout         timeoutTick 起床時間(tick) 0以下の場合は起床しない

    @return     成功、タイムアウト、ハンドルが閉じられた等の終了要因を返します。

*/
Result KSynchronization::Wait(
    int32_t*                pIndex,
    KSynchronizationObject* pObjs[],
    const int32_t           num,
    int64_t                 timeout)
{
    NN_KERN_THIS_ASSERT();

    KSynchronizationObject::ThreadIterator* threadIterator =
        static_cast<KSynchronizationObject::ThreadIterator*>(__builtin_alloca(sizeof(KSynchronizationObject::ThreadIterator) * num));

    // pThread == GetCurrentThread() を想定して実装しています。

    KThread* pThread = &GetCurrentThread();
    int32_t syncedObjectIndex = -1;
    KHardwareTimer* pTimer;

    {
        // スケジューリングロックとスリープ前処理
        KScopedSchedulingLockAndSleep sleep(&pTimer, pThread, timeout);

        // 既に条件を満たしていないかチェック
        // 一つでも同期オブジェクトがシグナル状態であれば待機から抜けます

        // num == 0 だったらタイムアウトが発生するまで待つ

        for (int32_t i = 0; i < num; i++)
        {
            NN_KERN_ASSERT(pObjs[i] != NULL);

            // 試しに同期オブジェクトに対して"取得"処理を呼びます
            if (pObjs[i]->IsSignaled())
            {
                // 最低一つ確保できたので返る
                *pIndex = i;

                // スリープをキャンセルして戻ります。
                sleep.Cancel();
                return ResultSuccess();
            }
        }

        // タイムアウト時間が0の場合
        if( timeout == 0 )
        {
            // 即タイムアウトで抜けます
            sleep.Cancel();
            return nn::svc::ResultTimeout();
        }

        if( pThread->IsTerminateRequested() )
        {
            sleep.Cancel();
            return nn::svc::ResultTerminateRequested();
        }

        if( pThread->IsWaitCanceled() )
        {
            sleep.Cancel();
            pThread->ClearWaitCanceled();
            return nn::svc::ResultCancelled();
        }

        for (int32_t i = 0; i < num; i++)
        {
            threadIterator[i] = pObjs[i]->RegisterWaitingThread(pThread);
        }

        pThread->SetCanCancel();

        pThread->SetSyncedObject(NULL, nn::svc::ResultTimeout());
        pThread->SetState(KThread::STATE_WAIT);
    }

    // Can Cancel は WAIT状態と合わせて判定する
    pThread->ClearCanCancel();

    // この時点オブジェクトのロックを取得できている
    // または問題が発生している

    if (pTimer != nullptr)
    {
        // タイマーのキャンセル
        pTimer->CancelTask(pThread);
    }

    Result waitResult;
    {
        KScopedSchedulingLock locker;
        KSynchronizationObject* pSyncedObj;
        waitResult = pThread->GetWaitResult(&pSyncedObj);

        for (int32_t i = 0; i < num; i++)
        {
            pObjs[i]->UnregisterWaitingThread(threadIterator[i]);
            if (pObjs[i] == pSyncedObj)
            {
                syncedObjectIndex = i;
            }
        }
    }

    // 番号を返す
    *pIndex = syncedObjectIndex;
    return waitResult;
}

/*!
    @brief      同期オブジェクトの待機状態が変化した際の通知

    同期オブジェクトがシグナル状態になるなど、
    状態が変化した際に呼ばれます。

    @param[in]      pObj    シグナル状態になった同期オブジェクト

*/
void KSynchronization::OnAvailable(KSynchronizationObject *pObj)
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock locker;

    if (pObj->IsSignaled())
    {
        for (KSynchronizationObject::ThreadIterator itr = pObj->GetWaitQueueBeginIterator(); itr != pObj->GetWaitQueueEndIterator(); ++itr)
        {
            KThread* pThread = itr.GetCurrentItem();
            if (pThread->GetState() == KThread::STATE_WAIT)
            {
                pThread->SetSyncedObject(pObj, ResultSuccess());
                pThread->SetState(KThread::STATE_RUNNABLE);
            }
        }
    }

}   // NOLINT(readability/fn_size)

/*!
    @brief      同期オブジェクトが閉じられる際に呼ばれます。

    @param[in]      pObj    閉じられる対象の同期オブジェクト
    @param[in]      reason  閉じた理由

*/
void KSynchronization::OnAbort(KSynchronizationObject* pObj, Result reason)
{
    NN_KERN_THIS_ASSERT();
    KScopedSchedulingLock locker;

    // この同期オブジェクトがブロッキングしているスレッドをたたき起こして回ります
    for (KSynchronizationObject::ThreadIterator itr = pObj->GetWaitQueueBeginIterator(); itr != pObj->GetWaitQueueEndIterator(); ++itr)
    {
        KThread* pThread = itr.GetCurrentItem();

        if (pThread->GetState() == KThread::STATE_WAIT)
        {
            pThread->SetSyncedObject(pObj, reason);
            pThread->SetState(KThread::STATE_RUNNABLE);
        }
    }
}

}}
