﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KReadableEvent.h"
#include "kern_KEvent.h"
#include "kern_KScopedSchedulingLock.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KReadableEvent);

void KReadableEvent::Initialize(KEvent* pParentEvent)
{
    NN_KERN_THIS_ASSERT();

    m_Signaled      = false;
    m_pParentEvent  = pParentEvent;
}

void KReadableEvent::Destroy()
{
    if (m_pParentEvent)
    {
        m_pParentEvent->Close();
    }
}

/*
    @brief      イベントはシグナル状態かを取得します。

    @return     シグナル状態だった場合 true

*/
bool KReadableEvent::IsSignaled() const
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    return m_Signaled;
}

/*
    @brief      イベントをシグナル状態にします。

*/
Result KReadableEvent::Signal()
{
    NN_KERN_THIS_ASSERT();

    KScopedSchedulingLock locker;

    if (!m_Signaled)
    {
        // シグナル状態のfalse -> trueトリガが発生したらスレッドのスケジューリングを行います。
        m_Signaled = true;
        NotifyAvailable();
    }

    return ResultSuccess();
}

Result KReadableEvent::Clear()
{
    NN_KERN_THIS_ASSERT();

    Reset();

    return ResultSuccess();
}

Result KReadableEvent::Reset()
{
    NN_KERN_THIS_ASSERT();

    KScopedSchedulingLock locker;

    if (m_Signaled)
    {
        // シグナル状態をリセットします
        m_Signaled = false;
        return ResultSuccess();
    }
    else
    {
        return nn::svc::ResultInvalidState();
    }
}


}}
