﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_KPageGroup.h"
#include "kern_KScheduler.h"
#include "kern_Kernel.h"
#include "kern_KScopedSchedulingLock.h"

namespace nn { namespace kern {

inline bool KBlockInfo::TryConcatenate(KVirtualAddress addr, size_t numPages)
{
    NN_KERN_THIS_ASSERT();
    // 0x1_0000_0000 と  0x0 が連続だと見なされないよう対処
    // KBlockInfo はブロックの順序を維持する必要があるので、後方にのみ連結可能。
    if(addr == m_Addr + m_NumPages * NN_KERN_FINEST_PAGE_SIZE && addr != 0)
    {
        m_NumPages += numPages;

        return true;
    }

    return false;
}

inline bool KBlockInfo::IsLowerThan(KVirtualAddress addr) const
{
    NN_KERN_THIS_ASSERT();
    KVirtualAddress farRight = m_Addr + m_NumPages * NN_KERN_FINEST_PAGE_SIZE;

    // 0x1_0000_0000 が 0x0 と見なされないよう対処
    if(m_Addr !=0 && farRight == 0)
    {
        return false;
    }
    else
    {
        return farRight < addr;
    }
}


void KPageGroup::Initialize(KBlockInfoManager* pManager)
{
    NN_KERN_THIS_ASSERT();
    m_pManager = pManager;
}

Result KPageGroup::AddBlock(KVirtualAddress addr, size_t numPages)
{
    NN_KERN_THIS_ASSERT();

    if (numPages == 0)
    {
        return ResultSuccess();
    }

    NN_KERN_ASSERT(addr < addr + numPages * NN_KERN_FINEST_PAGE_SIZE);

    // 末尾の BlockInfo に連結できないかチェックする。連結できたら関数を抜ける。
    if (!m_BlockList.empty())
    {
        BlockInfoList::iterator it = m_BlockList.end();
        --it;
        if (it->TryConcatenate(addr, numPages))
        {
            return ResultSuccess();
        }
    }

    // 末尾の BlockInfo に連結できなかったら、新しく BlockInfo を作って末尾に加える。
    KBlockInfo* pNewBlock = m_pManager->Allocate();
    if (pNewBlock == nullptr)
    {
        return nn::svc::ResultOutOfResource();
    }
    pNewBlock->Initialize(addr, numPages);
    m_BlockList.push_back(*pNewBlock);
    return ResultSuccess();
}


void KPageGroup::Finalize()
{
    NN_KERN_THIS_ASSERT();

    BlockInfoList::iterator it = m_BlockList.begin();
    while (it != m_BlockList.end())
    {
        KBlockInfo* pInfo = &*it;
        it = m_BlockList.erase(it);
        m_pManager->Free(pInfo);
    }
}

void KPageGroup::Open() const
{
    KMemoryManager& mm = Kernel::GetKernelHeapManager();

    for (BlockInfoList::const_iterator it = m_BlockList.begin(); it != m_BlockList.end(); it++)
    {
        mm.Open(it->GetBlockAddr(), it->GetNumPages());
    }
}

void KPageGroup::Close() const
{
    KMemoryManager& mm = Kernel::GetKernelHeapManager();

    for (BlockInfoList::const_iterator it = m_BlockList.begin(); it != m_BlockList.end(); it++)
    {
        mm.Close(it->GetBlockAddr(), it->GetNumPages());
    }
}

bool KPageGroup::IsSamePages(const KPageGroup& rhs) const
{
    BlockInfoList::const_iterator it1  = this->m_BlockList.begin();
    BlockInfoList::const_iterator it2  = rhs  .m_BlockList.begin();
    BlockInfoList::const_iterator end1 = this->m_BlockList.end();
    BlockInfoList::const_iterator end2 = rhs  .m_BlockList.end();

    while ((it1 != end1) && (it2 != end2))
    {
        if (!it1->IsSamePages(*it2))
        {
            return false;
        }

        ++it1;
        ++it2;
    }

    return (it1 == end1) && (it2 == end2);
}

size_t KPageGroup::GetTotalNumPages() const
{
    size_t numPages = 0;

    for (BlockInfoList::const_iterator it = m_BlockList.begin(); it != m_BlockList.end(); it++)
    {
        numPages += it->GetNumPages();
    }

    return numPages;
}

}}

