﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_KHandleTable.h"
#include "kern_Result.h"
#include "kern_Panic.h"
#include "kern_KTaggedAddress.h"
#include "kern_Kernel.h"
#include "kern_KResourceLimit.h"
#include "kern_KScopedResourceLimitTester.h"
#include "kern_KInterruptEvent.h"
#include <nn/nn_BitTypes.h>

namespace nn { namespace kern {

/* =======================================================================
        インスタンスメンバ / public
   ======================================================================== */

KHandleTable::KHandleTable()
: m_pTable(nullptr)
, m_pFreeHead(nullptr)
, m_Count(0)
{
    NN_KERN_THIS_ASSERT();
}

/*!
    @brief     テーブルを初期化します。

    @param[in]  tableSize   テーブル数

    @return    初期化が正常に行われた場合は成功を、テーブル数が大きすぎる場合は SUMMARY_OUT_OF_RESOURCE を返します。

*/
Result KHandleTable::Initialize(int32_t tableSize)
{
    NN_KERN_THIS_ASSERT();
    // TORIAEZU: 初期化は本当は Finalize で行われているかも。

    if( tableSize > NN_KERN_HANDLE_TABLE_SIZE )
    {
        return nn::svc::ResultOutOfMemory();
    }

    // メンバー変数のテーブルを使用する
    m_pTable        = m_Table;
    m_TableSize     = (tableSize <= 0) ? NN_KERN_HANDLE_TABLE_SIZE: tableSize;
    m_NextSerialId  = 1; // INFO: ハンドルの値を ALL 0 にしないために 1 から始める
    m_Count         = 0;
    m_MaxCount      = 0;

    // テーブルの初期化
    // 次の空き領域へのアドレスを格納する (フリーリストを繋いでいく)
    for(int32_t i = 0; i < m_TableSize - 1; ++i)
    {
        m_pTable[i].SetFree(&m_pTable[i + 1]);
    }
    m_pTable[m_TableSize - 1].SetFree(nullptr);

    m_pFreeHead = &m_pTable[0];

    return ResultSuccess();
}

/*!
    @brief     終了処理

    @return    SUMMARY_NOT_SUPPORTEDを返します。

*/
Result KHandleTable::Finalize()
{
    NN_KERN_THIS_ASSERT();

    // 全てのハンドルを解放する
    int32_t tableSize;
    TableEntry* pTable;
    {
        KScopedDisableDispatch dd;
        KScopedSimpleLock lock(&m_Lock);

        tableSize = m_TableSize;
        pTable = m_pTable;

        m_TableSize = 0;
        m_pTable = nullptr;
    }
    {
        for( int i = 0; i < tableSize; ++i )
        {
            TableEntry* p = &pTable[i];

            KAutoObject* pObj = p->GetObject();

            if (pObj)
            {
                // 自動オブジェクトの終了処理
                pObj->Close();

                // フリーエントリーに接続します
                FreeEntryImpl(p);
            }
        }
    }

    return ResultSuccess();
}


/*!
    @brief     ハンドルを削除します。

    @param[in]  handle   削除対象のハンドル

    @return    正常に削除できたら true を返します。

*/
bool KHandleTable::Remove(nn::svc::Handle handle)
{
    NN_KERN_THIS_ASSERT();

    KAutoObject* pObj;

    // 擬似ハンドルを渡してはいけない
    if (handle == nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS ||
            handle == nn::svc::PSEUDO_HANDLE_CURRENT_THREAD)
    {
        NN_WARNING(false, "Pseudo handle cannot remove");
        return false;
    }
    if (GetHandleFlags(HandleToBit32(handle)) != 0)
    {
        return false;
    }

    {
        KScopedDisableDispatch dd;
        KScopedSimpleLock lock(&m_Lock);

        TableEntry* pEntry;

        pEntry = FindEntry(handle);

        if( pEntry == nullptr )
        {
            // ハンドルが見つからない
            return false;
        }

        pObj = pEntry->GetObject();

        // エントリを解放
        FreeEntry(pEntry);
        m_Count--;
    }

    // 参照カウントを減らす
    pObj->Close();

    return true;
}



/* =======================================================================
        インスタンスメンバ / private
   ======================================================================== */
/*!
    @brief     オブジェクトを登録します。

    @param[out]  pOut   オブジェクトに対応するハンドル
    @param[in]  obj     登録するオブジェクト
    @param[in]  type    登録するオブジェクトの型情報

    @return    正常に登録できた場合は成功を、テーブルが溢れた場合は SUMMARY_OUT_OF_RESOURCE を返します。

*/
Result KHandleTable::Add(nn::svc::Handle* pOut, KAutoObject* obj, Bit16 type)
{
    NN_KERN_THIS_ASSERT();
    KScopedDisableDispatch dd;
    KScopedSimpleLock lock(&m_Lock);

    if( m_Count >= m_TableSize )
    {
        // これ以上はハンドルが最大値を超える
        return nn::svc::ResultMaxHandle();
    }

    // テーブルから空きを一つ得る
    TableEntry& entry = AllocateEntry();

    // ハンドル値を作成
    const int32_t index = &entry - m_pTable;
    const Bit32 handleValue = MakeHandleValue(index, m_NextSerialId, 0); // TORIAEZU: flags は常に 0

    // セットアップ
    entry.SetUsing(obj, m_NextSerialId, type);
    m_Count++;
    if( m_Count > m_MaxCount )
    {
        m_MaxCount = m_Count;
    }
    obj->Open();

    // シリアル ID 更新
    m_NextSerialId++;
    if( m_NextSerialId > SERIAL_ID_MAX )
    {
        // INFO: ハンドルの値を ALL 0 にしないために 1 から始める
        m_NextSerialId = 1;
    }

    *pOut = nn::svc::Handle(handleValue);
    return ResultSuccess();
}

Result KHandleTable::Reserve(nn::svc::Handle* pOut)
{
    NN_KERN_THIS_ASSERT();
    KScopedDisableDispatch dd;
    KScopedSimpleLock lock(&m_Lock);

    if( m_Count >= m_TableSize )
    {
        // これ以上はハンドルが最大値を超える
        return nn::svc::ResultMaxHandle();
    }

    // テーブルから空きを一つ得る
    TableEntry& entry = AllocateEntry();

    // ハンドル値を作成
    const int32_t index = &entry - m_pTable;
    const Bit32 handleValue = MakeHandleValue(index, m_NextSerialId, 0); // TORIAEZU: flags は常に 0

    // nullptrにすることで、FindEntry()で見つからない
    // このエントリは Register or Unreserve でしか使用できない
    NN_KERN_ASSERT(entry.GetObject() == nullptr);

    m_Count++;
    if( m_Count > m_MaxCount )
    {
        m_MaxCount = m_Count;
    }

    // シリアル ID 更新
    m_NextSerialId++;
    if( m_NextSerialId > SERIAL_ID_MAX )
    {
        // INFO: ハンドルの値を ALL 0 にしないために 1 から始める
        m_NextSerialId = 1;
    }

    *pOut = nn::svc::Handle(handleValue);
    return ResultSuccess();
}

void KHandleTable::Register(nn::svc::Handle handle, KAutoObject* obj, Bit16 type)
{
    NN_KERN_THIS_ASSERT();
    KScopedDisableDispatch dd;
    KScopedSimpleLock lock(&m_Lock);

    const Bit32 handleValue = HandleToBit32(handle);
    const int32_t index    = GetHandleIndex(handleValue);
    const int32_t serialId = GetHandleSerialId(handleValue);

    NN_KERN_ASSERT(serialId);
    NN_KERN_ASSERT(index < m_TableSize);

    TableEntry* pEntry = &m_pTable[index];

    NN_KERN_ASSERT(pEntry);
    NN_KERN_ASSERT(pEntry->GetObject() == nullptr);

    pEntry->SetUsing(obj, serialId, type);
    obj->Open();
}

void KHandleTable::Unreserve(nn::svc::Handle handle)
{
    NN_KERN_THIS_ASSERT();

    KScopedDisableDispatch dd;
    KScopedSimpleLock lock(&m_Lock);

    const Bit32 handleValue = HandleToBit32(handle);
    const int32_t index    = GetHandleIndex(handleValue);
    const int32_t serialId = GetHandleSerialId(handleValue);
    NN_UNUSED(serialId);

    NN_KERN_ASSERT(serialId);
    NN_KERN_ASSERT(index < m_TableSize);

    TableEntry* pEntry = &m_pTable[index];

    NN_KERN_ASSERT(pEntry);
    NN_KERN_ASSERT(pEntry->GetObject() == nullptr);

    FreeEntry(pEntry);
    m_Count--;
}

/*!
    @brief     テーブルエントリーを確保します。

    呼び出す前に必ずフリーエントリーがあるか確認する必要があります。

    @return     確保できたテーブル
*/
KHandleTable::TableEntry& KHandleTable::AllocateEntry()
{
    NN_KERN_THIS_ASSERT();
    TableEntry& entry = *m_pFreeHead;
    m_pFreeHead = entry.GetFreeEntry();
    return entry;
}

/*!
    @brief     フリーリストに追加します。

    エントリーをフリーし、フリーリストの先頭にエントリーを追加します。

    @param[in]  pEntry フリー対象/リスト追加対象のエントリー

*/
void KHandleTable::FreeEntryImpl(KHandleTable::TableEntry* pEntry)
{
    pEntry->SetFree(m_pFreeHead);
    m_pFreeHead = pEntry;
}
void KHandleTable::FreeEntry(KHandleTable::TableEntry* pEntry)
{
    NN_KERN_THIS_ASSERT();
    FreeEntryImpl(pEntry);
}


KAutoObject* KHandleTable::GetObjectForIpc(nn::svc::Handle handle) const
{
    NN_KERN_THIS_ASSERT();
    KScopedDisableDispatch dd;
    KScopedSimpleLock lock(&m_Lock);
    KAutoObject* pObject = GetObjectImpl(handle);

    if (pObject)
    {
        // KInterruptEvent は IPC 転送できない
        if (pObject->DynamicCast<KInterruptEvent*>() != nullptr)
        {
            return nullptr;
        }
        pObject->Open();
    }
    return pObject;
}


/* =======================================================================
        クラスメンバ / public
   ======================================================================== */
#if NN_KERN_ENABLE_OBJECT_INFO
int32_t KHandleTable::GetObjectCount(Bit16 typeId, Bit32 processId) const
{
    NN_KERN_THIS_ASSERT();
    KScopedDisableDispatch dd;
    KScopedSimpleLock lock(&m_Lock);

    int32_t count = 0;
    for ( int32_t i = 0; i < m_TableSize; ++i )
    {
        const TableEntry& entry = m_pTable[i];
        if (entry.GetObject() != nullptr
            && (processId == 0 || entry.GetObject()->GetParentId() == processId)
            && entry.GetType() == typeId)
        {
            ++count;
        }
    }
    return count;
}
#endif

/* =======================================================================
        クラスメンバ / private
   ======================================================================== */


}}

