﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>

///*!
//    @file

//    @brief デバッグイベントに関する API の宣言
#include "kern_KUseSlabAllocator.h"

namespace nn { namespace kern {

class KProcess;
class KThread;

/*!
    @brief      デバッグイベント クラスの定義です。

*/
class KEventInfo :
    public KUseSlabAllocator<KEventInfo>,
    public nn::util::IntrusiveListBaseNode<KEventInfo>
{
public:
    // CreateThread (event == DEBUG_EVENT_CREATE_THREAD)  時に有効な値が書き込まれます
    struct InfoCreateThread
    {
        Bit32            createthreadId;                         //!< 起動スレッド
        uintptr_t        localRegionBaseAddress;                 //!< TLSのアドレス
        uintptr_t        entryAddress;                           //!< スレッドのエントリーアドレス
    };

    // ExitProcess (event == DEBUG_EVENT_CREATE_THREAD) 時に有効な値が書き込まれます
    struct InfoExitProcess
    {
        nn::svc::ProcessExitReason exitReason;             //!< プロセスの終了情報
    };

    // ExitThread (event == DEBUG_EVENT_EXIT_PROCESS) 時に有効な値が書き込まれます
    struct InfoExitThread
    {
        nn::svc::ThreadExitReason  threadExitReason;       //!< スレッド終了イベントの情報
    };

    // Exception (event == DEBUG_EVENT_EXCEPTION)  時に有効な値が書き込まれます
    struct InfoException
    {
        nn::svc::DebugException exceptionCode;          //!< 例外コード
        int32_t                 numberOfExceptionInformation;
        uintptr_t               exceptionAddress;       //!< 例外の発生アドレス
        uintptr_t               exceptionInformation[4]; //!< 例外に関する副次的な情報
    };

    // Schedule (event == DEBUG_EVENT_SCHEDULE*) 時に有効な値が書き込まれます
    struct InfoSchedule
    {
        int64_t                 tick;                   //!< 時刻
        int32_t                 coreId;                 //!< スケジューリングのコア
        NN_PADDING4;
    };

    // SystemCall (event == DEBUG_EVENT_SYSTEM_CALL*) 時に有効な値が書き込まれます
    struct InfoSystemCall
    {
        int64_t                 tick;                   //!< 時刻
        int32_t                 systemCallNumber;       //!< システムコールのsvcId
        NN_PADDING4;
    };

    // OutputString (event == DEBUG_EVENT_OUTPUT_STRING) 時に有効な値が書き込まれます
    struct InfoOutputString
    {
        uintptr_t               bufferAddress;          //!< OutputStringのバッファアドレス
        size_t                  dataSize;               //!< OutputStringのバッファサイズ
    };

    // Map (event == DEBUG_EVENT_MAP) 時に有効な値が書き込まれます
    struct InfoMap
    {
        nn::svc::MemoryInfo     memoryInfo;             //!< メモリマップ情報
    };

    union uInfo
    {
        InfoCreateThread    createThread;
        InfoExitProcess     exitProcess;
        InfoExitThread      exitThread;
        InfoException       exception;
        InfoSystemCall      systemCall;
    };

    struct Info
    {
        uInfo   info;
    };

public:
    nn::svc::DebugEvent         event;                  //!< 発生したデバッグイベントの種別
    Bit32                       threadId;               //!< デバッグイベントの発生したスレッド, もしくは新規作成したスレッド
    Bit32                       flags;                  //!< enum nn::svc::DebugEvent もしくはNULL

    // for ContinueQueue
    bool                        isAttach;
    bool                        continueFlag;            //!< ContinueEventかどうか
    bool                        ignoreContinue;
    bool                        closeOnce;

    Info                        eventInfo;

    // for debug
    KThread*                    debugthread;
    NN_PADDING4;


    explicit KEventInfo() :
        isAttach(false),
        continueFlag(false),
        ignoreContinue(false)
    {
    }
    ~KEventInfo() {}
};


}}

