﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/svc/svc_BaseId.autogen.h>

#include "kern_Platform.h"
#include "kern_DebugSelect.h"
#include "kern_Kernel.h"
#include "kern_MemoryMap.h"
#include "kern_KProcess.h"
#include "kern_KSynchronization.h"
#include "kern_KTaggedAddress.h"
#include "kern_KScheduler.h"
#include "kern_DebugString.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_PageTableSelect.h"
#include "kern_MemoryCopySelect.h"
#include "kern_Utility.h"

namespace nn { namespace kern {
namespace
{
template <typename T>
    T* GetContextAtSuperVisorStackBottom(void* pBottom)
{
    return reinterpret_cast<T*>(reinterpret_cast<uintptr_t>(pBottom) - sizeof(KThread::ParamsOnStack)) - 1;
}

inline KDebugBase* GetDebugObject(KProcess* p)
{
    return reinterpret_cast<KDebugBase*>(p->GetDebugObject());
}

}

Result KDebugBase::QueryInfo(nn::svc::MemoryInfo* pBlock, nn::svc::PageInfo* pPage, KProcessAddress addr)
{
    KMemoryInfo mi = {};
    Result result;

    KScopedLightLock locker(&m_Mutex);

    if (!m_Process || m_Process->IsTerminated())
    {
        return nn::svc::ResultProcessTerminated();
    }

    result = m_Process->GetPageTable().QueryInfo(&mi, pPage, addr);
    *pBlock = mi.ConvertToSvcMemoryInfo();
    return result;
}

Result KDebugBase::ReadMemory(void* buf, KProcessAddress addr, size_t size)
{
    Result result;

    KScopedLightLock locker(&m_Mutex);

    if (!m_Process || m_Process->IsTerminated())
    {
        return nn::svc::ResultProcessTerminated();
    }

    KProcessPageTable& tgtPt    = m_Process->GetPageTable();
    KProcessPageTable& mntPt    = GetCurrentProcess().GetPageTable();
    if (!tgtPt.IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!mntPt.IsInRange(KProcessAddress(buf), size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcessAddress testAddr = addr;
    while (size)
    {
        KMemoryInfo mi;
        nn::svc::PageInfo pi;

        result = tgtPt.QueryInfo(&mi, &pi, testAddr);
        if (result.IsFailure())
        {
            return result;
        }

        if (mi.state == static_cast<KMemoryState>(nn::svc::MemoryState_Inaccessible))
        {
            return nn::svc::ResultInvalidAddress();
        }

        size_t testSize = size;
        if (static_cast<size_t>(KProcessAddress(mi.baseAddress + mi.size) - testAddr) < testSize)
        {
            testSize = (KProcessAddress(mi.baseAddress + mi.size) - testAddr);
        }

        if (mi.state == KMemoryState_Io)
        {
            if ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
            {
                return nn::svc::ResultInvalidAddress();
            }
            KPhysicalAddress physAddr;
            KProcessAddress ioAddr;
            tgtPt.GetPhysicalAddress(&physAddr, testAddr);
            result = mntPt.MapIo(RoundDown(physAddr, NN_KERN_FINEST_PAGE_SIZE), NN_KERN_FINEST_PAGE_SIZE, KMemoryPermission_UserRead);
            if (result.IsFailure())
            {
                return result;
            }
            if (RoundDown(testAddr + NN_KERN_FINEST_PAGE_SIZE, NN_KERN_FINEST_PAGE_SIZE) < testAddr + testSize)
            {
                testSize = RoundDown(testAddr + NN_KERN_FINEST_PAGE_SIZE, NN_KERN_FINEST_PAGE_SIZE) - testAddr;
            }
            result = mntPt.QueryIoMapping(&ioAddr, RoundDown(physAddr, NN_KERN_FINEST_PAGE_SIZE), NN_KERN_FINEST_PAGE_SIZE);
            NN_KERN_ASSERT(result.IsSuccess());
            ioAddr += (testAddr & (NN_KERN_FINEST_PAGE_SIZE - 1));

            switch ((testSize | GetAsInteger(testAddr)) & 3)
            {
            case 0:
                {
                    if (!ReadIoMemory32(buf, GetUntypedPointer(ioAddr), testSize))
                    {
                        result = nn::svc::ResultInvalidPointer();
                    }
                }
                break;
            case 2:
                {
                    if (!ReadIoMemory16(buf, GetUntypedPointer(ioAddr), testSize))
                    {
                        result = nn::svc::ResultInvalidPointer();
                    }
                }
                break;
            default:
                {
                    if (!ReadIoMemory8(buf, GetUntypedPointer(ioAddr), testSize))
                    {
                        result = nn::svc::ResultInvalidPointer();
                    }
                }
                break;
            }
            Result resultFree = mntPt.UnmapPages(RoundDown(ioAddr, NN_KERN_FINEST_PAGE_SIZE), 1, KMemoryState_Io);
            NN_KERN_ABORT_IF_FAILED(resultFree);
            if (result.IsFailure())
            {
                return result;
            }
        }
        else
        {
            result = tgtPt.ReadDebugMemory(buf, testAddr, testSize);
            if (result.IsFailure())
            {
                return result;
            }
        }

        buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + testSize);
        testAddr += testSize;
        size -= testSize;
    }

    return ResultSuccess();
}

Result KDebugBase::WriteMemory(const void* buf, KProcessAddress addr, size_t size)
{
    Result result;

    KScopedLightLock locker(&m_Mutex);

    if (!m_Process || m_Process->IsTerminated())
    {
        return nn::svc::ResultProcessTerminated();
    }

    KProcessPageTable& tgtPt    = m_Process->GetPageTable();
    KProcessPageTable& mntPt    = GetCurrentProcess().GetPageTable();

    if (!tgtPt.IsInRange(addr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!mntPt.IsInRange(KProcessAddress(buf), size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcessAddress testAddr = addr;
    while (size)
    {
        KMemoryInfo mi;
        nn::svc::PageInfo pi;

        result = tgtPt.QueryInfo(&mi, &pi, testAddr);
        if (result.IsFailure())
        {
            return result;
        }

        if (mi.state == static_cast<KMemoryState>(nn::svc::MemoryState_Inaccessible))
        {
            return nn::svc::ResultInvalidAddress();
        }

        size_t testSize = size;
        if (static_cast<size_t>(KProcessAddress(mi.baseAddress + mi.size) - testAddr) < testSize)
        {
            testSize = (KProcessAddress(mi.baseAddress + mi.size) - testAddr);
        }

        if (mi.state == KMemoryState_Io)
        {
            if ((mi.permission & KMemoryPermission_UserReadWrite) != KMemoryPermission_UserReadWrite)
            {
                return nn::svc::ResultInvalidAddress();
            }

            KPhysicalAddress physAddr;
            KProcessAddress ioAddr;
            tgtPt.GetPhysicalAddress(&physAddr, testAddr);
            result = mntPt.MapIo(RoundDown(physAddr, NN_KERN_FINEST_PAGE_SIZE), NN_KERN_FINEST_PAGE_SIZE, KMemoryPermission_UserReadWrite);
            if (result.IsFailure())
            {
                return result;
            }
            if (RoundDown(testAddr + NN_KERN_FINEST_PAGE_SIZE, NN_KERN_FINEST_PAGE_SIZE) < testAddr + testSize)
            {
                testSize = RoundDown(testAddr + NN_KERN_FINEST_PAGE_SIZE, NN_KERN_FINEST_PAGE_SIZE) - testAddr;
            }
            result = mntPt.QueryIoMapping(&ioAddr, RoundDown(physAddr, NN_KERN_FINEST_PAGE_SIZE), NN_KERN_FINEST_PAGE_SIZE);
            NN_KERN_ASSERT(result.IsSuccess());
            ioAddr += (testAddr & (NN_KERN_FINEST_PAGE_SIZE - 1));

            switch ((testSize | GetAsInteger(testAddr)) & 3)
            {
            case 0:
                {
                    if (!WriteIoMemory32(GetUntypedPointer(ioAddr), buf, testSize))
                    {
                        result = nn::svc::ResultInvalidPointer();
                    }
                }
                break;
            case 2:
                {
                    if (!WriteIoMemory16(GetUntypedPointer(ioAddr), buf, testSize))
                    {
                        result = nn::svc::ResultInvalidPointer();
                    }
                }
                break;
            default:
                {
                    if (!WriteIoMemory8(GetUntypedPointer(ioAddr), buf, testSize))
                    {
                        result = nn::svc::ResultInvalidPointer();
                    }
                }
                break;
            }
            Result resultFree = mntPt.UnmapPages(RoundDown(ioAddr, NN_KERN_FINEST_PAGE_SIZE), 1, KMemoryState_Io);
            NN_KERN_ABORT_IF_FAILED(resultFree);
            if (result.IsFailure())
            {
                return result;
            }
        }
        else
        {
            result = tgtPt.WriteDebugMemory(testAddr, buf, testSize);
            if (result.IsFailure())
            {
                return result;
            }
        }

        buf = reinterpret_cast<void*>(reinterpret_cast<uintptr_t>(buf) + testSize);
        testAddr += testSize;
        size -= testSize;
    }

    return ResultSuccess();
}

void KDebugBase::Initialize()
{
    m_ContinueFlag = 0;
    m_Process = nullptr;
}

KProcess* KDebugBase::GetProcess()
{
    KScopedLightLock locker(&m_Mutex);
    KProcess* pProcess = m_Process;
    if (pProcess)
    {
        pProcess->Open();
    }
    return pProcess;
}

bool KDebugBase::Is64Bit() const
{
    NN_KERN_ASSERT(m_Process);
    return m_Process->Is64Bit();
}

Result KDebugBase::Attach(KProcess* pProcess)
{
    NN_KERN_ASSERT(pProcess);

    {
        KScopedLightLock procLocker(&pProcess->GetStateMutex());
        KScopedLightLock procListLocker(&pProcess->GetListMutex());
        KScopedLightLock locker(&m_Mutex);
        KScopedSchedulingLock lock;

        if (pProcess->IsAttachedByDebugger())
        {
            return nn::svc::ResultBusy();
        }

        {
            KProcess::State state = pProcess->GetState();
            switch (state)
            {
            case KProcess::State_Initializing:
            case KProcess::State_Running:
            case KProcess::State_WaitAttach:
                break;
            case KProcess::State_PreAttached:
            case KProcess::State_Attached:
            case KProcess::State_Breaked:
                return nn::svc::ResultBusy();
            case KProcess::State_Terminating:
            case KProcess::State_Terminated:
                return nn::svc::ResultProcessTerminated();
            default:
                {
                    NN_KERN_ABORT();
                }
                break;
            }

            m_Process = pProcess;
            pProcess->Open();
            m_OldState = pProcess->SetDebugObject(this);

            PushDebugEvent(nn::svc::DebugEvent_CreateProcess);

            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                it->SuspendRequest(KThread::SuspendType_Debug);
                if (it->GetState() == KThread::STATE_WAIT || it->GetState() == KThread::STATE_RUNNABLE)
                {
                    it->SetDebugCreateThread();
                    PushDebugEvent(nn::svc::DebugEvent_CreateThread, it->GetId(), GetAsInteger(it->GetThreadLocalRegionAddr()), it->GetEntryAddress());
                }
            }
            KEventInfo* pInfo = pProcess->GetJitInfo();
            if (pInfo)
            {
                PushEventQueue(pInfo);
            }
            PushDebugEvent(nn::svc::DebugEvent_Exception, nn::svc::DebugException_AttachBreak);
            NotifyAvailable();
        }
    }

    return ResultSuccess();
}

Result KDebugBase::ContinueDebug(const Bit32 flags, const Bit64* threadIds, size_t numThreads)
{
    KProcess* pProcess = GetProcess();
    if (!pProcess)
    {
        return nn::svc::ResultProcessTerminated();
    }
    KScopedAutoObject<KProcess> autoCloser(pProcess);

    KScopedLightLock procLocker(&pProcess->GetStateMutex());
    KScopedLightLock procListLocker(&pProcess->GetListMutex());
    KScopedLightLock locker(&m_Mutex);
    KScopedSchedulingLock schlocker;

    if (m_Process != pProcess || pProcess->IsTerminated())
    {
        return nn::svc::ResultProcessTerminated();
    }
    if (!IsEmptyEventQueue())
    {
        return nn::svc::ResultBusy();
    }

    pProcess->ClearJitInfo();

    m_ContinueFlag = flags;

    KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
    // すべて実行中でないならば、Breaked
    bool hadBreakedThread = false;
    for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
    {
        if (m_ContinueFlag & nn::svc::ContinueFlag_ContinueAll)
        {
            if (m_ContinueFlag & nn::svc::ContinueFlag_ExceptionHandled)
            {
                it->SetDebugExceptionResult(nn::svc::ResultStopProcessingException());
            }
            it->Resume(KThread::SuspendType_Debug);
        }
        else if (m_ContinueFlag & nn::svc::ContinueFlag_ContinueOthers)
        {
            Bit64 threadId = it->GetId();
            size_t i;
            for (i = 0; i < numThreads; i++)
            {
                if (threadId == threadIds[i])
                {
                    break;
                }
            }
            if (i == numThreads)
            {
                if (m_ContinueFlag & nn::svc::ContinueFlag_ExceptionHandled)
                {
                    it->SetDebugExceptionResult(nn::svc::ResultStopProcessingException());
                }
                it->Resume(KThread::SuspendType_Debug);
            }
        }
        else
        {
            Bit64 threadId = it->GetId();
            for (size_t i = 0; i < numThreads; i++)
            {
                if (threadId == threadIds[i])
                {
                    if (m_ContinueFlag & nn::svc::ContinueFlag_ExceptionHandled)
                    {
                        it->SetDebugExceptionResult(nn::svc::ResultStopProcessingException());
                    }
                    it->Resume(KThread::SuspendType_Debug);
                    break;
                }
            }
        }
        if (it->IsSuspendRequested(KThread::SuspendType_Debug))
        {
            hadBreakedThread = true;
        }
    }
    if (hadBreakedThread)
    {
        pProcess->SetBreaked();
    }
    else
    {
        pProcess->SetAttached();
    }

    return ResultSuccess();
}


KEventInfo* KDebugBase::CreateDebugEvent(const nn::svc::DebugEvent eventtype, uintptr_t param1, uintptr_t param2, uintptr_t param3, uintptr_t param4, uintptr_t param5, Bit64 threadid)
{
    KEventInfo* pInfo = KEventInfo::Allocate();
    if (pInfo)
    {
        pInfo->threadId = 0;
        pInfo->event = eventtype;
        pInfo->flags = nn::svc::DebugEventFlag_Stopped;
        switch (eventtype)
        {
        case nn::svc::DebugEvent_CreateProcess:
            {
            }
            break;
        case nn::svc::DebugEvent_CreateThread:
            {
                pInfo->threadId                       = param1;

                KEventInfo::InfoCreateThread* pInfoThread = &pInfo->eventInfo.info.createThread;
                pInfoThread->createthreadId           = param1;
                pInfoThread->localRegionBaseAddress   = param2;
                pInfoThread->entryAddress             = param3;
            }
            break;
        case nn::svc::DebugEvent_ExitThread:
            {
                pInfo->threadId               = param1;

                KEventInfo::InfoExitThread* pInfoThread = &pInfo->eventInfo.info.exitThread;
                pInfoThread->threadExitReason = static_cast<nn::svc::ThreadExitReason>(param2);
            }
            break;
        case nn::svc::DebugEvent_ExitProcess:
            {
                KEventInfo::InfoExitProcess* pInfoExit = &pInfo->eventInfo.info.exitProcess;
                pInfoExit->exitReason = static_cast<nn::svc::ProcessExitReason>(param1);
                pInfo->threadId = 0;
                pInfo->flags = 0;
            }
            break;
        case nn::svc::DebugEvent_Exception:
            {
                pInfo->threadId = threadid;

                KEventInfo::InfoException* pInfoException  = &pInfo->eventInfo.info.exception;
                nn::svc::DebugException exception = static_cast<nn::svc::DebugException>(param1);
                pInfoException->exceptionCode = exception;
                pInfoException->numberOfExceptionInformation = 0;

                switch (exception)
                {
                case nn::svc::DebugException_UndefinedSystemCall:
                    {
                        pInfoException->exceptionAddress = param2;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param3;
                    }
                    break;
                case nn::svc::DebugException_UndefinedInstruction:
                    {
                        pInfoException->exceptionAddress = param2;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param3;
                    }
                    break;
                case nn::svc::DebugException_AttachBreak:
                    {
                        pInfo->threadId = 0;
                        pInfoException->exceptionAddress = 0;
                    }
                    break;
                case nn::svc::DebugException_DebuggerBreak:
                    // 強制ブレーク
                    {
                        pInfo->threadId = 0;
                        pInfoException->exceptionAddress = 0;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param2;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param3;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param4;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param5;
                    }
                    break;
                case nn::svc::DebugException_UserBreak:
                    {
                        pInfoException->exceptionAddress = param2;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param3;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param4;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param5;
                    }
                    break;
                case nn::svc::DebugException_BreakPoint:
                    {
                        pInfoException->exceptionAddress = param2;
                        pInfoException->exceptionInformation[pInfoException->numberOfExceptionInformation++] = param3;
                    }
                    break;
                case nn::svc::DebugException_MemorySystemError:
                    {
                        pInfoException->exceptionAddress = 0;
                    }
                    break;
                case nn::svc::DebugException_AccessViolationInstruction:
                case nn::svc::DebugException_DataTypeMissaligned:
                case nn::svc::DebugException_AccessViolationData:
                default:
                    {
                        pInfoException->exceptionAddress = param2;
                    }
                    break;
                }
            }
            break;
        default:
            break;
        }
    }

    return pInfo;
}

void KDebugBase::PushDebugEvent(const nn::svc::DebugEvent eventtype, const uintptr_t param1, const uintptr_t param2, const uintptr_t param3, const uintptr_t param4, const uintptr_t param5)
{
    KEventInfo* pInfo = CreateDebugEvent(eventtype, param1, param2, param3, param4, param5, GetCurrentThread().GetId());
    if (pInfo)
    {
        PushEventQueue(pInfo);
    }
}

Result KDebugBase::ProcessDebugEvent(const nn::svc::DebugEvent eventtype, const uintptr_t param1, const uintptr_t param2, const uintptr_t param3, const uintptr_t param4, const uintptr_t param5)
{
    Result result = ResultSuccess();
    KProcess* pProcess = &GetCurrentProcess();

    if (eventtype == nn::svc::DebugEvent_CreateThread)
    {
        // アタッチ時に 既に作成済み
        if (GetCurrentThread().GetDebugCreateThread())
        {
            return ResultSuccess();
        }
    }

    for (;;)
    {
        KScopedLightLock procLocker(&pProcess->GetStateMutex());
        KScopedLightLock procListLocker(&pProcess->GetListMutex());
        KScopedSchedulingLock locker;

        if (GetCurrentThread().IsTerminateRequested())
        {
            return ResultSuccess();
        }
        KDebugBase* pDebug = GetDebugObject(pProcess);
        if (pDebug == nullptr)
        {
            return ResultSuccess();
        }

        if (eventtype == nn::svc::DebugEvent_Exception && !(pDebug->GetContinueFlag() & nn::svc::ContinueFlag_EnableExceptionEvent))
        {
            GetCurrentThread().SetDebugExceptionResult(ResultSuccess());
            return nn::svc::ResultNotHandled();
        }

        if (GetCurrentThread().IsSuspended())
        {
            continue;
        }

        KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
        for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
        {
            it->SuspendRequest(KThread::SuspendType_Debug);
        }

        pDebug->PushDebugEvent(eventtype, param1, param2, param3, param4, param5);
        pDebug->NotifyAvailable();
        pProcess->SetBreaked();

        if (eventtype == nn::svc::DebugEvent_Exception)
        {
            GetCurrentThread().SetDebugExceptionResult(ResultSuccess());
        }
        break;
    }
    // 停止

    if (eventtype == nn::svc::DebugEvent_Exception)
    {
        KScopedSchedulingLock locker;
        if (GetCurrentThread().IsTerminateRequested())
        {
            // 例外メッセージを抑制する
            return nn::svc::ResultStopProcessingException();
        }

        KDebugBase* pDebug = GetDebugObject(pProcess);
        if (pDebug)
        {
            result = GetCurrentThread().GetDebugExceptionResult();
        }
        else
        {
            result = nn::svc::ResultStopProcessingException();
        }
    }

    return result;
}

Result KDebugBase::OnDebugEvent(
        const nn::svc::DebugEvent eventtype, const uintptr_t param1, const uintptr_t param2, const uintptr_t param3, const uintptr_t param4, const uintptr_t param5)
{
    KProcess* pProcess = &GetCurrentProcess();
    if (pProcess && pProcess->IsAttachedByDebugger())
    {
        return ProcessDebugEvent(eventtype, param1, param2, param3, param4, param5);
    }

    return ResultSuccess();
}

Result KDebugBase::OnDebugEvent(
        const nn::svc::DebugEvent eventtype, const nn::svc::DebugException exception_type,
        const uintptr_t param2, const uintptr_t param3, const uintptr_t param4, const uintptr_t param5)
{
    return OnDebugEvent(eventtype, static_cast<uintptr_t>(exception_type), param2, param3, param4, param5);
}

bool KDebugBase::IsEmptyEventQueue() const
{
    KScopedSchedulingLock locker;
    return m_DebugEventList.empty();
}

void KDebugBase::PushEventQueue(KEventInfo* pInfo)
{
    KScopedSchedulingLock locker;
    m_DebugEventList.push_back(*pInfo);
}

Result KDebugBase::GetDebugEventInfo(nn::svc::DebugEventInfo* pDebugEventInfo)
{
    KEventInfo* pcInfo;
    KProcess* pProcess = GetProcess();
    if (!pProcess)
    {
        return nn::svc::ResultProcessTerminated();
    }
    KScopedAutoObject<KProcess> autoCloser(pProcess);

    {
        KScopedSchedulingLock locker;
        if (m_DebugEventList.empty())
        {
            return nn::svc::ResultNoEvent();
        }

        pcInfo = &m_DebugEventList.front();
        m_DebugEventList.pop_front();
    }

    pDebugEventInfo->event    = pcInfo->event;
    pDebugEventInfo->threadId = pcInfo->threadId;
    pDebugEventInfo->flags    = pcInfo->flags;

    switch(pcInfo->event)
    {
    case nn::svc::DebugEvent_Exception:
        {
            nn::svc::DebugInfoException* pInfoException = &pDebugEventInfo->info.exception;
            const KEventInfo::InfoException* pcInfoException = &pcInfo->eventInfo.info.exception;
            pInfoException->exceptionCode = pcInfoException->exceptionCode;
            pInfoException->exceptionAddress = pcInfoException->exceptionAddress;
            switch(pInfoException->exceptionCode)
            {
            case nn::svc::DebugException_DebuggerBreak:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 4);
                    nn::svc::DebugInfoExceptionDebuggerBreak* pInfoExceptionDebuggerBreak = &pInfoException->detail.debuggerBreak;
                    NN_STATIC_ASSERT(NN_ARRAY_SIZE(pInfoExceptionDebuggerBreak->currentThreadId) == 4);
                    pInfoExceptionDebuggerBreak->currentThreadId[0] = pcInfoException->exceptionInformation[0];
                    pInfoExceptionDebuggerBreak->currentThreadId[1] = pcInfoException->exceptionInformation[1];
                    pInfoExceptionDebuggerBreak->currentThreadId[2] = pcInfoException->exceptionInformation[2];
                    pInfoExceptionDebuggerBreak->currentThreadId[3] = pcInfoException->exceptionInformation[3];
                }
                break;
            case nn::svc::DebugException_UserBreak:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 3);
                    nn::svc::DebugInfoExceptionUserBreak* pInfoExceptionUserBreak = &pInfoException->detail.userBreak;
                    pInfoExceptionUserBreak->reason = static_cast<nn::svc::BreakReason>(pcInfoException->exceptionInformation[0]);
                    pInfoExceptionUserBreak->data = pcInfoException->exceptionInformation[1];
                    pInfoExceptionUserBreak->size = pcInfoException->exceptionInformation[2];
                }
                break;
            case nn::svc::DebugException_UndefinedInstruction:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 1);
                    nn::svc::DebugInfoExceptionUndefinedInstruction* pInfoExceptionUndef = &pInfoException->detail.undefinedInstruction;
                    pInfoExceptionUndef->code = pcInfoException->exceptionInformation[0];
                }
                break;
            case nn::svc::DebugException_UndefinedSystemCall:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 1);
                    nn::svc::DebugInfoExceptionUndefinedSystemCall* pInfoExceptionUndefSyscall = &pInfoException->detail.undefinedSystemCall;
                    pInfoExceptionUndefSyscall->svcId = pcInfoException->exceptionInformation[0];
                }
                break;
            case nn::svc::DebugException_BreakPoint:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 1);
                    nn::svc::DebugInfoExceptionBreakPoint* pInfo = &pInfoException->detail.breakPoint;
                    pInfo->type = static_cast<nn::svc::BreakPointType>(pcInfoException->exceptionInformation[0]);
                    pInfo->dataAddress = 0;
                }
                break;
            default:
                {
                }
                break;
            }
        }
        break;
    case nn::svc::DebugEvent_CreateProcess:
        {
            nn::svc::DebugInfoCreateProcess* pInfoCreateProcess = &pDebugEventInfo->info.createProcess;
            pInfoCreateProcess->programId = pProcess->GetProgramId();
            std::memcpy(pInfoCreateProcess->programName, pProcess->GetName(), sizeof(pInfoCreateProcess->programName));
            pInfoCreateProcess->processId = pProcess->GetId();
            pInfoCreateProcess->flag = pProcess->GetCreateProcessParameterFlag();
            pInfoCreateProcess->processRegionAddress = GetAsInteger(pProcess->GetProcessLocalRegionAddr());
        }
        break;
    case nn::svc::DebugEvent_CreateThread:
        {
            nn::svc::DebugInfoCreateThread* pInfoCreateThread  = &pDebugEventInfo->info.createThread;
            const KEventInfo::InfoCreateThread* pInfoThread = &pcInfo->eventInfo.info.createThread;
            pInfoCreateThread->id = pInfoThread->createthreadId;
            pInfoCreateThread->localRegionBaseAddress = pInfoThread->localRegionBaseAddress;
            pInfoCreateThread->entryAddress = pInfoThread->entryAddress;
        }
        break;
    case nn::svc::DebugEvent_ExitThread:
        {
            nn::svc::DebugInfoExitThread* pInfoExitThread  = &pDebugEventInfo->info.exitThread;
            const KEventInfo::InfoExitThread* pInfoThread = &pcInfo->eventInfo.info.exitThread;
            pInfoExitThread->exitReason = pInfoThread->threadExitReason;
        }
        break;
    case nn::svc::DebugEvent_ExitProcess:
        {
            nn::svc::DebugInfoExitProcess* pInfoExitProcess = &pDebugEventInfo->info.exitProcess;
            const KEventInfo::InfoExitProcess* pInfoExit = &pcInfo->eventInfo.info.exitProcess;
            pInfoExitProcess->exitReason = pInfoExit->exitReason;
        }
        break;
    default:
        {
        }
        break;
    }

    KEventInfo::Free(pcInfo);

    return ResultSuccess();
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result KDebugBase::GetDebugEventInfo(nn::svc::ilp32::DebugEventInfo* pDebugEventInfo)
{
    KEventInfo* pcInfo;
    KProcess* pProcess = GetProcess();
    if (!pProcess)
    {
        return nn::svc::ResultProcessTerminated();
    }
    KScopedAutoObject<KProcess> autoCloser(pProcess);

    {
        KScopedSchedulingLock locker;
        if (m_DebugEventList.empty())
        {
            return nn::svc::ResultNoEvent();
        }

        pcInfo = &m_DebugEventList.front();
        m_DebugEventList.pop_front();
    }

    pDebugEventInfo->event    = pcInfo->event;
    pDebugEventInfo->threadId = pcInfo->threadId;
    pDebugEventInfo->flags    = pcInfo->flags;

    switch(pcInfo->event)
    {
    case nn::svc::DebugEvent_Exception:
        {
            nn::svc::ilp32::DebugInfoException* pInfoException = &pDebugEventInfo->info.exception;
            const KEventInfo::InfoException* pcInfoException = &pcInfo->eventInfo.info.exception;
            pInfoException->exceptionCode = pcInfoException->exceptionCode;
            pInfoException->exceptionAddress = pcInfoException->exceptionAddress;
            switch(pInfoException->exceptionCode)
            {
            case nn::svc::DebugException_DebuggerBreak:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 4);
                    nn::svc::ilp32::DebugInfoExceptionDebuggerBreak* pInfoExceptionDebuggerBreak = &pInfoException->detail.debuggerBreak;
                    NN_STATIC_ASSERT(NN_ARRAY_SIZE(pInfoExceptionDebuggerBreak->currentThreadId) == 4);
                    pInfoExceptionDebuggerBreak->currentThreadId[0] = pcInfoException->exceptionInformation[0];
                    pInfoExceptionDebuggerBreak->currentThreadId[1] = pcInfoException->exceptionInformation[1];
                    pInfoExceptionDebuggerBreak->currentThreadId[2] = pcInfoException->exceptionInformation[2];
                    pInfoExceptionDebuggerBreak->currentThreadId[3] = pcInfoException->exceptionInformation[3];
                }
                break;
            case nn::svc::DebugException_UserBreak:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 3);
                    nn::svc::ilp32::DebugInfoExceptionUserBreak* pInfoExceptionUserBreak = &pInfoException->detail.userBreak;
                    pInfoExceptionUserBreak->reason = static_cast<nn::svc::BreakReason>(pcInfoException->exceptionInformation[0]);
                    pInfoExceptionUserBreak->data = pcInfoException->exceptionInformation[1];
                    pInfoExceptionUserBreak->size = pcInfoException->exceptionInformation[2];
                }
                break;
            case nn::svc::DebugException_UndefinedInstruction:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 1);
                    nn::svc::ilp32::DebugInfoExceptionUndefinedInstruction* pInfoExceptionUndef = &pInfoException->detail.undefinedInstruction;
                    pInfoExceptionUndef->code = pcInfoException->exceptionInformation[0];
                }
                break;
            case nn::svc::DebugException_UndefinedSystemCall:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 1);
                    nn::svc::ilp32::DebugInfoExceptionUndefinedSystemCall* pInfoExceptionUndefSyscall = &pInfoException->detail.undefinedSystemCall;
                    pInfoExceptionUndefSyscall->svcId = pcInfoException->exceptionInformation[0];
                }
                break;
            case nn::svc::DebugException_BreakPoint:
                {
                    NN_KERN_ASSERT(pcInfoException->numberOfExceptionInformation == 1);
                    nn::svc::ilp32::DebugInfoExceptionBreakPoint* pInfo = &pInfoException->detail.breakPoint;
                    pInfo->type = static_cast<nn::svc::BreakPointType>(pcInfoException->exceptionInformation[0]);
                    pInfo->dataAddress = 0;
                }
                break;
            default:
                {
                }
                break;
            }
        }
        break;
    case nn::svc::DebugEvent_CreateProcess:
        {
            nn::svc::ilp32::DebugInfoCreateProcess* pInfoCreateProcess = &pDebugEventInfo->info.createProcess;
            pInfoCreateProcess->programId = pProcess->GetProgramId();
            std::memcpy(pInfoCreateProcess->programName, pProcess->GetName(), sizeof(pInfoCreateProcess->programName));
            pInfoCreateProcess->processId = pProcess->GetId();
            pInfoCreateProcess->flag = pProcess->GetCreateProcessParameterFlag();
            pInfoCreateProcess->processRegionAddress = GetAsInteger(pProcess->GetProcessLocalRegionAddr());
        }
        break;
    case nn::svc::DebugEvent_CreateThread:
        {
            nn::svc::ilp32::DebugInfoCreateThread* pInfoCreateThread  = &pDebugEventInfo->info.createThread;
            const KEventInfo::InfoCreateThread* pInfoThread = &pcInfo->eventInfo.info.createThread;
            pInfoCreateThread->id = pInfoThread->createthreadId;
            pInfoCreateThread->localRegionBaseAddress = pInfoThread->localRegionBaseAddress;
            pInfoCreateThread->entryAddress = pInfoThread->entryAddress;
        }
        break;
    case nn::svc::DebugEvent_ExitThread:
        {
            nn::svc::ilp32::DebugInfoExitThread* pInfoExitThread  = &pDebugEventInfo->info.exitThread;
            const KEventInfo::InfoExitThread* pInfoThread = &pcInfo->eventInfo.info.exitThread;
            pInfoExitThread->exitReason = pInfoThread->threadExitReason;
        }
        break;
    case nn::svc::DebugEvent_ExitProcess:
        {
            nn::svc::ilp32::DebugInfoExitProcess* pInfoExitProcess = &pDebugEventInfo->info.exitProcess;
            const KEventInfo::InfoExitProcess* pInfoExit = &pcInfo->eventInfo.info.exitProcess;
            pInfoExitProcess->exitReason = pInfoExit->exitReason;
        }
        break;
    default:
        {
        }
        break;
    }

    KEventInfo::Free(pcInfo);

    return ResultSuccess();
}
#endif

bool KDebugBase::IsAsynchronousDebugEvent(const nn::svc::DebugEvent eventtype)
{
    switch (eventtype)
    {
    case nn::svc::DebugEvent_ExitProcess:
        return true;
    default:
        break;
    }
    return false;
}

Result KDebugBase::OnExitProcess(KProcess* pProcess)
{
    NN_KERN_ASSERT(pProcess);
    if (pProcess->IsAttachedByDebugger())
    {
        KScopedSchedulingLock locker;

        KDebugBase* pDebug = GetDebugObject(pProcess);
        if (pDebug)
        {
            pDebug->PushDebugEvent(nn::svc::DebugEvent_ExitProcess, (uintptr_t)nn::svc::ProcessExitReason_ExitProcess);
            pDebug->NotifyAvailable();
        }
    }
    return ResultSuccess();
}

Result KDebugBase::OnTerminateProcess(KProcess* pProcess)
{
    NN_KERN_ASSERT(pProcess);
    if (pProcess->IsAttachedByDebugger())
    {
        KScopedSchedulingLock locker;

        KDebugBase* pDebug = GetDebugObject(pProcess);
        if (pDebug)
        {
            pDebug->PushDebugEvent(nn::svc::DebugEvent_ExitProcess, (uintptr_t)nn::svc::ProcessExitReason_TerminateProcess);
            pDebug->NotifyAvailable();
        }
    }
    return ResultSuccess();
}

Result KDebugBase::OnExitThread(KThread* pThread)
{
    NN_KERN_ASSERT(pThread);
    KProcess* pProcess = pThread->GetParentPointer();
    if (pProcess && pProcess->IsAttachedByDebugger())
    {
        nn::svc::ThreadExitReason tReason;
        if (pThread->IsTerminateRequested())
        {
            tReason = nn::svc::ThreadExitReason_TerminateThread;
        }
        else
        {
            tReason = nn::svc::ThreadExitReason_ExitThread;
        }

        return OnDebugEvent(nn::svc::DebugEvent_ExitThread, (uintptr_t)pThread->GetId(), (uintptr_t)tReason);
    }

    return ResultSuccess();
}

Result KDebugBase::BreakProcess()
{
    KProcess* pProcess = GetProcess();
    if (!pProcess)
    {
        return nn::svc::ResultProcessTerminated();
    }
    KScopedAutoObject<KProcess> autoCloser(pProcess);

    KScopedLightLock procLocker(&pProcess->GetStateMutex());
    KScopedLightLock procListLocker(&pProcess->GetListMutex());
    KScopedLightLock locker(&m_Mutex);

    Bit64 threadId[4];
    {
        KScopedSchedulingLock schlocker;
        if (!m_Process || m_Process->IsTerminated())
        {
            return nn::svc::ResultProcessTerminated();
        }

        for (size_t i = 0; i < NN_ARRAY_SIZE(threadId); i++)
        {
            threadId[i] = 0xffffffffffffffffull; // ResultNoThread();
            if (i < KCPU::NUM_CORE)
            {
                KThread* pThread = m_Process->GetRunningThread(i);
                if (m_Process->GetRunningThreadIdleCount(i) == Kernel::GetScheduler(i).GetIdleCount())
                {
                    if (pThread != nullptr &&  static_cast<size_t>(pThread->GetRunningProcessor()) == i)
                    {
                        threadId[i] = pThread->GetId();
                    }
                    else
                    {
                        threadId[i] = 0xfffffffffffffffeull; // ResultUnknownThread();
                    }
                }
            }
        }

        KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
        for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
        {
            it->SuspendRequest(KThread::SuspendType_Debug);
        }

        PushDebugEvent(nn::svc::DebugEvent_Exception, nn::svc::DebugException_DebuggerBreak, threadId[0], threadId[1], threadId[2], threadId[3]);
        NotifyAvailable();
        pProcess->SetBreaked();
    }

    return ResultSuccess();
}

Result KDebugBase::OnTerminateThread(KThread* pThread)
{
    KProcess* pProcess = pThread->GetParentPointer();
    if (pProcess && pProcess->IsAttachedByDebugger())
    {
        KScopedLightLock procLocker(&pProcess->GetStateMutex());
        KScopedLightLock procListLocker(&pProcess->GetListMutex());
        KScopedSchedulingLock locker;

        KDebugBase* pDebug = GetDebugObject(pProcess);
        if (pDebug)
        {
            nn::svc::ThreadExitReason tReason;
            if (pThread->IsTerminateRequested())
            {
                tReason = nn::svc::ThreadExitReason_TerminateThread;
            }
            else
            {
                tReason = nn::svc::ThreadExitReason_ExitThread;
            }

            KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
            for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
            {
                it->SuspendRequest(KThread::SuspendType_Debug);
            }

            pDebug->PushDebugEvent(nn::svc::DebugEvent_ExitThread, (uintptr_t)pThread->GetId(), (uintptr_t)tReason);
            pDebug->NotifyAvailable();
        }
    }

    return ResultSuccess();
}

Result KDebugBase::GetThreadContext(nn::svc::ThreadContext* pContext, const Bit32 threadId, const Bit32 controlFlags)
{
    KScopedLightLock locker(&m_Mutex);

    KThread* pThread = KThread::GetThreadById(threadId);
    if (!pThread)
    {
        return nn::svc::ResultInvalidId();
    }
    KScopedAutoObject<KThread> autoCloser(pThread);

    if (pThread->GetParentPointer() != m_Process)
    {
        return nn::svc::ResultInvalidId();
    }

    if (pThread->GetState() == KThread::STATE_TERMINATED)
    {
        return nn::svc::ResultTerminateRequested();
    }

retry:
    {
        KScopedSchedulingLock locker;
        if (!pThread->IsSuspendRequested(KThread::SuspendType_Debug))
        {
            return nn::svc::ResultInvalidState();
        }
        // カーネル内部のロック競合回避のため実行中となることがある。その場合は既にカーネル内で実行中。
        if (pThread->GetStateRaw() != KThread::STATE_RUNNABLE)
        {
            // 停止完了保証
            for (int i = 0; i < KCPU::NUM_CORE; ++i)
            {
                KThread* pCurrentThread = Kernel::GetCurrentSet(i).pCurrentThread;
                KCPU::DataMemoryBarrier();
                if (pCurrentThread == pThread)
                {
                    goto retry;
                }
            }
        }

        return GetThreadContext_core(pContext, pThread, controlFlags);
    }
}

Result KDebugBase::SetThreadContext(const nn::svc::ThreadContext& context, Bit32 threadId, const Bit32 controlFlags)
{
    KScopedLightLock locker(&m_Mutex);

    KThread* pThread = KThread::GetThreadById(threadId);
    if (!pThread)
    {
        return nn::svc::ResultInvalidId();
    }
    KScopedAutoObject<KThread> autoCloser(pThread);

    if (pThread->GetParentPointer() != m_Process)
    {
        return nn::svc::ResultInvalidId();
    }

    if (pThread->GetState() == KThread::STATE_TERMINATED)
    {
        return nn::svc::ResultTerminateRequested();
    }

    {
        KScopedSchedulingLock locker;
        if (!pThread->IsSuspendRequested(KThread::SuspendType_Debug))
        {
            return nn::svc::ResultInvalidState();
        }
        // カーネル内部のロック競合回避のため実行中となることがある。その場合は既にカーネル内で実行中。
        if (pThread->GetStateRaw() != KThread::STATE_RUNNABLE)
        {
            // 停止完了保証
            for (int i = 0; i < KCPU::NUM_CORE; ++i)
            {
                KThread* pCurrentThread;
                do
                {
                    pCurrentThread = Kernel::GetCurrentSet(i).pCurrentThread;
                    KCPU::DataMemoryBarrier();
                }
                while (pCurrentThread == pThread);
            }
        }

        if (pThread->IsCallingSvc() &&
                (pThread->GetSvcNo() != NN_SVC_ID_BREAK && pThread->GetSvcNo() != NN_SVC_ID_RETURN_FROM_EXCEPTION))
        {
            return nn::svc::ResultInvalidState();
        }

        return SetThreadContext_core(context, pThread, controlFlags);
    }
}

bool KDebugBase::IsSignaled() const
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    return !IsEmptyEventQueue() || !m_Process || m_Process->IsTerminated();
}

Result KDebugBase::TerminateProcess()
{
    KProcess* pProcess = GetProcess();
    if (pProcess)
    {
        {
            KScopedAutoObject<KProcess> autoCloser(pProcess);
            KScopedLightLock procLocker(&pProcess->GetStateMutex());
            KScopedLightLock procListLocker(&pProcess->GetListMutex());
            KScopedLightLock locker(&m_Mutex);

            if (m_Process)
            {
                NN_KERN_ASSERT(m_Process == pProcess);
                {
                    KScopedSchedulingLock schlocker;

                    KProcess::State state = pProcess->GetState();

                    if (state == KProcess::State_Initializing || state == KProcess::State_PreAttached)
                    {
                        return nn::svc::ResultInvalidState();
                    }

                    KProcess::State newState;
                    if (state == KProcess::State_Attached)
                    {
                        newState = KProcess::State_Running;
                    }
                    else if (state == KProcess::State_Breaked)
                    {
                        newState = KProcess::State_WaitAttach;
                    }
                    else
                    {
                        newState = state;
                    }
                    m_Process->ClearDebugObject(newState);
                    m_Process = nullptr;
                    m_ContinueFlag = 0;
                }

            }
        }
        pProcess->Terminate();
        pProcess->Close();
    }

    return ResultSuccess();
}

void KDebugBase::OnFinalizeSynchronization()
{
    KProcess* pProcess = GetProcess();

    if (pProcess)
    {
        {
            KScopedAutoObject<KProcess> autoCloser(pProcess);
            KScopedLightLock procLocker(&pProcess->GetStateMutex());
            KScopedLightLock procListLocker(&pProcess->GetListMutex());
            KScopedLightLock locker(&m_Mutex);

            if (m_Process)
            {
                NN_KERN_ASSERT(m_Process == pProcess);
                {
                    KScopedSchedulingLock schlocker;

                    pProcess->ClearDebugObject(m_OldState);

                    bool isResumed = (pProcess->GetState() != KProcess::State_WaitAttach);

                    KProcess::ThreadList::iterator end = pProcess->GetThreadList().end();
                    for (KProcess::ThreadList::iterator it = pProcess->GetThreadList().begin(); it != end; ++it)
                    {
                        if (isResumed)
                        {
                            it->Resume(KThread::SuspendType_Debug);
                        }
                        else
                        {
                            it->SuspendRequest(KThread::SuspendType_Debug);
                        }
                    }
                    m_Process = nullptr;
                }

            }
        }
        pProcess->Close();
    }

    {
        KScopedSchedulingLock locker;

        while (!m_DebugEventList.empty())
        {
            KEventInfo* pInfo = &m_DebugEventList.front();
            m_DebugEventList.pop_front();
            KEventInfo::Free(pInfo);
        }
    }
}

}}
