﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KPort.h"
#include "kern_KSynchronization.h"
#include "kern_KSession.h"
#include "kern_KLightSession.h"
#include "kern_KScheduler.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KScopedResourceLimitTester.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KClientPort);


    namespace
    {
        struct IncrementIfLessThan
        {
            int32_t comparand;        //!< 比較値
            int32_t newVal;

            bool operator()(int32_t* x)
            {
                if( *x < comparand )
                {
                    newVal = ++*x;
                    return true;
                }
                else
                {
                    return false;
                }
            }
        };

        struct SwapIfLessThan
        {
            int32_t comparand;        //!< 比較値

            bool operator()(int32_t* x)
            {
                if( *x < comparand )
                {
                    *x = comparand;
                    return true;
                }
                else
                {
                    return false;
                }
            }
        };

    }

/* ------------------------------------------------------------------------
        初期化/破棄
   ------------------------------------------------------------------------ */

void KClientPort::Initialize(KPort* p, int32_t maxSessions)
{
    m_NumSessions = 0;
    m_PeakSessions = 0;
    m_MaxSessions = maxSessions;
    m_pParent = p;
}

void KClientPort::Destroy()
{
    m_pParent->OnClientClosed();
    m_pParent->Close();
}



/* ------------------------------------------------------------------------
        KSynchronizationObject
   ------------------------------------------------------------------------ */

bool KClientPort::IsSignaled() const
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    return m_NumSessions < m_MaxSessions;
}


/* ------------------------------------------------------------------------
        xxxx
   ------------------------------------------------------------------------ */
Result KClientPort::CreateSession(KClientSession** ppSession)
{
    NN_KERN_THIS_ASSERT();
    KScopedResourceLimitTester sessionTester(GetCurrentProcessPointer(), nn::svc::LimitableResource_SessionCountMax);
    if (sessionTester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    // セッション数の上限テスト
    IncrementIfLessThan updater = { m_MaxSessions, 0 };

    if( ! m_NumSessions.AtomicUpdateConditional(&updater) )
    {
        return nn::svc::ResultMaxSessions();
    }

    SwapIfLessThan updater2 = { updater.newVal };
    m_PeakSessions.AtomicUpdateConditional(&updater2);

    // Sessionオブジェクトを作成
    // (スラブから一つオブジェクトを生成します)
    KSession* pSession = KSession::Create();
    if (!pSession)
    {
        // スラブに空きがありませんでした
        const int32_t prev = m_NumSessions--;

        if( prev == m_MaxSessions )
        {
            NotifyAvailable();
        }
        return nn::svc::ResultOutOfResource();
    }

    // 初期化とセッションコンテナへの登録
    pSession->Initialize(this, m_pParent->GetName());
    sessionTester.Accepted();
    KSession::Register(pSession);

    // サーバーセッションキューに登録する
    Result result = m_pParent->EnqueueSession(&pSession->GetServerSession());
    if (result.IsFailure())
    {
        pSession->GetClientSession().Close();
        pSession->GetServerSession().Close();
        return result;
    }

    // クライアントセッションを返す
    *ppSession = &pSession->GetClientSession();
    return ResultSuccess();
}

Result KClientPort::CreateLightSession(KLightClientSession** ppSession)
{
    NN_KERN_THIS_ASSERT();
    KScopedResourceLimitTester sessionTester(GetCurrentProcessPointer(), nn::svc::LimitableResource_SessionCountMax);
    if (sessionTester.IsTestFailed())
    {
        return nn::svc::ResultLimit();
    }

    // セッション数の上限テスト
    IncrementIfLessThan updater = { m_MaxSessions, 0 };

    if( ! m_NumSessions.AtomicUpdateConditional(&updater) )
    {
        return nn::svc::ResultMaxSessions();
    }

    // Sessionオブジェクトを作成
    // (スラブから一つオブジェクトを生成します)
    KLightSession* pSession = KLightSession::Create();
    if (!pSession)
    {
        // スラブに空きがありませんでした
        const int32_t prev = m_NumSessions--;

        if( prev == m_MaxSessions )
        {
            NotifyAvailable();
        }
        return nn::svc::ResultOutOfResource();
    }

    // 初期化とセッションコンテナへの登録
    pSession->Initialize(this, m_pParent->GetName());
    sessionTester.Accepted();
    KLightSession::Register(pSession);

    // サーバーセッションキューに登録する
    Result result = m_pParent->EnqueueSession(&pSession->GetServerSession());
    if (result.IsFailure())
    {
        pSession->GetClientSession().Close();
        pSession->GetServerSession().Close();
        return result;
    }

    // クライアントセッションを返す
    *ppSession = &pSession->GetClientSession();
    return ResultSuccess();
}


void KClientPort::OnSessionFinalized()
{
    KScopedSchedulingLock locker;

    const int32_t prev = m_NumSessions--;

    if (prev == m_MaxSessions)
    {
        NotifyAvailable();
    }
}

bool KClientPort::IsLight() const
{
    return m_pParent->IsLight();
}

void KClientPort::OnServerClosed()
{
    NN_KERN_THIS_ASSERT();
}

}}
