﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*!
    @file

    @brief AffinityMask に関する API の宣言

*/

#pragma once

#include <nn/nn_BitTypes.h>
#include "kern_Assert.h"

namespace nn { namespace kern {

/*!
    @brief      AffinityMask クラスの定義です。

    AffinityMask とは、プロセスやスレッドがどの CPU に割り当て可能かを
    示すビットマスク値です。

*/
class KAffinityMask
{
private:
    //! AffinityMask のビットフィールドです
    Bit64 m_Mask;
public:
    //! コンストラクタ
    KAffinityMask(): m_Mask(0)
    {
        NN_KERN_THIS_ASSERT();
    }

    /*!
        @brief      直接メモリを参照して AffinityMask を取得します。
        @return     AffinityMask のビット列を返します。

    */
    Bit64 GetAffinityMask() const { return m_Mask; }

    /*!
        @brief      バッファにある AffinityMask を設定します。

        引数の値のチェックはしません。

    */
    void SetAffinityMask(Bit64 mask)
    {
        NN_KERN_ASSERT(!(~((static_cast<Bit64>(1) << KCPU::NUM_CORE) - 1) & mask));
        m_Mask = mask;
    }

    /*!
        @brief      対象のコアの Affinity を取得します。

    */
    bool GetAffinity(int32_t core) const;

    /*!
        @brief      対象のコアの Affinity を設定します。

    */
    void SetAffinity(int32_t core, bool affinity = true);

    /*!
        @brief      すべてのコアの Affinity を有効にします。

    */
    void SetAll();

};

}}

