﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_CPUSelect.h"
#include "kern_KMemoryManager.h"
#include "kern_KScheduler.h"
#include "kern_Kernel.h"
#include "kern_Current.h"
#include "kern_KThread.h"
#include "kern_KProcess.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_DpcManager.h"
#include "kern_Utility.h"
#include "kern_KLightMutex.h"
#include "kern_KLightConditionvariable.h"
#include <cstring>

namespace nn { namespace kern {
namespace DpcManager {
namespace {
class DpcTask
{
public:
    static void Request(DpcTask* pTask)
    {
        KScopedLightLock locker(&g_Mutex);
        NN_KERN_ABORT_UNLESS(g_RequestedTask == nullptr);
        g_RequestedTask = pTask;
        {
            KScopedLightLock locker(&g_SyncMutex);
            NN_KERN_ABORT_UNLESS(g_TargetCores == 0);

            for (int coreNo = 0; coreNo < KCPU::NUM_CORE; coreNo++)
            {
                g_TargetCores |= (1ull << coreNo);
            }
            g_SyncCv.Broadcast();

            while (g_TargetCores != 0)
            {
                g_SyncCv.Wait(&g_SyncMutex, -1);
            }
        }
        g_RequestedTask = nullptr;
    }

    static void WaitRequest()
    {
        int coreNo = GetCurrentCpuNo();
        KScopedLightLock locker(&g_SyncMutex);
        while ((g_TargetCores & (1ull << coreNo)) == 0)
        {
            g_SyncCv.Wait(&g_SyncMutex, -1);
        }
    }
    static bool WaitRequest(int64_t timeout)
    {
        int coreNo = GetCurrentCpuNo();
        KScopedLightLock locker(&g_SyncMutex);
        while ((g_TargetCores & (1ull << coreNo)) == 0)
        {
            g_SyncCv.Wait(&g_SyncMutex, timeout);
            if (Kernel::GetHardwareTimer().GetTick() >= timeout)
            {
                return false;
            }
        }
        return true;
    }
    static void PostRequest()
    {
        int coreNo = GetCurrentCpuNo();
        KScopedLightLock locker(&g_SyncMutex);
        g_TargetCores &= ~(1ull << coreNo);
        if (g_TargetCores == 0)
        {
            g_SyncCv.Broadcast();
        }
    }

    static void DoTask()
    {
        g_RequestedTask->Task();
    }
    virtual void Task() {}
private:
    static KLightMutex             g_Mutex;
    static KLightMutex             g_SyncMutex;
    static KLightConditionvariable g_SyncCv;
    static Bit64                   g_TargetCores;
    static DpcTask*                g_RequestedTask;
};

KLightMutex             DpcTask::g_Mutex;
KLightMutex             DpcTask::g_SyncMutex;
KLightConditionvariable DpcTask::g_SyncCv;
Bit64                   DpcTask::g_TargetCores;
DpcTask*                DpcTask::g_RequestedTask;

const int DpcManagerThreadPriority = 3;
int g_TimeSlicePriority[KCPU::NUM_CORE];
const int64_t TimeSliceInterval = nn::svc::Tick(TimeSpan::FromMilliSeconds(10));

void ManagerThread(uintptr_t p)
{
    NN_UNUSED(p);
    for (;;)
    {
        DpcTask::WaitRequest();
        DpcTask::DoTask();
        DpcTask::PostRequest();
    }
}

void ManagerThreadWithTimeSlice(uintptr_t p)
{
    NN_UNUSED(p);
    int64_t timeout = KHardwareTimer::GetTick() + TimeSliceInterval;
    for (;;)
    {
        if (DpcTask::WaitRequest(timeout))
        {
            DpcTask::DoTask();
            DpcTask::PostRequest();
        }
        else
        {
            KScopedSchedulingLock lock;
            for (int coreNo = 0; coreNo < KCPU::NUM_CORE; coreNo++)
            {
                int priority = g_TimeSlicePriority[coreNo];
                if (priority > DpcManagerThreadPriority)
                {
                    KScheduler::RotateReadyQueue(priority, coreNo);
                }
            }
            timeout = KHardwareTimer::GetTick() + TimeSliceInterval;
        }
    }
}
}

void DpcHandler()
{
    NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());
    NN_KERN_ASSERT(!KScheduler::IsSchedulerLocked());

    KThread* pCurrentThread = &GetCurrentThread();

    if (pCurrentThread->IsTerminateRequested())
    {
        KEnableInterrupt interruptEnabled;
        pCurrentThread->Exit();
        NN_KERNEL_PANIC("never reach here");
    }
}

void Sync()
{
    NN_KERN_ASSERT(!KScheduler::IsSchedulerLocked());

    class DpcTask req;
    DpcTask::Request(&req);
}

void Initialize(int priority)
{
    NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_ThreadCountMax, 1));
    int coreNo = GetCurrentCpuNo();
    KThread* pThread = KThread::Create();
    NN_KERN_ABORT_UNLESS(pThread);
    g_TimeSlicePriority[coreNo] = priority;
    if (coreNo == KCPU::NUM_CORE - 1)
    {
        NN_KERN_ABORT_IF_FAILED(InitializeThreadForKernel(pThread, &ManagerThreadWithTimeSlice, 0, DpcManagerThreadPriority, coreNo));
    }
    else
    {
        NN_KERN_ABORT_IF_FAILED(InitializeThreadForKernel(pThread, &ManagerThread, 0, DpcManagerThreadPriority, coreNo));
    }
    KThread::Register(pThread);

    pThread->Run();
    NN_LOG("    CoreMaskManagerThread ID=%lld, Core=%d, Affinity=%lx State=%d, Priority=%d\n",
            pThread->GetId(),
            pThread->GetRunningProcessor(),
            pThread->GetAffinityMask().GetAffinityMask(),
            pThread->GetState(),
            pThread->GetPriority());
}

}


}}
