﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include "../../kern_KPageGroup.h"
#include "../../kern_KMemoryManager.h"
#include "../../kern_PageTableSelect.h"

namespace nn { namespace kern { namespace tegra {

typedef uint64_t KDeviceVirtualAddress;

class KDevicePageTable
{
public:
    Result Initialize(uint64_t spaceAddress, uint64_t spaceSize);
    Result Attach(nn::svc::DeviceName deviceName, uint64_t spaceAddress, uint64_t spaceSize);
    Result Detach(nn::svc::DeviceName deviceName);
    Result Map(size_t* pMapSize, const KPageGroup& pageGroup, KDeviceVirtualAddress deviceAddress, nn::svc::MemoryPermission devicePermission, bool breakAllocate);
    Result Unmap(const KPageGroup& pageGroup, KDeviceVirtualAddress deviceAddress);
    void Finalize();
    static void Initialize();
    static void Sleep();
    static void Wakeup();
    static void Lock();
    static void Unlock();
    static void Dump();
    static void PrepareInitialize(const KProcessAddress* pRegisterBase);
    static KPhysicalAddress GetRegisterAddress(int index);
    static int GetRegisterNum();
    static size_t GetRegisterSize(int index);

private:
    enum
    {
#if defined NN_BUILD_CONFIG_SOC_TEGRA_K1
        NumOfTable = 1
#elif defined NN_BUILD_CONFIG_SOC_TEGRA_X1
        NumOfTable = 4
#else
#error not defined NN_BUILD_CONFIG_SOC
#endif
    };

    Bit8 m_Asid[NumOfTable];
    KVirtualAddress m_Table[NumOfTable];

    Bit64 m_AttachedDevice;
    Bit32 m_AttachedValue;
    Bit32 m_DetachedValue;
    Bit32 m_HighSupportAttachedValue;
    Bit32 m_HighSupportDetachedValue;
    static_assert(nn::svc::DeviceName_Num < sizeof(Bit64) * 8, "");

private:
    static bool IsHeapVirtualAddress(KVirtualAddress va)
    {
        return KPageTable::IsHeapVirtualAddress(va);
    }
    static bool IsHeapPhysicalAddress(KPhysicalAddress pa)
    {
        return KPageTable::IsHeapPhysicalAddress(pa);
    }
    static KVirtualAddress GetHeapVirtualAddress(KPhysicalAddress pa)
    {
        return KPageTable::GetHeapVirtualAddress(pa);
    }
    static KPhysicalAddress GetHeapPhysicalAddress(KVirtualAddress va)
    {
        return KPageTable::GetHeapPhysicalAddress(va);
    }
    static KVirtualAddress GetPageTableVirtualAddress(KPhysicalAddress pa)
    {
        return KPageTable::GetPageTableVirtualAddress(pa);
    }
    static KPhysicalAddress GetPageTablePhysicalAddress(KVirtualAddress va)
    {
        return KPageTable::GetPageTablePhysicalAddress(va);
    }

    bool Compare(const KPageGroup& pageGroup, KDeviceVirtualAddress deviceAddress) const;
    Result MakePageGroup(KPageGroup* pOut, KDeviceVirtualAddress addr, uint64_t size) const;
    bool IsFree(KDeviceVirtualAddress addr, uint64_t size) const;

    Result MapDevicePage(size_t* pMapSize, int* pNumPageTable, int pageTableLimit, KPhysicalAddress paddr, uint64_t size, KDeviceVirtualAddress addr, nn::svc::MemoryPermission devicePermission);
    Result MapImpl(size_t* pMapSize, int* pNumPageTable, int pageTableLimit, const KPageGroup& pageGroup, KDeviceVirtualAddress deviceAddress, nn::svc::MemoryPermission devicePermission);

    void Unmap(KDeviceVirtualAddress addr, uint64_t size, bool force);

};

}}} // namespace

