﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/svc/svc_BaseId.autogen.h>
#include "../../kern_Platform.h"
#include "kern_ExceptionHandler.h"
#include "kern_ExceptionHandlerPrint.h"
#include "../../kern_DebugSelect.h"
#include "../../kern_KProcess.h"
#include "../../kern_KThread.h"
#include "../../kern_MemoryCopySelect.h"
#include "../../kern_DpcManager.h"
#include "kern_SystemControl.h"
#include "../../kern_KTrace.h"
#include <cstring>

namespace nn { namespace kern { namespace svc {
void RestoreContext(uintptr_t sp);
}}}

namespace nn { namespace kern { namespace ARMv8A {

namespace
{
    Bit32 GetCode(const ARM64::ExceptionContext* pContext, Bit64 esr)
    {
        Bit64 mode = (pContext->psr & HW_PSR_CPU_MODE_MASK);
        Bit64 il = ((esr >> 25) & 0x1);
        Bit32 code = 0;

        if (mode == HW_PSR_USR_MODE && (pContext->psr & HW_PSR_THUMB_STATE))
        {
            code = *reinterpret_cast<uint16_t*>(pContext->pc & ~0x1);
            if (il)
            {
                code = (code << 16) | *reinterpret_cast<uint16_t*>((pContext->pc & ~0x1) + 2);
            }
        }
        else
        {
            code = *reinterpret_cast<Bit32*>(pContext->pc);
        }
        return code;
    }

    void HandleUserModeException(ARM64::ExceptionContext* pContext, Bit64 esr, Bit64 far, Bit64 afsr0, Bit64 afsr1, Bit32 code)
    {
        KProcess& process = GetCurrentProcess();
        bool isDemandLoadException = false;

        Bit64 ec  = (esr >> 26) & 0x3F;

        switch (ec)
        {
        case HW_ESR_EC_UNKNOWN:
        case HW_ESR_EC_ILLEGAL_EXECUTION:   // Exception from an Illegal execution state
        case HW_ESR_EC_BKPT:
        case HW_ESR_EC_BRK:
        case HW_ESR_EC_MISALIGN_PC: // PC alignment fault
        case HW_ESR_EC_MISALIGN_SP: // SP alignment fault
        case HW_ESR_EC_BREAKPOINT:  // Breakpoint exception
        case HW_ESR_EC_STEP:        // SoftwareStep exception
        case HW_ESR_EC_WATCHPOINT:  // Watchpoint exception
        case HW_ESR_EC_SVC32:
        case HW_ESR_EC_SVC64:
        case HW_ESR_EC_SERROR_INTERRUPT:
            break;
        case HW_ESR_EC_IABORT_EL0:  // Exception from an Instruction abort
        case HW_ESR_EC_DABORT_EL0:  // Exception from a Data abort
        default:
            {
                // デマンドロード対象かチェック
                KMemoryInfo mi;
                nn::svc::PageInfo pi;
                if (process.GetPageTable().QueryInfo(&mi, &pi, far).IsSuccess())
                {
                    if (mi.state == KMemoryState_Code && ((mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead))
                    {
                        isDemandLoadException = true;
                    }
                }
            }
            break;
        }

        if ((isDemandLoadException || KTargetSystem::IsUserExceptionHandlerEnabled()) &&
                !((ec == HW_ESR_EC_ILLEGAL_EXECUTION || ec == HW_ESR_EC_UNKNOWN || ec == HW_ESR_EC_BKPT || ec == HW_ESR_EC_BRK) && process.IsAttachedByDebugger() &&  KDebug::IsBreakInstruction(code, pContext->psr)) &&
                !(ec == HW_ESR_EC_BREAKPOINT || ec == HW_ESR_EC_STEP || ec == HW_ESR_EC_WATCHPOINT))
        {
            if (process.EnterUserException())
            {
                Bit32 cause = 0;

                if ((pContext->psr & HW_PSR_AARCH_MASK) == HW_PSR_AARCH32)
                {
                    nn::svc::aarch32::ProcessLocalRegion* pProcLocal = GetTypedPointer<nn::svc::aarch32::ProcessLocalRegion>(process.GetProcessLocalRegionAddr());
                    nn::svc::aarch32::ExceptionInfo* pInfo = &pProcLocal->info;
                    for (size_t i = 0; i < sizeof(pInfo->r) / sizeof(pInfo->r[0]); i++)
                    {
                        pInfo->r[i] = pContext->x[i];
                    }
                    pInfo->sp = pContext->x[13];
                    pInfo->lr = pContext->x[14];
                    pInfo->pc = pContext->pc;
                    pInfo->flags = 1;
                    pInfo->status.on64.pstate = (pContext->psr & HW_PSR_EL0_MASK);
                    pInfo->status.on64.afsr0 = afsr0;
                    pInfo->status.on64.afsr1 = afsr1;
                    pInfo->status.on64.esr = esr;
                    pInfo->status.on64.far = far;
                }
                else
                {
                    nn::svc::aarch64::ProcessLocalRegion* pProcLocal = GetTypedPointer<nn::svc::aarch64::ProcessLocalRegion>(process.GetProcessLocalRegionAddr());
                    nn::svc::aarch64::ExceptionInfo* pInfo = &pProcLocal->info;
                    for (size_t i = 0; i < sizeof(pInfo->r) / sizeof(pInfo->r[0]); i++)
                    {
                        pInfo->r[i] = pContext->x[i];
                    }
                    pInfo->sp = pContext->sp;
                    pInfo->lr = pContext->x[30];
                    pInfo->pc = pContext->pc;
                    pInfo->pstate = (pContext->psr & HW_PSR_EL0_MASK);
                    pInfo->afsr0 = afsr0;
                    pInfo->afsr1 = afsr1;
                    pInfo->esr = esr;
                    pInfo->far = far;
                }

                // backup

                GetCurrentThread().SaveDebugParams(far, esr, code);
                switch (ec)
                {
                case HW_ESR_EC_UNKNOWN:
                case HW_ESR_EC_ILLEGAL_EXECUTION:  // Exception from an Illegal execution state
                case HW_ESR_EC_BKPT:
                case HW_ESR_EC_BRK:
                case HW_ESR_EC_CP15_MCR_MRC:
                case HW_ESR_EC_CP15_MCRR_MRRC:
                case HW_ESR_EC_CP14_MCR_MRC:
                case HW_ESR_EC_CP14_MRRC:
                case HW_ESR_EC_SYSINST64:
                    {
                        cause = nn::svc::ExceptionType_UndefinedInstruction;
                    }
                    break;
                case HW_ESR_EC_MISALIGN_PC:  // PC alignment fault
                    {
                        cause = nn::svc::ExceptionType_UnalignedInstruction;
                    }
                    break;
                case HW_ESR_EC_MISALIGN_SP:  // SP alignment fault
                    {
                        cause = nn::svc::ExceptionType_UnalignedData;
                    }
                    break;
                case HW_ESR_EC_IABORT_EL0:  // Exception from an Instruction abort
                    {
                        cause = nn::svc::ExceptionType_InstructionAbort;
                    }
                    break;
                case HW_ESR_EC_SVC32:
                    {
                        cause = nn::svc::ExceptionType_InvalidSystemCall;
                    }
                    break;
                case HW_ESR_EC_SVC64:
                    {
                        cause = nn::svc::ExceptionType_InvalidSystemCall;
                    }
                    break;
                case HW_ESR_EC_SERROR_INTERRUPT:
                    {
                        cause = nn::svc::ExceptionType_MemorySystemError;
                    }
                    break;
                case HW_ESR_EC_DABORT_EL0:  // Exception from a Data abort
                default:
                    {
                        cause = nn::svc::ExceptionType_DataAbort;
                    }
                    break;
                }

                pContext->pc = GetAsInteger(process.GetEntryPoint());
                pContext->x[0] = cause;
                if ((pContext->psr & HW_PSR_AARCH_MASK) == HW_PSR_AARCH32)
                {
                    pContext->x[1] = GetAsInteger(process.GetProcessLocalRegionAddr() + offsetof(nn::svc::aarch32::ProcessLocalRegion, info));
                    nn::svc::aarch32::ProcessLocalRegion* pProcLocal = GetTypedPointer<nn::svc::aarch32::ProcessLocalRegion>(process.GetProcessLocalRegionAddr());
                    pContext->x[13] = ((reinterpret_cast<uintptr_t>(pProcLocal->work) + sizeof(pProcLocal->work)) & ~(0x8ul - 1));
                    pContext->psr = HW_PSR_ARM_STATE | HW_PSR_USR_MODE;
                }
                else
                {
                    pContext->x[1] = GetAsInteger(process.GetProcessLocalRegionAddr() + offsetof(nn::svc::aarch64::ProcessLocalRegion, info));
                    nn::svc::aarch64::ProcessLocalRegion* pProcLocal = GetTypedPointer<nn::svc::aarch64::ProcessLocalRegion>(process.GetProcessLocalRegionAddr());
                    pContext->sp = ((reinterpret_cast<uintptr_t>(pProcLocal->work) + sizeof(pProcLocal->work)) & ~(0x10ul - 1));
                    pContext->psr = HW_PSR_EL0T_MODE;
                }
                return;
            }
        }

        {
            Result result = ResultSuccess();
            nn::svc::DebugException except;
            uintptr_t param2 = 0;
            uintptr_t param3 = 0;
            switch (ec)
            {
                // Exception from an Illegal execution state
            case HW_ESR_EC_UNKNOWN:
            case HW_ESR_EC_ILLEGAL_EXECUTION:  // PSTATE.IL is 1
            case HW_ESR_EC_BKPT:
            case HW_ESR_EC_BRK:
                {
                    except = nn::svc::DebugException_UndefinedInstruction;
                    param2 = far;
                    param3 = code;
                }
                break;

                // PC or SP alignment fault
            case HW_ESR_EC_MISALIGN_PC:  // PC alignment fault
            case HW_ESR_EC_MISALIGN_SP:  // SP alignment fault
                {
                    except = nn::svc::DebugException_DataTypeMissaligned;
                    param2 = far;
                }
                break;

                // Exception from an Instruction abort
            case HW_ESR_EC_IABORT_EL0:
                {
                    except = nn::svc::DebugException_AccessViolationInstruction;
                    param2 = far;
                }
                break;

            case HW_ESR_EC_SVC32:
            case HW_ESR_EC_SVC64:
                {
                    except = nn::svc::DebugException_UndefinedSystemCall;
                    param2 = far;
                    param3 = esr & 0xFF;
                }
                break;

            case HW_ESR_EC_BREAKPOINT:
            case HW_ESR_EC_STEP:
                {
                    except = nn::svc::DebugException_BreakPoint;
                    param2 = far;
                    param3 = nn::svc::BreakPointType_HardwareInstruction;
                }
                break;

            case HW_ESR_EC_WATCHPOINT:
                {
                    except = nn::svc::DebugException_BreakPoint;
                    param2 = far;
                    param3 = nn::svc::BreakPointType_HardwareData;
                }
                break;

            case HW_ESR_EC_SERROR_INTERRUPT:
                {
                    except = nn::svc::DebugException_MemorySystemError;
                }
                break;

                // Exception from a Data abort
            default:
            case HW_ESR_EC_DABORT_EL0:
                {
                    except = nn::svc::DebugException_AccessViolationData;
                    param2 = far;
                }
                break;
            }

            result = KDebug::OnDebugEvent(nn::svc::DebugEvent_Exception, except, param2, param3);

            if (result <= nn::svc::ResultStopProcessingException())
            {
                return;
            }

            NN_KERN_FATAL_LOG("Exception occurred. %016lx\n", process.GetProgramId());
            CommonExceptionHandlerDebug(pContext, esr, far);
            KDebug::PrintBacktrace();

            NN_KERN_KTRACE_STOP();

            if (!(result <= nn::svc::ResultNotHandled()))
            {
                if (process.EnterJitDebug(nn::svc::DebugEvent_Exception, except, param2, param3))
                {
                    NN_KERN_KTRACE_DELAYED_DUMP_WITH_ADDITIONAL_INFO();
                    return;
                }
            }
        }

        NN_KERN_KTRACE_DUMP_WITH_ADDITIONAL_INFO();
        process.Exit();
    }
}

void FpuContextSwitchHandler()
{
    KThread& t = GetCurrentThread();
    KContext::FpuSwitchHandler(&t);
}

void HandleException( ARM64::ExceptionContext* pContext )
{
    // INFO: 割り込み禁止状態
    NN_KERN_ASSERT( ! KInterruptManager::IsInterruptEnabled() );

    const Bit64 mode = (pContext->psr & HW_PSR_CPU_MODE_MASK);
    const bool inUserMode = ((mode == HW_PSR_USR_MODE) || (mode == HW_PSR_EL0T_MODE));

    Bit64 esr = 0;
    Bit64 far = 0;
    Bit64 afsr0 = 0;
    Bit64 afsr1 = 0;
    Bit32 code = 0;

    HW_GET_ESR_EL1(esr);
    HW_GET_AFSR0_EL1(afsr0);
    HW_GET_AFSR1_EL1(afsr1);

    Bit64 ec  = (esr >> 26) & 0x3F;

    switch (ec)
    {
    case HW_ESR_EC_BREAKPOINT:
        {
            far = pContext->pc;
        }
        break;

        // Exception from an Illegal execution state
    case HW_ESR_EC_UNKNOWN:
    case HW_ESR_EC_ILLEGAL_EXECUTION:
    case HW_ESR_EC_BKPT:
    case HW_ESR_EC_BRK:
        {
            code = GetCode(pContext, esr);
            far = pContext->pc;
        }
        break;

    case HW_ESR_EC_SVC32:
        {
            if (pContext->psr & HW_PSR_THUMB_STATE)
            {
                pContext->pc -= 2;
            }
            else
            {
                pContext->pc -= 4;
            }
            far = pContext->pc;
        }
        break;

    case HW_ESR_EC_SVC64:
        {
            pContext->pc -= 4;
            far = pContext->pc;
        }
        break;

    default:
        {
            HW_GET_FAR_EL1(far);
        }
        break;
    }

    GetCurrentThread().SetInExceptionHandler();
    if (inUserMode)
    {
        KEnableInterrupt interruptEnabled;

        HandleUserModeException(pContext, esr, far, afsr0, afsr1, code);
    }
    else
    {
        if (KTargetSystem::IsDevelopmentHardware())
        {
            CommonExceptionHandlerDebug(pContext, esr, far);
        }
        NN_KERNEL_PANIC("");
    }

    // INFO: 割り込み禁止状態
    NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());

    while (GetCurrentThread().IsDpcRegistered())
    {
        DpcManager::DpcHandler();
    }
    GetCurrentThread().ClearInExceptionHandler();
}

void ReturnFromException(Result resultValue)
{
    KThread* pCurrent = &GetCurrentThread();

    ARM64::ExceptionContext* pContext = reinterpret_cast<ARM64::ExceptionContext*>(
            reinterpret_cast<uintptr_t>(pCurrent->GetKernelStackBottom()) - sizeof(KThread::ParamsOnStack)) - 1;
    KProcess& process = GetCurrentProcess();

    union
    {
        nn::svc::aarch32::ExceptionInfo info32;
        nn::svc::aarch64::ExceptionInfo info64;
    } info = {};

    if ((pContext->psr & HW_PSR_AARCH_MASK) == HW_PSR_AARCH32)
    {
        nn::svc::aarch32::ProcessLocalRegion* pProcLocal = GetTypedPointer<nn::svc::aarch32::ProcessLocalRegion>(process.GetProcessLocalRegionAddr());
        info.info32 = pProcLocal->info;  // コピーする
    }
    else
    {
        nn::svc::aarch64::ProcessLocalRegion* pProcLocal = GetTypedPointer<nn::svc::aarch64::ProcessLocalRegion>(process.GetProcessLocalRegionAddr());
        info.info64 = pProcLocal->info;  // コピーする
    }

    if (process.LeaveUserException())
    {
        if ((pContext->psr & HW_PSR_AARCH_MASK) == HW_PSR_AARCH32)
        {
            for (size_t i = 0; i < sizeof(info.info32.r) / sizeof(info.info32.r[0]); i++)
            {
                pContext->x[i] = info.info32.r[i];
            }
            pContext->x[13] = info.info32.sp;
            pContext->x[14] = info.info32.lr;
            pContext->pc = info.info32.pc;
            pContext->psr = ((info.info32.status.on64.pstate & HW_PSR_EL0_MASK) | (pContext->psr & ~HW_PSR_EL0_MASK));
        }
        else
        {
            for (size_t i = 0; i < sizeof(info.info64.r) / sizeof(info.info64.r[0]); i++)
            {
                pContext->x[i] = info.info64.r[i];
            }
            pContext->x[30] = info.info64.lr;
            pContext->sp = info.info64.sp;
            pContext->pc = info.info64.pc;
            pContext->psr = ((info.info64.pstate & HW_PSR_EL0_MASK) | (pContext->psr & ~HW_PSR_EL0_MASK));
        }
        pContext->write = 1;

        if (resultValue.IsSuccess())
        {
            nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(pContext));
        }
        else
        {
            uintptr_t far;
            uintptr_t esr;
            uintptr_t code;
            GetCurrentThread().RestoreDebugParams(&far, &esr, &code);
            Bit64 ec  = (esr >> 26) & 0x3F;
            Result result;
            nn::svc::DebugException except;
            uintptr_t param2 = 0;
            uintptr_t param3 = 0;

            switch (ec)
            {
                // Exception from an Illegal execution state
            case HW_ESR_EC_UNKNOWN:
            case HW_ESR_EC_ILLEGAL_EXECUTION:  // PSTATE.IL is 1
            case HW_ESR_EC_BKPT:
            case HW_ESR_EC_BRK:
                {
                    except = nn::svc::DebugException_UndefinedInstruction;
                    param2 = far;
                    param3 = code;
                }
                break;

                // PC or SP alignment fault
            case HW_ESR_EC_MISALIGN_PC:  // PC alignment fault
            case HW_ESR_EC_MISALIGN_SP:  // SP alignment fault
                {
                    except = nn::svc::DebugException_DataTypeMissaligned;
                    param2 = far;
                }
                break;

                // Exception from an Instruction abort
            case HW_ESR_EC_IABORT_EL0:
                {
                    except = nn::svc::DebugException_AccessViolationInstruction;
                    param2 = far;
                }
                break;

            case HW_ESR_EC_SVC32:
            case HW_ESR_EC_SVC64:
                {
                    except = nn::svc::DebugException_UndefinedSystemCall;
                    param2 = far;
                    param3 = esr & 0xFF;
                }
                break;

            case HW_ESR_EC_SERROR_INTERRUPT:
                {
                    except = nn::svc::DebugException_MemorySystemError;
                }
                break;

                // Exception from a Data abort
            default:
            case HW_ESR_EC_DABORT_EL0:
                {
                    except = nn::svc::DebugException_AccessViolationData;
                    param2 = far;
                }
                break;
            }

            result = KDebug::OnDebugEvent(nn::svc::DebugEvent_Exception, except, param2, param3);

            if (result <= nn::svc::ResultStopProcessingException())
            {
                nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(pContext));
            }

            CommonExceptionHandlerDebug(pContext, esr, far);
            KDebug::PrintBacktrace();

            NN_KERN_KTRACE_STOP();

            // デバッガでアタッチされているが、ハンドリングされなかった
            if (!(result <= nn::svc::ResultNotHandled()))
            {
                if (GetCurrentProcess().EnterJitDebug(nn::svc::DebugEvent_Exception, except, param2, param3))
                {
                    NN_KERN_KTRACE_DELAYED_DUMP_WITH_ADDITIONAL_INFO();
                    nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(pContext));
                }
            }

            if (resultValue <= nn::svc::ResultDebug())
            {
                nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(pContext));
            }
        }
    }
    else
    {
        Bit32 esr;
        if (process.Is64Bit())
        {
            pContext->pc -= 4;
            esr = ((HW_ESR_EC_SVC64 << 26) | NN_SVC_ID_RETURN_FROM_EXCEPTION);
        }
        else
        {
            if (pContext->psr & HW_PSR_THUMB_STATE)
            {
                pContext->pc -= 2;
            }
            else
            {
                pContext->pc -= 4;
            }
            esr = ((HW_ESR_EC_SVC32 << 26) | NN_SVC_ID_RETURN_FROM_EXCEPTION);
        }

        Bit32 far = pContext->pc;
        HandleUserModeException(pContext, esr, far, 0, 0, 0);
        nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(pContext));
    }

    NN_KERN_FATAL_LOG("Exception occurred. %016lx\n", GetCurrentProcess().GetProgramId());
    NN_KERN_KTRACE_DUMP_WITH_ADDITIONAL_INFO();
    GetCurrentProcess().Exit();
}

}}}

