﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Platform.h"
#include "kern_KPageTableBody.h"

namespace nn { namespace kern { namespace ARMv7A {
namespace
{
enum L1EntryType
{
    L1_ENTRY_TYPE_FAULT,
    L1_ENTRY_TYPE_COARSE,
    L1_ENTRY_TYPE_SECTION,
    L1_ENTRY_TYPE_RESERVED
};
enum L2EntryType
{
    L2_ENTRY_TYPE_FAULT,
    L2_ENTRY_TYPE_LARGE,
    L2_ENTRY_TYPE_SMALL_EXECUTABLE,
    L2_ENTRY_TYPE_SMALL
};

struct BitTraits
{
    int32_t pos;
    int32_t length;
};

struct Pte32
{
    Bit32 value;

    explicit Pte32(Bit32 x) : value(x) {}
    Bit32 GetBits(int32_t pos, int32_t length) const { return (value >> pos) & (~0u >> (32 - length)); }
    Bit32 ExtractBits(int32_t pos, int32_t length) const { return value & ( (~0u >> (32 - length)) << pos ); }
};

struct VirtualAddress : public Pte32
{
    explicit VirtualAddress(uintptr_t address) : Pte32(address) {}
    uintptr_t GetSmallPageOffset()   const { return ExtractBits( 0, 12); }
    uintptr_t GetLargePageOffset()   const { return ExtractBits( 0, 16); }
    uintptr_t GetSectionOffset()     const { return ExtractBits( 0, 20); }
    uintptr_t GetSuperSectionOffset()const { return ExtractBits( 0, 24); }
    size_t GetL2Index()              const { return GetBits    (12,  8); }
    size_t GetL1Index()              const { return GetBits    (20, 12); }
};
struct L1Entry : public Pte32
{
    explicit L1Entry(Bit32 entry) : Pte32(entry) {}
    Bit32 GetTypeImpl()         const { return GetBits    ( 0,  2); }
    uintptr_t  GetCoarseBase()       const { return ExtractBits(10, 22); }
    bool  IsSuperSection()      const { return GetBits    (18,  1) != 0; }
    uintptr_t  GetSectionBase()      const { return ExtractBits(20, 12); }
    uintptr_t  GetSuperSectionBase() const { return ExtractBits(24,  8); }

    L1EntryType GetType()       const { return static_cast<L1EntryType>(GetTypeImpl()); }
};
struct L2Entry : public Pte32
{
    explicit L2Entry(Bit32 entry) : Pte32(entry) {}
    Bit32 GetTypeImpl()         const { return GetBits    ( 0,  2); }
    uintptr_t  GetSmallPageBase()    const { return ExtractBits(12, 20); }
    uintptr_t  GetLargePageBase()    const { return ExtractBits(16, 16); }

    L2EntryType GetType()       const { return static_cast<L2EntryType>(GetTypeImpl()); }
};

const Bit32 UNMAPPED_ENTRY = 0;  // ALL 0

#if defined NN_KERN_ENABLE_DUMP_PAGETABLE
void DumpTableL2(const Bit32* pTable, uintptr_t baseAddress);
void DumpTable(const Bit32* pTable, int32_t begin, int32_t end);

void DumpTableL2(const Bit32* pTable, uintptr_t baseAddress)
{
    static const char* TYPE_NAMES[] = {"xx", "lg", "sm", "sm"};
    Bit32 prevType = 1;
    int32_t type0Begin = 0;

    NN_KERN_RELEASE_LOG("    indx addr     value    tp addr     cbtex xap S G X\n");

    for( int i = 0; i < 256; ++i )
    {
        const Bit32 entry = pTable[i];
        const Bit32 type   = ((entry & 0x3)  >> 0);
        const Bit32 b   = ((entry & 0x1) >> 2);
        const Bit32 c   = ((entry & 0x1) >> 3);
        const Bit32 ap  = ((entry & 0x3) >> 4);
        const Bit32 apx = ((entry & 0x1) >> 9);
        const Bit32 s   = ((entry & 0x1) >> 10);
        const Bit32 nG  = ((entry & 0x1) >> 11);

        if( type == 0 )
        {
            if( prevType != 0 )
            {
                type0Begin = i;
            }
            prevType = 0;
        }
        else
        {
            if( prevType == 0 )
            {
                NN_KERN_RELEASE_LOG("        -- %08x-%08x no entry\n", baseAddress + NN_KERN_FINEST_PAGE_SIZE * type0Begin,
                        baseAddress + NN_KERN_FINEST_PAGE_SIZE * i );
            }
            prevType = type;

            NN_KERN_RELEASE_LOG("    %4d %08x %08x %2s ", i, baseAddress + NN_KERN_FINEST_PAGE_SIZE * i, entry, TYPE_NAMES[type]);

            switch( type )
            {
            case 1:
                {
                    const Bit32 tex  = ((entry & 0x7) >> 12);
                    const Bit32 xn   = ((entry & 0x1) >> 15);
                    const Bit32 addr = ((entry & 0xFFFF0000) >> 0);
                    NN_KERN_RELEASE_LOG("%08x %d%d%3d %d %2d\n", addr, c, b, tex, apx, ap, s, nG, xn);
                }
                break;

            case 2:
            case 3:
                {
                    const Bit32 tex  = ((entry & 0x7) >> 6);
                    const Bit32 xn   = ((entry & 0x1) >> 0);
                    const Bit32 addr = ((entry & 0xFFFFF000) >> 0);
                    NN_KERN_RELEASE_LOG("%08x %d%d%3d %d %2d\n", addr, c, b, tex, apx, ap, s, nG, xn);
                }
                break;
            default:
                {
                    NN_KERN_ASSERT(0);
                }
                break;
            }
        }
    }

    if( prevType == 0 )
    {
        NN_KERN_RELEASE_LOG("        -- %08x-%08x no entry\n", baseAddress + NN_KERN_FINEST_PAGE_SIZE * type0Begin,
                baseAddress + NN_KERN_FINEST_PAGE_SIZE * 256 );
    }
}

void DumpTable(const Bit32* pTable, int32_t begin, int32_t end)
{
    static const char* TYPE_NAMES[] = {"xx", "co", "sc", "rs"};
    Bit32 prevType = 1;
    int32_t type0Begin = begin;

    NN_KERN_RELEASE_LOG("indx addr     value    tp dm addr     cbtex ap\n");

    for( int i = begin; i < end; ++i )
    {
        const Bit32 entry = pTable[i];
        const Bit32 type   = ((entry & 0x3)  >> 0);
        const Bit32 domain = ((entry & 0x1E) >> 5);

        if( (type == 0) || (type == 3) )
        {
            if( prevType != 0 )
            {
                type0Begin = i;
            }
            prevType = 0;
        }
        else
        {
            if( prevType == 0 )
            {
                NN_KERN_RELEASE_LOG("    -- %08x-%08x no entry\n", type0Begin * 0x100000, i * 0x100000);
            }
            prevType = type;

            NN_KERN_RELEASE_LOG("%4d %08x %08x %2s ", i, 1024 * 1024 * i, entry, TYPE_NAMES[type]);

            if( type == 1 )
            {
                const uintptr_t addr   = ((entry & 0xFFFFFC00) >> 0);
                NN_KERN_RELEASE_LOG("%2d %08x\n", domain, addr);
                if( (NN_KERN_P_ADDR_L2_PAGE_TABLE <= addr) && (addr < NN_KERN_P_ADDR_L2_PAGE_TABLE_END) )
                {
                    DumpTableL2(reinterpret_cast<const Bit32*>(addr + (NN_KERN_V_ADDR_L2_PAGE_TABLE - NN_KERN_P_ADDR_L2_PAGE_TABLE)), i * 0x100000);
                }
                else
                {
                    DumpTableL2(GetTypedPointer<const Bit32>(KMemoryLayout::ToLinearVirtualAddress(addr)), i * 0x100000);
                }
            }
            else    // type == 3
            {
                const Bit32 c    = ((entry & 0x1) >> 3);
                const Bit32 b    = ((entry & 0x1) >> 2);
                const Bit32 tex  = ((entry & 0x7) >> 12);
                const Bit32 ap   = ((entry & 0x3) >> 10);
                const Bit32 addr = ((entry & 0xFFF00000) >> 0);
                NN_KERN_RELEASE_LOG("%2d %08x %d%d%3d %2d\n", domain, addr, c, b, tex, ap);
            }
        }
    }

    if( prevType == 0 )
    {
        NN_KERN_RELEASE_LOG("-- %08x-%08x no entry\n", type0Begin * 0x100000, end * 0x100000);
    }
}
#endif
}

void KPageTableBody::InitializeForProcess(void* pTable, KProcessAddress begin, KProcessAddress end)
{
    m_pTable = static_cast<Bit32*>(pTable);
    m_MinIndex = VirtualAddress(GetAsInteger(begin)).GetL1Index();
    m_MaxIndex = VirtualAddress(GetAsInteger(end - 1)).GetL1Index();

    // 全てのエントリを非グローバル・アクセス不能に
    for (size_t i = m_MinIndex; i <= m_MaxIndex; i++)
    {
        m_pTable[i] = UNMAPPED_ENTRY;
    }

}

void KPageTableBody::InitializeForKernel(void* pTable, KProcessAddress begin, KProcessAddress end)
{
    m_pTable = static_cast<Bit32*>(pTable);
    m_MinIndex = VirtualAddress(GetAsInteger(begin)).GetL1Index();
    m_MaxIndex = VirtualAddress(GetAsInteger(end - 1)).GetL1Index();
}

Bit32* KPageTableBody::Finalize()
{
    return m_pTable;
}

void KPageTableBody::Dump(uintptr_t begin, size_t size) const
{
    NN_UNUSED(begin);
    NN_UNUSED(size);
#if defined NN_KERN_ENABLE_DUMP_PAGETABLE
    DumpTable(m_pTable, VirtualAddress(begin).GetL1Index(), VirtualAddress(begin + size - 1).GetL1Index());
#endif
}

bool KPageTableBody::GetAddrFromL1Entry(
        size_t*             pNumPages,
        KPhysicalAddress*   pOut,       ///< 取得したアドレスを格納する
        const Bit32**    ppL2,       ///< 取得した L2 エントリへのポインタを格納する
        Bit32            entryValue, ///< 対象の L1 エントリ値
        KProcessAddress     srcAddress  ///< 物理アドレスを取得する仮想アドレス
        )
{
    const L1Entry entry(entryValue);
    const VirtualAddress srcAddr(GetAsInteger(srcAddress));
    const L1EntryType largeType = entry.GetType();

    switch( largeType )
    {
    case L1_ENTRY_TYPE_COARSE:
        {
            const uintptr_t coarseBase                   = entry.GetCoarseBase();
            const KVirtualAddress coarseBaseAddr    = GetPageTableVirtualAddress(coarseBase);
            const Bit32* pCoarseTable               = GetTypedPointer<Bit32>(coarseBaseAddr);

            const size_t coarseIndex = srcAddr.GetL2Index();

            *ppL2 = &pCoarseTable[coarseIndex];

            return GetAddrFromL2Entry(pNumPages, pOut, **ppL2, srcAddress);
        }

    case L1_ENTRY_TYPE_SECTION:
        {
            const Bit32 isSuper = entry.IsSuperSection();
            *ppL2 = NULL;

            if( isSuper )
            {
                // super section
                const Bit32 paBase  = entry.GetSuperSectionBase();
                const Bit32 offset  = srcAddr.GetSuperSectionOffset();
                *pOut = (paBase | offset);
                *pNumPages = 4096;
            }
            else
            {
                // section
                const Bit32 paBase  = entry.GetSectionBase();
                const Bit32 offset  = srcAddr.GetSectionOffset();
                *pOut = (paBase | offset);
                *pNumPages = 256;
            }
            return true;
        }

    default:
        {
            *ppL2   = NULL;
            *pNumPages = 256;
        }
        return false;
    }
}

bool KPageTableBody::GetAddrFromL2Entry(size_t* pNumPages, KPhysicalAddress* pOut, Bit32 entryValue, KProcessAddress srcAddress)
{
    const L2Entry entry(entryValue);
    const VirtualAddress srcAddr(GetAsInteger(srcAddress));
    const L2EntryType type       = entry.GetType();

    if( type == L2_ENTRY_TYPE_LARGE )
    {
        const Bit32 paBase  = entry.GetLargePageBase();
        const Bit32 offset  = srcAddr.GetLargePageOffset();
        *pOut = (paBase | offset);
        *pNumPages = 16;
        return true;
    }
    else if( type >= L2_ENTRY_TYPE_SMALL_EXECUTABLE )
    {
        const Bit32 paBase  = entry.GetSmallPageBase();
        const Bit32 offset  = srcAddr.GetSmallPageOffset();
        *pOut = (paBase | offset);
        *pNumPages = 1;
        return true;
    }

    *pNumPages = 1;
    return false;
}


bool KPageTableBody::TraverseBegin(TraverseData* pOut, TraverseContext* pContext, KProcessAddress vaddr) const
{
    const VirtualAddress addr(GetAsInteger(vaddr));
    const size_t largeIndex = addr.GetL1Index();
    bool isValid;

    if (largeIndex < m_MinIndex || m_MaxIndex < largeIndex)
    {
        pContext->pL1 = &m_pTable[m_MaxIndex + 1];
        pContext->pL2 = NULL;
        pOut->address = 0;
        pOut->size = 256 * NN_KERN_FINEST_PAGE_SIZE;
        return false;
    }

    const Bit32* pL1 = &m_pTable[largeIndex];
    pContext->pL1 = &m_pTable[largeIndex];

    size_t numPages;
    isValid = GetAddrFromL1Entry(&numPages, &pOut->address, &pContext->pL2, *pL1, vaddr);
    pOut->size = numPages * NN_KERN_FINEST_PAGE_SIZE;

    switch (numPages)
    {
    case 4096:
        {
            pL1 += 16 - addr.GetSuperSectionOffset() / 0x00100000;
        }
        break;
    case 256:
        {
            pL1 += 1;
        }
        break;
    case 16:
        {
            pL1 += 1;
            pContext->pL2 += 16 - addr.GetLargePageOffset() / 0x00001000;
        }
        break;
    case 1:
        {
            pL1 += 1;
            pContext->pL2 += 1;
        }
        break;
    default:
        {
            NN_KERN_ASSERT(0);
        }
        break;
    }

    pContext->pL1 = pL1;
    return isValid;
}

bool KPageTableBody::TraverseNext(TraverseData* pOut, TraverseContext* pContext) const
{
    bool isValid;
    size_t numPages;

    if( reinterpret_cast<uintptr_t>(pContext->pL2) % 0x400 == 0x000 )
    {
        const Bit32* pL1 = pContext->pL1;
        size_t largeIndex = pL1 - m_pTable;

        if (largeIndex < m_MinIndex || m_MaxIndex < largeIndex)
        {
            pContext->pL1 = &m_pTable[m_MaxIndex + 1];
            pContext->pL2 = NULL;
            pOut->address = 0;
            pOut->size = 256 * NN_KERN_FINEST_PAGE_SIZE;
            return false;
        }

        isValid = GetAddrFromL1Entry(&numPages, &pOut->address, &pContext->pL2, *pL1, 0);

        if( numPages >= 256 )
        {
            pL1 += numPages / 256;
        }
        else
        {
            pL1 += 1;
            pContext->pL2 += numPages;
        }

        pContext->pL1 = pL1;
    }
    else
    {
        isValid = GetAddrFromL2Entry(&numPages, &pOut->address, *pContext->pL2, 0);
        pContext->pL2 += numPages;
    }

    pOut->size = numPages * NN_KERN_FINEST_PAGE_SIZE;
    return isValid;
}


bool KPageTableBody::GetPhysicalAddress(KPhysicalAddress* pOut, KProcessAddress vaddr) const
{
    const VirtualAddress srcAddr(GetAsInteger(vaddr));
    const L1Entry largeEntry(m_pTable[srcAddr.GetL1Index()]);
    const L1EntryType largeType = largeEntry.GetType();

    if( largeType == L1_ENTRY_TYPE_COARSE )
    {
        const uintptr_t coarseBase                   = largeEntry.GetCoarseBase();
        const KVirtualAddress coarseBaseAddr    = GetPageTableVirtualAddress(coarseBase);
        const Bit32* pCoarseTable               = GetTypedPointer<Bit32>(coarseBaseAddr);

        const size_t coarseIndex = srcAddr.GetL2Index();
        const L2Entry coarseEntry(pCoarseTable[coarseIndex]);
        const L2EntryType coarseType = coarseEntry.GetType();

        if( coarseType == L2_ENTRY_TYPE_LARGE )
        {
            const Bit32 paBase  = coarseEntry.GetLargePageBase();
            const Bit32 offset  = srcAddr.GetLargePageOffset();
            *pOut = (paBase | offset);
            return true;
        }
        else if( coarseType >= L2_ENTRY_TYPE_SMALL_EXECUTABLE )
        {
            const Bit32 paBase  = coarseEntry.GetSmallPageBase();
            const Bit32 offset  = srcAddr.GetSmallPageOffset();
            *pOut = (paBase | offset);
            return true;
        }
    }
    else if( largeType == L1_ENTRY_TYPE_SECTION )
    {
        const Bit32 isSuper = largeEntry.IsSuperSection();

        if( isSuper )
        {
            // super section
            const Bit32 paBase  = largeEntry.GetSuperSectionBase();
            const Bit32 offset  = srcAddr.GetSuperSectionOffset();
            *pOut = (paBase | offset);
            return true;
        }
        else
        {
            // section
            const Bit32 paBase  = largeEntry.GetSectionBase();
            const Bit32 offset  = srcAddr.GetSectionOffset();
            *pOut = (paBase | offset);
            return true;
        }
    }

    return false;
}

} } }

