﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "../../kern_Platform.h"
#include "kern_KInterruptManager.h"
#include "../../kern_InterruptControllerSelect.h"
#include "../../kern_Kernel.h"
#include "../../kern_KScheduler.h"
#include "../../kern_KInterruptTaskManager.h"
#include "../../kern_KTrace.h"

// TORIAEZU
namespace nn { namespace kern {
    namespace ARMv7A {

KSimpleLock                          KInterruptManager::s_Lock;
KInterruptManager::GlobalInterruptTableEntry KInterruptManager::s_GlobalInterruptTable[KInterruptController::NumGlobalInterrupts];
KInterruptController::GlobalState    KInterruptManager::s_GlobalState;
bool                                 KInterruptManager::s_GlobalSaved;

/*===========================================================================*
 * public メンバ関数
 *===========================================================================*/

void KInterruptManager::Initialize(int coreNo)
{
    for (size_t i = 0; i < sizeof(m_CoreLocalInterruptTable) / sizeof(*m_CoreLocalInterruptTable); ++i)
    {
        m_CoreLocalInterruptTable[i].pHandler        = nullptr;
        m_CoreLocalInterruptTable[i].isManualClear   = false;
        m_CoreLocalInterruptTable[i].isWaitingClear  = false;
        m_CoreLocalInterruptTable[i].priority        = KInterruptController::PriorityLevel_Low;
    }
    if (coreNo == 0)
    {
        for (size_t i = 0; i < sizeof(s_GlobalInterruptTable) / sizeof(*s_GlobalInterruptTable); ++i)
        {
            s_GlobalInterruptTable[i].pHandler          = nullptr;
            s_GlobalInterruptTable[i].isManualClear     = false;
            s_GlobalInterruptTable[i].isWaitingClear    = false;
        }
    }
    m_InterruptController.Initialize(coreNo);
}

void KInterruptManager::Finalize(int coreNo)
{
    m_InterruptController.Finalize(coreNo);
}
void KInterruptManager::Save(int coreNo)
{
    KCPU::SynchronizeAllCore();
    if (coreNo == 0)
    {
        NN_KERN_ABORT_UNLESS(!s_GlobalSaved);
        m_InterruptController.DisableGlobalAll(&s_GlobalState);
        s_GlobalSaved = true;
    }
    KCPU::SynchronizeAllCore();
    NN_KERN_ABORT_UNLESS(!m_LocalSaved);
    m_InterruptController.DisableCoreLocalAll(&m_LocalState);
    m_LocalSaved = true;
    KCPU::SynchronizeAllCore();
    if (coreNo != 0)
    {
        m_InterruptController.Finalize(coreNo);
    }
    KCPU::SynchronizeAllCore();
    if (coreNo == 0)
    {
        m_InterruptController.Finalize(coreNo);
    }
}
void KInterruptManager::Restore(int coreNo)
{
    KCPU::SynchronizeAllCore();
    if (coreNo == 0)
    {
        m_InterruptController.Initialize(coreNo);
    }
    KCPU::SynchronizeAllCore();
    if (coreNo != 0)
    {
        m_InterruptController.Initialize(coreNo);
    }
    KCPU::SynchronizeAllCore();
    m_InterruptController.RestoreCoreLocal(m_LocalState);
    m_LocalSaved = false;
    KCPU::SynchronizeAllCore();
    if (coreNo == 0)
    {
        m_InterruptController.RestoreGlobal(s_GlobalState);
        s_GlobalSaved = false;
    }
    KCPU::SynchronizeAllCore();
}

Result KInterruptManager::ClearInterrupt(int32_t irqNo, int32_t coreNo)
{
    if( KInterruptController::IsGlobal(irqNo) )
    {
        KDisableInterrupt di;
        KScopedSimpleLock lock(&s_Lock);

        return ClearGlobal(irqNo);
    }
    else if( KInterruptController::IsLocal(irqNo) )
    {
        KDisableInterrupt di;
        return ClearLocal(irqNo);
    }

    return nn::svc::ResultOutOfRange();
}

Result KInterruptManager::ClearInterrupt(int32_t irqNo)
{
    if( KInterruptController::IsGlobal(irqNo) )
    {
        KDisableInterrupt di;
        KScopedSimpleLock lock(&s_Lock);

        return ClearGlobal(irqNo);
    }

    return nn::svc::ResultOutOfRange();
}

Result KInterruptManager::BindHandler(KInterruptHandler* pHandler, int32_t irqNo, int32_t coreNo, int32_t priority, bool isManualClear, bool isLevel)
{
    if( KInterruptController::IsLocal(irqNo) )
    {
        KDisableInterrupt di;
        NN_KERN_ASSERT( coreNo == GetCurrentCpuNo() );
        return BindLocal(pHandler, irqNo, priority, isManualClear);
    }
    else if( KInterruptController::IsGlobal(irqNo) )
    {
        KDisableInterrupt di;
        KScopedSimpleLock lock(&s_Lock);

        return BindGlobal(pHandler, irqNo, coreNo, priority, isManualClear, isLevel);
    }

    return nn::svc::ResultOutOfRange();
}

Result KInterruptManager::UnbindHandler(int32_t irqNo, int32_t coreNo)
{
    if (KInterruptController::IsLocal(irqNo))
    {
        KDisableInterrupt di;
        NN_KERN_ASSERT(coreNo == GetCurrentCpuNo());
        return UnbindLocal(irqNo);
    }
    else if (KInterruptController::IsGlobal(irqNo))
    {
        KDisableInterrupt di;
        KScopedSimpleLock lock(&s_Lock);

        return UnbindGlobal(irqNo);
    }

    return nn::svc::ResultOutOfRange();
}

void KInterruptManager::InterruptRequestHandler(bool fromUserMode)
{
    // INFO: 割り込み禁止状態

    const bool needScheduling = Kernel::GetInterruptManager().OnInterruptRequest();

    if (needScheduling)
    {
        // スケジューラ呼び出しは、プリエンプションロックされているなら遅延されるべきなので、
        // RequestScheduling() がふさわしい。
        // TODO: この周辺の条件分岐は必ず一方向にしかいかないことがあるはずなので、
        //       最適化の余地がある。
        GetCoreScheduler().RequestSchedulingOnInterrupt();
    }

    if( fromUserMode )
    {
        KThread& currentThread = GetCurrentThread();

        if( currentThread.IsTerminateRequested() )
        {
            KEnableInterrupt interruptEnabled;
            currentThread.Exit();
        }
    }
}





/*===========================================================================*
 * private メンバ関数
 *===========================================================================*/

bool KInterruptManager::OnInterruptRequest()
{
    // INFO: 割り込み禁止状態

    Bit32   irq;
    int32_t interruptNo;

    {
        irq = m_InterruptController.GetIrq();
        interruptNo     = KInterruptController::IrqToInterruptNo(irq);

        NN_KERN_KTRACE_IRQ(interruptNo);

        // スプリアス割り込みなら何もしない
        if (interruptNo < 0)
        {
            return false;
        }
    }


    KInterruptTask* pTask = nullptr;

    if (KInterruptController::IsLocal(interruptNo))
    {
        const int tableIndex = interruptNo - KInterruptController::LocalInterruptsMin;
        CoreLocalInterruptTableEntry& entry = m_CoreLocalInterruptTable[tableIndex];

#ifdef NN_KERN_ENABLE_INTERRUPT_PROFILE
        if( KInterruptController::IsSoftwareInterrupt(interruptNo) )
        {
            Kernel::IncrementNumSoftwareInterrupts();
        }
        else
        {
            Kernel::IncrementNumHardwareInterrupts();
        }
#endif

        if (entry.pHandler != nullptr)
        {
            if (entry.isManualClear)
            {
                m_InterruptController.SetPriority(interruptNo, KInterruptController::PriorityLevel_Low);
                entry.isWaitingClear = true;
            }

            pTask = entry.pHandler->OnInterrupt(interruptNo);
        }
        else
        {
            NN_WARNING(false, "No exception handler matches. %d", interruptNo);
        }
    }
    else if (KInterruptController::IsGlobal(interruptNo))
    {
        // 割り込み禁止で呼ばれる
        KScopedSimpleLock lock(&s_Lock);
        const int tableIndex = interruptNo - KInterruptController::GlobalInterruptsMin;
        GlobalInterruptTableEntry& entry = s_GlobalInterruptTable[tableIndex];

#ifdef NN_KERN_ENABLE_INTERRUPT_PROFILE
        Kernel::IncrementNumHardwareInterrupts();
#endif

        if (entry.pHandler != nullptr)
        {
            if (entry.isManualClear)
            {
                m_InterruptController.Disable(interruptNo);
                entry.isWaitingClear = true;
            }

            pTask = entry.pHandler->OnInterrupt(interruptNo);
        }
        else
        {
            NN_WARNING(false, "No exception handler matches. (source=%d)", interruptNo);
        }
    }
    else
    {
        NN_WARNING(false, "invalid innterruptNo(=%d)", interruptNo);
    }

    m_InterruptController.EndOfInterrupt(irq);

    if (pTask != nullptr)
    {
        // ダミータスクの場合は、スケジューリングだけ行う
        if (pTask != KScheduler::GetSchedulingDummyTask())
        {
            GetCoreInterruptTaskManager().EnqueueTask(pTask);
        }

        return true;
    }
    else
    {
        return false;
    }
}

Result KInterruptManager::BindGlobal(KInterruptHandler* pHandler, int32_t irqNo, int32_t coreNo, int32_t priority, bool isManualClear, bool isLevel)
{
    NN_KERN_ASSERT(KInterruptController::IsGlobal(irqNo));

    const int tableIndex = irqNo - KInterruptController::GlobalInterruptsMin;
    GlobalInterruptTableEntry& entry = s_GlobalInterruptTable[tableIndex];

    if ((priority < 0) || (KInterruptController::PriorityLevel_Low <= priority))
    {
        return nn::svc::ResultOutOfRange();
    }
    if (entry.pHandler != nullptr)
    {
        return nn::svc::ResultBusy();
    }

    entry.isWaitingClear    = false;
    entry.isManualClear     = isManualClear;
    entry.pHandler          = pHandler;

    if (isLevel)
    {
        m_InterruptController.SetLevel(irqNo);
    }
    else
    {
        m_InterruptController.SetEdge(irqNo);
    }
    m_InterruptController.Clear(irqNo);
    m_InterruptController.SetTarget(coreNo, irqNo);
    m_InterruptController.SetPriority(irqNo, priority);
    m_InterruptController.Enable(irqNo);

    return ResultSuccess();
}

Result KInterruptManager::UnbindGlobal(int32_t irqNo)
{
    NN_KERN_ASSERT(KInterruptController::IsGlobal(irqNo));
    const int tableIndex = irqNo - KInterruptController::GlobalInterruptsMin;
    GlobalInterruptTableEntry& entry = s_GlobalInterruptTable[tableIndex];

    // NULL ならばチェックをパスする
    for (int coreNo = 0; coreNo < KCPU::NUM_CORE; coreNo++)
    {
        m_InterruptController.ClearTarget(coreNo, irqNo);
    }
    m_InterruptController.SetPriority(irqNo, KInterruptController::PriorityLevel_Low);
    m_InterruptController.Disable(irqNo);
    entry.pHandler = nullptr;

    return ResultSuccess();
}

Result KInterruptManager::BindLocal(KInterruptHandler* pHandler, int32_t irqNo, int32_t priority, bool isManualClear)
{
    NN_KERN_ASSERT(KInterruptController::IsLocal(irqNo));

    const int tableIndex = irqNo - KInterruptController::LocalInterruptsMin;
    CoreLocalInterruptTableEntry& entry = m_CoreLocalInterruptTable[tableIndex];

    if ((priority < 0) || (KInterruptController::PriorityLevel_Low <= priority))
    {
        return nn::svc::ResultOutOfRange();
    }
    if (entry.pHandler != nullptr)
    {
        return nn::svc::ResultBusy();
    }

    entry.isWaitingClear    = false;
    entry.isManualClear     = isManualClear;
    entry.pHandler          = pHandler;
    entry.priority          = static_cast<int8_t>(priority);

    m_InterruptController.Clear(irqNo);
    m_InterruptController.SetPriority(irqNo, priority);
    m_InterruptController.Enable(irqNo);

    return ResultSuccess();
}

Result KInterruptManager::UnbindLocal(int32_t irqNo)
{
    NN_KERN_ASSERT(KInterruptController::IsLocal(irqNo));

    const int tableIndex = irqNo - KInterruptController::LocalInterruptsMin;
    CoreLocalInterruptTableEntry& entry = m_CoreLocalInterruptTable[tableIndex];

    if (entry.pHandler == nullptr)
    {
        return nn::svc::ResultInvalidState();
    }

    m_InterruptController.SetPriority(irqNo, KInterruptController::PriorityLevel_Low);
    m_InterruptController.Disable(irqNo);

    entry.pHandler = nullptr;

    return ResultSuccess();
}

Result KInterruptManager::ClearGlobal(int32_t irqNo)
{
    const int tableIndex = irqNo - KInterruptController::GlobalInterruptsMin;
    GlobalInterruptTableEntry& entry = s_GlobalInterruptTable[tableIndex];

    if (entry.pHandler == nullptr)
    {
        return nn::svc::ResultInvalidState();
    }
    if (!entry.isManualClear)
    {
        return ResultSuccess();
    }
    if (!entry.isWaitingClear)
    {
        return ResultSuccess();
    }

    entry.isWaitingClear = false;
    m_InterruptController.Enable(irqNo);

    return ResultSuccess();
}

Result  KInterruptManager::ClearLocal(int32_t irqNo)
{
    const int tableIndex = irqNo - KInterruptController::LocalInterruptsMin;
    CoreLocalInterruptTableEntry& entry = m_CoreLocalInterruptTable[tableIndex];

    if (entry.pHandler == nullptr)
    {
        return nn::svc::ResultInvalidState();
    }
    if (!entry.isManualClear)
    {
        return ResultSuccess();
    }
    if (!entry.isWaitingClear)
    {
        return ResultSuccess();
    }

    entry.isWaitingClear = false;
    m_InterruptController.SetPriority(irqNo, entry.priority);

    return ResultSuccess();
}



    }
}}
