﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_AssemblyOffset.h"
#include "../ARM/kern_Assembly.h"

// IRQ 割り込みの動作
//
// まず割り込みがユーザモードで発生したのか特権モードで発生したのかを調べる
//
// ユーザモードの場合
//   SVC モードへ移行しユーザモードのコンテキストで必要なものを
//   SVC モードのスタックへ保存する。
//   さらに SVC モードのコンテキストをスタックに保存して
//   KInterruptManager::InterruptRequestHandler に渡す。
//
//   コンテキスト復帰後はさらにユーザモードのコンテキストを復帰して
//   割り込み発生元へ戻る
//
// 特権モードの場合
//   IRQ モードのスタック位置をリセットし、IRQ モードのスタックに
//   割り込み発生時のコンテキストを保存し、
//   KInterruptManager::InterruptRequestHandler に渡す。
//
//   コンテキスト復帰後は直接割り込み発生元へ戻る。
//
EXP_VECTOR_ENTRY(_ZN2nn4kern3ARM23InterruptRequestHandlerEv)
    // IRQ モード -------

    sub     lr, lr, #4          // 復帰先は LR - 4

    // 割り込み発生時の状況を調べる
    mov     sp, r0                              // r0 を sp へ保存
    mrs     r0, spsr;
    and     r0, r0, #HW_PSR_CPU_MODE_MASK
    cmp     r0, #HW_PSR_USR_MODE
    mov     r0, sp                              // r0 を sp から復帰
    bne     privilege_mode_sequence


// ユーザモードだった場合
user_mode_sequence:
    // SVC モードへ移行しつつ復帰用レジスタを保存する
    srsdb   #HW_PSR_SVC_MODE!                          // ユーザモードの PC と CPSR を保存
    cps     #HW_PSR_SVC_MODE

    // SVC モード -------

    // ユーザモードレジスタを保存
    push    {ip, lr}
    mrc     p15, 0, ip, c13, c0, 2
    push    {ip, lr}
    stmdb   sp, {r13-r14}^              // ユーザモードの SP、LR を保存
    sub     sp, sp, #8

    // 揮発レジスタをを保存
    push    {r0-r11}

    // 割り込みメイン処理関数へ
    mov     r0, #1          // fromUserMode = true
    ldr     lr, =_ZN2nn4kern6ARMv7A17KInterruptManager23InterruptRequestHandlerEb
    blx     lr

    // 揮発レジスタを復帰
    // ユーザモードレジスタを復帰
    pop     {r0-r11}
    ldmia   sp, {r13-r14}^
    add     sp, sp, #8
    pop     {ip, lr}
    mcr     p15, 0, ip, c13, c0, 2
    pop     {ip, lr}

    // 割り込み元へ返る
    rfeia   sp!                 // ユーザモードの PC と CPSR を復帰


// 特権モードだった場合
privilege_mode_sequence:
    // IRQ モード -------

    // SVC モードのスタック残量をチェックする
    ldr     sp, = NN_KERN_V_ADDR_STACK_CMN_END
    push    {r0-r1}
    cps     #HW_PSR_SVC_MODE
    // SVC モード -----------
    mov     r0, sp
    cps     #HW_PSR_IRQ_MODE
    // IRQ モード -----------
    sub     r0, r0, #1  // r0 が SVC スタックの BOTTOM のケースを考慮して、1引いておく
    mov     r1, r0, LSR #NN_KERN_THREAD_SVC_STACK_SIZE_SHIFT // 2 命令で and r1, r0, #0xFFFFF000 を行う
    mov     r1, r1, LSL #NN_KERN_THREAD_SVC_STACK_SIZE_SHIFT // r1 が SVC スタックの TOP になる
    sub     r1, r0, r1  // r1 が SVC スタックの残量になる

    // 残量が少ないなら KernelPanic
    subs    r1, r1, #0x0300
    blt     kernel_panic

    // 残量があるなら処理継続

    // 割り込み発生時のコンテキストを保存
    pop     {r0-r1}
    srsdb   #HW_PSR_SVC_MODE!                  // 元のモードの PC と CPSR を SVC スタックに保存
    cps     #HW_PSR_SVC_MODE

    // SVC モード -----------
    push    {r0-r12,r14}

    //    この時点でのスタックの状況
    //
    //    sp+ 0  r0
    //      + 4  r1
    //      + 8  r2
    //      +12  r3
    //      +16  r12
    //      +20  SVC モードの LR
    //      +20  SVC モードの PC
    //      +24  SVC モードの CPSR
    //      +28                        -- 呼び出し時のスタック位置

    // スタックを 8 バイトアライメントする
    mov     r12, sp
    bic     sp, sp, #0x7
    push    {r12, lr}

    // 割り込みメイン処理関数へ
    mov     r0, #0          // fromUserMode = false
    ldr     lr, =_ZN2nn4kern6ARMv7A17KInterruptManager23InterruptRequestHandlerEb
    blx     lr

    // アライメント分を戻す
    ldr     sp, [sp]

    // コンテキストを復帰
    pop     {r0-r12,r14}

    // 割り込み元へ返る
    rfeia   sp!                         // 元のモードの PC と CPSR を復帰

kernel_panic:
    ldr     lr, =_ZN2nn4kern5PanicEPKciS2_z
    adr     r0, file_name
    ldr     r1, =__LINE__
    adr     r2, panic_msg
    bx      lr
file_name: .string  __FILE__
panic_msg: .string  "svc stack is too small."
SET_SIZE(_ZN2nn4kern3ARM23InterruptRequestHandlerEv)

