﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Assert.h"
#include "../../kern_KTaggedAddress.h"

namespace nn { namespace kern {
namespace ARM {

#define PADDING(pos,dst)    Bit8 padding ## pos [(dst)-(pos)]
struct GicDistributer
{
    volatile Bit32 ctlr;
    volatile Bit32 typer;
    volatile Bit32 iidr;
    PADDING( 0x00C, 0x080 );

    // 0x080
    volatile Bit32 igroupr[32];
    volatile Bit32 isenabler[32];
    volatile Bit32 icenabler[32];
    volatile Bit32 ispendr[32];
    volatile Bit32 icpendr[32];
    volatile Bit32 isactiver[32];
    volatile Bit32 icactiver[32];

    // 0x400
    union
    {
        volatile Bit32 words[255];
        volatile Bit8  bytes[1020];

    } ipriorityr;
    PADDING( 0x7FC, 0x800 );

    // 0x800
    union
    {
        volatile Bit32 words[255];
        volatile Bit8  bytes[1020];

    } itargetsr;
    PADDING( 0xBFC, 0xC00 );

    // 0xC00
    volatile Bit32 icfgr[64];

    // 0xD00
    volatile Bit32 ppisr;
    volatile Bit32 spisr[31];
    PADDING( 0xD80, 0xE00 );

    // 0xE00
    volatile Bit32 nsacr[64];

    // 0xF00
    volatile Bit32 sgir;
    PADDING( 0xF04, 0xF10 );
    volatile Bit32 cpendsgir[4];
    volatile Bit32 spendsgir[4];
    PADDING( 0xF30, 0xFD0 );
    volatile Bit32 pidr4;
    volatile Bit32 pidr5;
    volatile Bit32 pidr6;
    volatile Bit32 pidr7;
    volatile Bit32 pidr0;
    volatile Bit32 pidr1;
    volatile Bit32 pidr2;
    volatile Bit32 pidr3;
    volatile Bit32 cidr0;
    volatile Bit32 cidr1;
    volatile Bit32 cidr2;
    volatile Bit32 cidr3;

    enum TargetListFilter
    {
        TargetListFilter_CpuTargetList  = 0u << 24,
        TargetListFilter_All            = 1u << 24,
        TargetListFilter_Self           = 2u << 24,
    };
    enum
    {
        CpuTargetListShift = 16,
    };
};


struct GicCpuInterface
{
    volatile Bit32 ctlr;
    volatile Bit32 pmr;
    volatile Bit32 bpr;
    volatile Bit32 iar;
    volatile Bit32 eoir;
    volatile Bit32 rpr;
    volatile Bit32 hppir;
    volatile Bit32 abpr;
    volatile Bit32 aiar;
    volatile Bit32 aeoir;
    volatile Bit32 ahppir;
    PADDING( 0x02C, 0x0D0 );

    volatile Bit32 apr0;
    PADDING( 0x0D4, 0x0E0 );
    volatile Bit32 nsapr0;
    PADDING( 0x0E4, 0x0FC );

    volatile Bit32 iidr;
    PADDING( 0x0100, 0x1000 );

    volatile Bit32 dir;
    PADDING( 0x1004, 0x2000 );
};

struct StaticAssert
{
    static_assert( sizeof(GicDistributer) == 0x1000, "" );
    static_assert( sizeof(GicCpuInterface) == 0x2000, "" );
};
#undef PADDING

class KInterruptController
{
public:
    Bit32 GetIrq() const
    {
        return m_GicCpuInterfaceBase->iar;
    }
    static int IrqToInterruptNo(Bit32 irq)
    {
        return (irq == SpuriousInterruptNo)? -1: (irq & 0x3ff);
    }
    void EndOfInterrupt(Bit32 irq) const
    {
        m_GicCpuInterfaceBase->eoir = irq;
    }
    void Disable(int interruptNo) const
    {
        m_GicDistributerBase->icenabler[interruptNo / NN_BITSIZEOF(Bit32)] = (1u << (interruptNo % NN_BITSIZEOF(Bit32)));
    }
    void Enable(int interruptNo) const
    {
        m_GicDistributerBase->isenabler[interruptNo / NN_BITSIZEOF(Bit32)] = (1u << (interruptNo % NN_BITSIZEOF(Bit32)));
    }
    void Clear(int interruptNo) const
    {
        m_GicDistributerBase->icpendr[interruptNo / NN_BITSIZEOF(Bit32)] = (1u << (interruptNo % NN_BITSIZEOF(Bit32)));
    }
    void SetTarget(int cpuNo, int interruptNo) const
    {
        m_GicDistributerBase->itargetsr.bytes[interruptNo] |= GetIrqDest(cpuNo);
    }
    void ClearTarget(int cpuNo, int interruptNo) const
    {
        m_GicDistributerBase->itargetsr.bytes[interruptNo] &= ~GetIrqDest(cpuNo);
    }
    void SetLevel(int interruptNo) const
    {
        Bit32 cfg = m_GicDistributerBase->icfgr[interruptNo / (NN_BITSIZEOF(Bit32) / 2)];
        cfg &= ~(0x3u << (2 * (interruptNo % (NN_BITSIZEOF(Bit32) / 2))));
        cfg |= (0x0u << (2 * (interruptNo % (NN_BITSIZEOF(Bit32) / 2))));
        m_GicDistributerBase->icfgr[interruptNo / (NN_BITSIZEOF(Bit32) / 2)] = cfg;
    }
    void SetEdge(int interruptNo) const
    {
        Bit32 cfg = m_GicDistributerBase->icfgr[interruptNo / (NN_BITSIZEOF(Bit32) / 2)];
        cfg &= ~(0x3u << (2 * (interruptNo % (NN_BITSIZEOF(Bit32) / 2))));
        cfg |= (0x2u << (2 * (interruptNo % (NN_BITSIZEOF(Bit32) / 2))));
        m_GicDistributerBase->icfgr[interruptNo / (NN_BITSIZEOF(Bit32) / 2)] = cfg;
    }
    void SetPriority(int interruptNo, int priority) const
    {
        NN_KERN_ASSERT(0 <= priority && priority < NumLevels);
        m_GicDistributerBase->ipriorityr.bytes[interruptNo] = ToGicPriority(priority);
    }
    int GetPriority(int interruptNo) const
    {
        return FromGicPriority(m_GicDistributerBase->ipriorityr.bytes[interruptNo]);
    }
    int GetCurrentPriority() const
    {
        return FromGicPriority(m_GicCpuInterfaceBase->rpr);
    }
    void SetPriorityLevel(int priority) const
    {
        NN_KERN_ASSERT(0 <= priority && priority < NumLevels);
        m_GicCpuInterfaceBase->pmr = ToGicPriority(priority);
    }
    static bool IsLocal(int interruptNo)
    {
        NN_KERN_ASSERT(0 <= interruptNo);
        return interruptNo < NumLocalInterrupts;
    }
    static bool IsGlobal(int interruptNo)
    {
        NN_KERN_ASSERT(interruptNo < NumInterrupts);
        return interruptNo >= GlobalInterruptsMin;
    }
    static bool IsSoftwareInterrupt(int interruptNo)
    {
        NN_KERN_ASSERT(0 <= interruptNo);
        return interruptNo < 16;
    }

    static KPhysicalAddress GetRegisterAddress(int index);
    static size_t GetRegisterSize(int index);

    void Initialize(int coreNo);
    void Finalize(int coreNo);


    void SendIpi(Bit64 cpuSet, int id) const
    {
        NN_KERN_ASSERT(0 < cpuSet && cpuSet < (1ull << 8));
        NN_KERN_ASSERT(0 <= id && id < 0x10);
        Bit64 mask = GicDistributer::TargetListFilter_CpuTargetList | id;
        while (cpuSet)
        {
            int cpuNo = __builtin_ctzll(cpuSet);
            mask |= (GetIrqDest(cpuNo) << GicDistributer::CpuTargetListShift);
            cpuSet &= ~(1ul << cpuNo);
        }
        m_GicDistributerBase->sgir = mask;
    }
    void SendIpi(int id) const
    {
        NN_KERN_ASSERT(0 <= id && id < 0x10);
        m_GicDistributerBase->sgir = GicDistributer::TargetListFilter_All | id;
    }

    bool IsDefinedInterrupt(int32_t sourceId) const
    {
        int numInterruptLines = ((m_GicDistributerBase->typer >> 0) & 0x1F) * 32 + 32;
        if (numInterruptLines > 1020)
        {
            numInterruptLines = 1020;
        }
        return (0 <= sourceId && sourceId < numInterruptLines);
    }

public:
    enum
    {
        NumMaps = 2,
        NumLocalInterrupts = 32,
        NumGlobalInterrupts = 988,
        NumInterrupts = NumLocalInterrupts + NumGlobalInterrupts,
        NumLevels = 4,
        LocalInterruptsMin = 0,
        GlobalInterruptsMin = LocalInterruptsMin + NumLocalInterrupts,
    };
    enum PriorityLevel
    {
        PriorityLevel_High = 0,
        PriorityLevel_Low = NumLevels - 1,

        PriorityLevel_DevInterrupt = 0,     // cache, dev
        PriorityLevel_TimInterrupt = 1,     // timer
        PriorityLevel_SchInterrupt = 2,     // sch
    };
    enum
    {
        SpuriousInterruptNo = 1023,
    };

public:
    struct LocalState
    {
        Bit32 isenabler[NumLocalInterrupts / 32];
        Bit32 ipriorityr[NumLocalInterrupts / 4];
        Bit32 itargetsr[NumLocalInterrupts / 4];
        Bit32 icfgr[NumLocalInterrupts / 16];
    };

    struct GlobalState
    {
        Bit32 isenabler[NumGlobalInterrupts / 32];
        Bit32 ipriorityr[NumGlobalInterrupts / 4];
        Bit32 itargetsr[NumGlobalInterrupts / 4];
        Bit32 icfgr[NumGlobalInterrupts / 16];
    };

public:
    static void PrepareInitialize(const KProcessAddress registerBase[NumMaps]);
    void DisableCoreLocalAll(LocalState* pBuffer) const;
    void DisableGlobalAll(GlobalState* pBuffer) const;
    void RestoreCoreLocal(const LocalState& state) const;
    void RestoreGlobal(const GlobalState& state) const;

private:
    static Bit8 ToGicPriority(int priority)
    {
         return (priority << (NN_BITSIZEOF(Bit8) - __builtin_ctz(NumLevels))) |  ((1u << (NN_BITSIZEOF(Bit8) - __builtin_ctz(NumLevels))) - 1);
    }
    static int FromGicPriority(Bit8 priority)
    {
         return (priority >> (NN_BITSIZEOF(Bit8) - __builtin_ctz(NumLevels))) & (NumLevels - 1);
    }

    static Bit32 GetIrqDest(int cpuNo)
    {
        return s_GicMask[cpuNo];
    }

    Bit32 GetOwnMask()
    {
        return m_GicDistributerBase->itargetsr.bytes[0];
    }
    GicDistributer*  m_GicDistributerBase;
    GicCpuInterface* m_GicCpuInterfaceBase;
    static GicDistributer*  s_GicDistributerBase;
    static GicCpuInterface* s_GicCpuInterfaceBase;
    static Bit32            s_GicMask[8];
};

}
}}

