﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   多重待ちに関する C 言語用 API の宣言
 */

#pragma once

#include <nnc/nn_Macro.h>
#include <nnc/nn_Common.h>
#include <nnc/os/os_Macro.h>
#include <nnc/os/os_MultipleWaitTypes.h>
#include <nnc/os/os_MultipleWaitCommon.h>
#include <nnc/os/detail/os_MacroImpl.h>

//--------------------------------------------------------------------------
//  C 向けの宣言
//--------------------------------------------------------------------------

//! @name 多重待ち関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   multiWait オブジェクトヘッダを初期化します。
 *
 * @details
 *  詳細は nn::os::InitializeMultiWait() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosInitializeMultiWait( nnosMultiWaitType* multiWait );


//--------------------------------------------------------------------------
/**
 * @brief   multiWait オブジェクトヘッダをファイナライズします。
 *
 * @details
 *  詳細は nn::os::FinalizeMultiWait() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosFinalizeMultiWait( nnosMultiWaitType* multiWait );



//--------------------------------------------------------------------------
/**
 * @brief   multiWait リスト内のいずれかの multiWaitHolder を待機します。
 *
 * @details
 *  詳細は nn::os::WaitAny() を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnosMultiWaitHolderType* nnosWaitAny( nnosMultiWaitType* multiWait );


//--------------------------------------------------------------------------
/**
 * @brief   multiWait リスト内のいずれかの multiWaitHolder をポーリングします。
 *
 * @details
 *  詳細は nn::os::TryWaitAny() を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnosMultiWaitHolderType* nnosTryWaitAny( nnosMultiWaitType* multiWait );


//--------------------------------------------------------------------------
/**
 * @brief   multiWait リスト内のいずれかの multiWaitHolder を待機します。
 *
 * @details
 *  詳細は nn::os::TimedWaitAny() を参照して下さい。@n
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C nnosMultiWaitHolderType* nnosTimedWaitAny( nnosMultiWaitType* multiWait, int64_t timeout );


//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder オブジェクトをファイナライズします。
 *
 * @details
 *  詳細は nn::os::FinalizeMultiWaitHolder() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosFinalizeMultiWaitHolder( nnosMultiWaitHolderType* multiWaitHolder );


//--------------------------------------------------------------------------
/**
 * @brief   multiWait ヘッダに多重待ちオブジェクト multiWaitHolder をリンクします。
 *
 * @details
 *  詳細は nn::os::LinkMultiWaitHolder() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosLinkMultiWaitHolder(nnosMultiWaitType* multiWait, nnosMultiWaitHolderType* multiWaitHolder);


//--------------------------------------------------------------------------
/**
 * @brief   multiWait ヘッダから、多重待ちオブジェクト multiWaitHolder をアンリンクします。
 *
 * @details
 *  詳細は nn::os::UnlinkMultiWaitHolder() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosUnlinkMultiWaitHolder(nnosMultiWaitHolderType* multiWaitHolder);


//--------------------------------------------------------------------------
/**
 * @brief   multiWait ヘッダにリンクされている全ての多重待ちオブジェクトホルダーをアンリンクします。
 *
 * @details
 *  詳細は nn::os::UnlinkAllMultiWaitHolder() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosUnlinkAllMultiWaitHolder(nnosMultiWaitType* multiWait);


//--------------------------------------------------------------------------
/**
 * @brief   srcMultiWait ヘッダにリンクされている全ての多重待ちオブジェクトホルダーをアンリンクし dstMultiWait にリンクします。
 *
 * @details
 *  詳細は nn::os::MoveAllMultiWaitHolder() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosMoveAllMultiWaitHolder(nnosMultiWaitType* dstMultiWait, nnosMultiWaitType* srcMultiWait);


//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder にユーザデータの値を設定します
 *
 * @details
 *  詳細は nn::os::SetMultiWaitHolderUserData() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void    nnosSetMultiWaitHolderUserData(nnosMultiWaitHolderType* multiWaitHolder, uintptr_t userData);


//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder に設定されているユーザデータ値を取得します
 *
 * @details
 *  詳細は nn::os::GetMultiWaitHolderUserData() を参照して下さい。
 *
 */
NN_OS_EXTERN_C uintptr_t nnosGetMultiWaitHolderUserData(const nnosMultiWaitHolderType* multiWaitHolder);

//! @}
