﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief         mbuf の操作に使用する API を宣言します。
 */

#pragma once

#include <nnc/mbuf/mbuf_Definitions.h>
#include <nnc/mbuf/mbuf_Macro.h>

//---------------------------------------------------------------------------
//  C 向けの宣言
//---------------------------------------------------------------------------

/**
 * @brief         mbuf チェインを確保します。
 *
 * @param[in]     orig         未実装 (nullptr を指定してください)
 * @param[in]     len          確保する mbuf チェインのバイト長
 * @param[in]     how          アロケーションする際にブロックを許容するか否か
 * @param[in]     type         mbuf チェインの用途を指定してください
 *
 * @retval        nullptr      mbuf チェインの確保に失敗しました。
 * @retval        nullptr 以外 確保された mbuf チェインです。
 *
 * @pre
 *                - orig == nullptr
 *
 * @details       @a len に指定されたサイズのデータを格納するのに十分なサイズの
 *                mbuf チェインを新しく確保します。
 *
 *                ただし、 @a how に @ref MbufAllocationMode_DontWait を指定した場合、
 *                十分な容量の mbuf チェインを確保できなければ nullptr を返します。
 *                @ref MbufAllocationMode_Wait を指定した場合は空きができるまで待機するため、
 *                nullptr を返すことはありません。
 *
 *                また、@a type は mbuf チェインの確保元となるプールの選択に使用します。
 *                mbuf ライブラリの初期化時に nn::mbuf::Initialize() に渡した Config の
 *                type メンバが一致する mbuf プールから割り当てを行います。
 *                該当の mbuf プールが複数存在する場合、
 *                先頭の mbuf プールから優先して割り当てを試行しますが、
 *                空きが無かった場合にのみ後続の mbuf プールが使用されます。
 */
NN_MBUF_EXTERN_C nnmbufMbuf* nnmbufMbufGetm(
    nnmbufMbuf* orig, size_t len,
    nnmbufMbufAllocationMode how, int type);

/**
 * @brief         mbuf チェインを解放します。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_EXTERN_C void nnmbufMbufFreem(nnmbufMbuf* pMbuf);

/**
 * @brief         mbuf チェインを先頭あるいは末尾をトリムしてサイズを変更します。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     len          トリムするバイト長
 *
 * @return        処理の成否です。
 *
 * @pre
 *                - pMbuf != nullptr
 *
 * @details       @a len が正の場合には先頭から、負の場合には末尾からトリムします。
 */
NN_MBUF_EXTERN_C bool nnmbufMbufAdj(nnmbufMbuf* pMbuf, int len);

/**
 * @brief         mbuf チェインの末尾にデータを追加します。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     len          追加するデータのバイト長
 * @param[in]     cp           追加するデータ
 *
 * @return        処理の成否です。
 *
 * @pre
 *                - pMbuf != nullptr
 *                - cp != nullptr
 *
 * @details       mbuf チェインのサイズが不足している場合、
 *                必要に応じて新しい mbuf を割り当てて mbuf チェインを拡張します。
 */
NN_MBUF_EXTERN_C bool nnmbufMbufAppend(nnmbufMbuf* pMbuf, size_t len, const void* cp);

/**
 * @brief         新しい mbuf チェインを確保して与えられた mbuf チェインの先頭にリンクします。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     len          先頭に挿入する mbuf チェインのバイト長
 * @param[in]     how          アロケーションする際にブロックを許容するか否か
 *
 * @retval        nullptr      処理に失敗しました。
 * @retval        nullptr 以外 mbuf チェインの先頭へのポインタです。
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_EXTERN_C nnmbufMbuf* nnmbufMbufPrependWithAllocation(
    nnmbufMbuf* pMbuf, size_t len, nnmbufMbufAllocationMode how);

/**
 * @brief         新しい mbuf チェインを確保して与えられた mbuf チェインの先頭にリンクします。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     len          先頭に挿入する mbuf チェインのバイト長
 * @param[in]     how          アロケーションする際にブロックを許容するか否か
 *
 * @retval        nullptr      処理に失敗しました。
 * @retval        nullptr 以外 mbuf チェインの先頭へのポインタです。
 *
 * @pre
 *                - pMbuf != nullptr
 *
 * @details       @ref nnmbufMbufReserve() で予約された領域に @a len が収まるようであれば、
 *                新しい mbuf チェインをアロケートせず予約された領域を利用します。
 *                @ref nnmbufMbufPrependWithAllocation() は必ず mbuf を確保する点が異なります。
 */
NN_MBUF_EXTERN_C nnmbufMbuf* nnmbufMbufPrepend(
    nnmbufMbuf* pMbuf, size_t len, nnmbufMbufAllocationMode how);

/**
 * @brief         mbuf チェインの先頭データの連続性を確保します。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     len          連続性を確保したいサイズ
 *
 * @retval        nullptr      処理に失敗しました。
 * @retval        nullptr 以外 新しい mbuf チェインの先頭へのポインタです。
 *
 * @pre
 *                - pMbuf != nullptr
 *
 * @details       連続性を確保することによって mbuf チェインに格納されたデータに
 *                ポインタでアクセスできるようになります。 @a len に指定できるサイズの最大長は
 *                @ref nn::mbuf::MbufServer のコンストラクタに渡した
 *                @ref nn::mbuf::Config の unitSize によって決まります。
 *
 *                この関数の実行で新しい mbuf が割り当てられデータが移動する可能性があるため、
 *                古い mbuf チェインへの参照は無効になることに注意してください。
 *
 * @see           nnmbufMbufTod()
 */
NN_MBUF_EXTERN_C nnmbufMbuf* nnmbufMbufPullup(nnmbufMbuf* pMbuf, size_t len);

/**
 * @brief         mbuf チェインを複製します。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     how          アロケーションする際にブロックを許容するか否か
 *
 * @retval        nullptr      処理に失敗しました。
 * @retval        nullptr 以外 複製された mbuf チェインの先頭へのポインタです。
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_EXTERN_C nnmbufMbuf* nnmbufMbufDup(const nnmbufMbuf* pMbuf, nnmbufMbufAllocationMode how);

/**
 * @brief         mbuf チェインから指定されたバッファへデータをコピーします。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     offset       コピーの開始位置を指定する mbuf チェイン中のオフセット
 * @param[in]     len          コピーするデータのバイト長
 *                             (@ref nnmbufMbufCopyall を指定した場合は全てのデータをコピー)
 * @param[in]     buf          コピー先のバッファ
 *
 * @return        処理の結果です。
 *
 * @pre
 *                - pMbuf != nullptr
 *                - buf != nullptr
 */
NN_MBUF_EXTERN_C bool nnmbufMbufCopydata(
    const nnmbufMbuf* pMbuf, size_t offset, size_t len, void* buf);

/**
 * @brief         mbuf チェインにデータをセットします。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     offset       コピー先の位置を指定する mbuf チェイン中のオフセット値
 * @param[in]     len          コピーするデータのバイト長
 * @param[in]     buf          コピー元のバッファ
 *
 * @return        処理の結果です。
 *
 * @pre
 *                - pMbuf != nullptr
 *                - buf != nullptr
 *
 * @details       mbuf チェインのサイズが不足している場合、
 *                必要に応じて新しい mbuf を割り当てて mbuf チェインを拡張します。
 */
NN_MBUF_EXTERN_C bool nnmbufMbufCopyback(
    nnmbufMbuf* pMbuf, size_t offset, size_t len, const void* buf);

/**
 * @brief         mbuf チェインの末尾に別の mbuf チェインを連結します。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     pMbufNext    pMbuf の末尾に連結される mbuf チェイン
 *
 * @return        処理の結果です。
 *
 * @pre
 *                - pMbuf != nullptr
 *                - pMbufNext != nullptr
 */
NN_MBUF_EXTERN_C bool nnmbufMbufCat(nnmbufMbuf* pMbuf, nnmbufMbuf* pMbufNext);

/**
 * @brief         mbuf チェインを 2 つに分割します。
 *
 * @param[in]     pMbuf        分割対象の mbuf チェイン
 * @param[in]     len          前半部分のバイト長
 * @param[in]     how          アロケーションする際にブロックを許容するか否か
 *
 * @retval        nullptr      処理に失敗しました。
 * @retval        nullptr 以外 後半部分の mbuf チェインです。
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_EXTERN_C nnmbufMbuf* nnmbufMbufSplit(
    nnmbufMbuf* pMbuf, size_t len, nnmbufMbufAllocationMode how);

/**
 * @brief         mbuf チェインに格納されているデータのサイズを取得します。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[out]    pOutLastMbuf mbuf チェインの最後のクラスタの出力先 (不要な場合 NULL を指定)
 *
 * @return        mbuf チェインのデータ長です。
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_EXTERN_C size_t nnmbufMbufLength(nnmbufMbuf* pMbuf, nnmbufMbuf** last);

/**
 * @brief         mbuf チェインに格納されているデータを引数にコールバック関数を実行します。
 *
 * @param[in]     pMbuf        対象の mbuf チェイン
 * @param[in]     offset       mbuf チェイン中のデータオフセット
 * @param[in]     len          データサイズ
 * @param[in]     func         コールバック関数
 * @param[in]     arg          コールバック関数に渡す任意の引数
 *
 * @return        処理の結果です。
 * @retval        0            処理に成功しました。
 * @retval        -1           処理に失敗しました。
 * @retval        -1 以外      @a func が返したその他のエラーです。
 *
 * @pre
 *                - pMbuf != nullptr
 *                - func != nullptr
 *
 * @details       mbuf チェインに格納された @a offset から @a len Byte の全データに対して、
 *                そのデータを引数としてコールバック関数の @a func が実行されます。
 *                コールバック関数は mbuf チェインを構成する mbuf の構成に応じて、
 *                複数回実行される可能性があることに注意してください。
 *
 *                また、コールバック関数 @a func は成功時に 0 を返すように設計してください。
 *                0 以外の値を返した場合にはエラーが発生したものとして扱われ、
 *                まだ全てのデータに対してコールバック関数が呼び出されていなかったとしても、
 *                その時点で処理が中断されて @a func の結果が返り値として返されます。
 */
NN_MBUF_EXTERN_C int nnmbufMbufApply(nnmbufMbuf* pMbuf, size_t offset, size_t len, int (* func)(void* arg, void* data, size_t len), void* arg);

/**
 * @brief         mbuf データ領域を指定したバイト境界にアライメントします。
 *
 * @param[in]     pMbuf        対象の mbuf
 * @param[in]     align        アライメントサイズ
 * @param[in]     offset       バイト境界からのオフセット
 *
 * @return        処理の成否です。
 *
 * @pre
 *                - pMbuf != nullptr
 *                - align が 2 のべき乗
 *                - offset < align
 *
 * @details       @a align で指定されたバイト境界にアライメントした後、
 *                @a offset を足したアドレスをデータ領域の先頭アドレスとします。
 *                データ領域の空き容量を使用して調整を行うため、
 *                十分な空き容量が存在しない場合には失敗します。
 */
NN_MBUF_EXTERN_C bool nnmbufMbufAlignWithOffset(nnmbufMbuf* pMbuf, size_t align, size_t offset);

/**
 * @brief         mbuf のデータ領域の先頭から、
 *                実際にデータが格納されているアドレスまでのオフセット値を取得します。
 *
 * @param[in]     pMbuf        対象の mbuf
 *
 * @return        実際にデータが格納されているアドレスまでのオフセット値です。
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_C_INLINE size_t nnmbufMbufLeadingspace(const nnmbufMbuf* pMbuf)
{
    return pMbuf->_top;
}

/**
 * @brief         mbuf のデータ領域の空き容量を取得します。
 *
 * @param[in]     pMbuf        対象の mbuf
 *
 * @return        データ領域の空き容量です。
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_C_INLINE size_t nnmbufMbufTrailingspace(const nnmbufMbuf* pMbuf)
{
    return pMbuf->_capacity - (pMbuf->_top + pMbuf->_len);
}

/**
 * @brief         mbuf のデータ領域へのポインタを取得します。
 *
 * @param[in]     pMbuf        対象の mbuf
 *
 * @return        データ領域へのポインタです。
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_C_INLINE void* nnmbufMbufTod(nnmbufMbuf* pMbuf)
{
    return &(pMbuf->_data[pMbuf->_top]);
}

/**
 * @brief         任意の用途に利用できるポインタを mbuf にセットします。
 *
 * @param[in]     pMbuf        対象の mbuf
 * @param[in]     ptr          セットするポインタ
 *
 * @pre
 *                - pMbuf != nullptr
 *
 * @see           nnmbufMbufGetUserPointer()
 */
NN_MBUF_C_INLINE void nnmbufMbufSetUserPointer(nnmbufMbuf* pMbuf, void* ptr)
{
    pMbuf->_pGeneral = ptr;
}

/**
 * @brief         任意の用途に利用できるポインタを mbuf から取得します。
 *
 * @param[in]     pMbuf        対象の mbuf
 *
 * @return        @ref nnmbufMbufSetUserPointer() で設定したポインタです。
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_C_INLINE void* nnmbufMbufGetUserPointer(const nnmbufMbuf* pMbuf)
{
    return pMbuf->_pGeneral;
}

/**
 * @brief         mbuf のデータ長を加算します。
 *
 * @param[in]     pMbuf        対象の mbuf
 * @param[in]     len          加算するバイト長
 *
 * @return        処理の成否 (最大サイズを超える場合に失敗)
 *
 * @pre
 *                - pMbuf != nullptr
 */
NN_MBUF_C_INLINE bool nnmbufMbufExpand(nnmbufMbuf* pMbuf, size_t len)
{
    if( pMbuf->_top + pMbuf->_len + len > pMbuf->_capacity )
    {
        return false;
    }

    pMbuf->_len += (int16_t)len;
    return true;
}

/**
 * @brief         mbuf のデータ領域の使用を予約します。
 *
 * @param[in]     pMbuf        対象の mbuf
 * @param[in]     len          予約するバイト長
 *
 * @pre
 *                - pMbuf != nullptr
 *                - nnmbufMbufLength(pMbuf) == 0
 *                - len <= nnmbufMbufTrailingspace(pMbuf)
 */
NN_MBUF_C_INLINE bool nnmbufMbufReserve(nnmbufMbuf* pMbuf, size_t len)
{
    if( pMbuf->_len == 0 && pMbuf->_top + len <= pMbuf->_capacity )
    {
        pMbuf->_top += (uint16_t)len;
        return true;
    }
    else
    {
        return false;
    }
}

/**
 * @brief         指定した mbuf プールの空きmbuf unit数を返します。
 *
 * @param[in]     typef        対象の mbuf プール
 *
 * @return        指定した mbuf プールの空きmbuf unit数。指定した mbuf プールが存在しない場合-1が返ります。
 *
 */
NN_MBUF_EXTERN_C int nnmbufMbufGetFreeCount(int type);
