﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/nn_Result.h>

#include <nn/tsc/tsc_Types.h>

namespace nn { namespace tsc {

//--------------------------------------------------------------------------
/**
*   @brief    The class to configure IP information of underlying TCP/IP stack
*
*   @details
*       To configure IP information of underlying TCP/IP stack, user is required to set and
*       apply IP information throuth the object of this class. After initializing TSC library
*       by @ref Initialize, user will first set network interface name, MTU size and IP information
*       by methods provided in this class. At this moment, configured information still stays
*       just inside TSC. Note that appropriate network interface name needs to be obtained by
*       the interface provided by network interface driver.
*       Then user will call @ApplyConfig to apply such the configuration to underlying TCP/IP
*       stack. Because @ApplyConfig is asynchronous request, user is required to wait for
*       completion of the request on the event obtained by @ref GetEventPointer.
*       Until the event is notified, it is not allowed to generate another @ref ApplyConfig
*       request and to finalize library by @ref Finalize. In such the situation, user is
*       required to first cancel ongoing asynchronous request by @ref CancelApplyConfig.
*/
class ConfigContext
{
private:
    nn::os::Event          m_Event;
    nn::os::SystemEvent    m_SystemEvent;
    nn::os::TimerEvent     m_TimerEvent;
    nn::os::EventClearMode m_EventClerMode;
    nn::Result             m_ApplyResult;
    uint32_t               m_MsecTimeout;
    uint32_t               m_Mtu;
    char                   m_InterfaceName[g_MaxInterfaceNameLength];
    bool                   m_IsUseSystemEvent;

public:
    explicit ConfigContext(nn::os::EventClearMode eventClearMode);
    virtual ~ConfigContext();

    virtual Result ApplyConfig(int modeMask) NN_NOEXCEPT = 0;
    virtual Result ApplyConfig(int modeMask, int msecTimeout) NN_NOEXCEPT = 0;
    virtual Result CancelApplyConfig() NN_NOEXCEPT  = 0;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get pointer to the timer evnet object.
    *   @details    The timer event obtained by this method will be notified when @refApplyConfig
    *               compeletes or after specified time set by @ref SetMsecTimeout.
    *   @pre
    *   @post
    *
    *   @return     Pointer to the timer event object is returned
    */
    nn::os::TimerEvent* GetTimerEventPointer() NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get pointer to the evnet object.
    *   @details    The event obtained by this method will be notified when @refApplyConfig
    *               completes.
    *   @pre
    *   @post
    *
    *   @return     Pointer to the event object is returned
    */
    nn::os::Event* GetEventPointer() NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get pointer to the system evnet object.
    *   @details    The event obtained by this method will be notified when the event is reported
    *               from underlying network stack. Once this pointer is obtained, the system event
    *               is passed to the underlying ApplyConfig() function. nn::os::EventClearMode
    *               which is passed to the constructor of this class will be passed to the
    *               underlying network stack along with the system event.
    *   @pre
    *   @post
    *
    *   @return     Pointer to the system event object is returned
    */
    nn::os::SystemEvent* GetSystemEventPointer() NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Set MTU size.
    *   @details    Passed MTU value needs to be bigger than @ref g_MinMtuSize and smaller than
    *               @ref g_MaxMtuSize.
    *               This API sets it to internal configuration in TSC. Actual configuration will be
    *               done by @ref ApplyIpConfig.
    *   @pre        User has already called tsc::Initialize.
    *   @post       User needs to call @ref ApplyConfig to apply configuration
    *
    *   @param[in]  mtuValue    MTU value to set
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultMtuInvalid               Passed MTU value is not in the valid range
    */
    nn::Result SetMtu(uint32_t mtuValue) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get MTU size.
    *   @pre        User has already called tsc::Initialize.
    *   @post
    *
    *   @param[out] pMtuValue    Pointer to the memory where obtained MTU value is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    */
    nn::Result GetMtu(uint32_t* pMtuValue) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Set network interface name.
    *   @details    This API specifies network interface to configure. The interface name set with
    *               this API needs to be obtained by the API of network interface driver.
    *   @pre        User has already called tsc::Initialize.
    *   @post       User needs to call @ref ApplyConfig to apply configuration
    *
    *   @param[in]  pInInterfaceName    Pointer to the memory where interface name is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Passed interface name is too long
    */
    nn::Result SetInterfaceName(const char* pInInterfaceName) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get interface name stored locally inside TSC.
    *   @details    This API obtains network interface name which is stored inside TSC library.
    *               The name which is set by @refSetInterfaceName will be obtaitned.
    *   @pre        User has already called tsc::Initialize.
    *   @post
    *
    *   @param[out] pOutInterfaceName    Pointer to the memory where obtained interface name is stored
    *   @param[in]  nameBufferLen        Length of pInterfaceName
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultBufferTooShort           Passed buffer length (@ref nameBufferLen) is too short
    */
    nn::Result GetInterfaceName(char* pOutInterfaceName, int nameBufferLen) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get result of @ref ApplyResult.
    *   @details    Obtains result of @ref ApplyConfig. Since @ref ApplyConfig is non-blocking API,
    *               the caller is requried to obtain the actual result of @ref ApplyConfig operation.
    *   @pre        User has already called tsc::Initialize, had called tsc::ConfigContext::ApplyConfig
    *               and the event obtained by  @ref GetEventPointer has been already notified.
    *   @post
    *
    *   @return     Result of @ref ApplyConfig.
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultProcessAborted           @ref ApplyConfig call is canceled by @ref CancelApplyIpConfig
    *   @retval     ResultProcessTimeout           Timeout occurred in TCP/IP stack
    *   @retval     ResultInterfaceNameInvalid     Interface name is not specified yet or invalid
    *   @retval     ResultIpv4ConfigMethodInvalid  Stored IP configuration method is invalid
    *   @retval     ResultIpv4AddressInvalid       Stored interface address is invalid
    *   @retval     ResultIpv4SubnetMaskInvalid    Stored subnet mask is invalid
    *   @retval     ResultMtuInvalid               Stored MTU value is invalid
    *   @retval     ResultMaxInterfaces            The maximum number of network interface is already in use
    *   @retval     ResultResourceBusy             Failed to process the request due to insufficient internal resource
    *   @retval     ResultInternalModuleFailed     Failed to process necessary modules
    */
    nn::Result GetApplyResult() NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Set result of @ref ApplyResult.
    *   @pre
    *   @post
    *
    *   @param[in]  result    result to set
    *
    *   @return     None
    */
    void SetApplyResult(nn::Result result) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Set timeout for internal @ref ApplyConfig operation (optional).
    *   @details    Internal operation for @ref ApplyConfig will be aborted after specified
    *               time set by this API.
    *   @pre        User has already called tsc::Initialize.
    *   @post
    *
    *   @param[in]  msecTimout    Timeout to set in milliseconds
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess Success
    */
    nn::Result SetMsecTimeout(uint32_t msecTimeout) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get timeout value in msec
    *   @pre
    *   @post
    *   @return     Timeout value set by @ref SetMsecTimeout is returned
    */
    uint32_t GetMsecTimeout() NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get a pointer to the event clear mode object.
    *   @pre
    *   @post
    *   @return     Pointer to the event clear mode object
    */
    nn::os::EventClearMode* GetEventClearModePointer() NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get the status of system event usage
    *   @pre
    *   @post
    *   @return     true when the system event is used, false when it's not used
    */
    bool IsUseSystemEvent() NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Notify network inteface down to the TCP/IP stack
    *   @details    This API tells TCP/IP stack that link of the interface specified by
    *               @ref SetInterfaceName became DOWN.
    *   @pre        User has already called tsc::Initialize, has already configured interface
    *               by @ref ApplyConfig and the interface is running UP.
    *   @post
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid     Interface name is not set yet
    *   @retval     ResultInterfaceNotReady        There's no interface to notify
    *   @retval     ResultConfigurationInProgress  TSC library is in the process of configuration requested by @ref ApplyConfig
    *   @retval     ResultInternalModuleFailed     Failed to notify interface down due the failure in low level
    */
    Result NotifyInterfaceDown() NN_NOEXCEPT;
};

class Ipv4ConfigContext : public ConfigContext
{
private:
    Ipv4Config m_IpConfig;

public:
    Ipv4ConfigContext(const char* pInInterfaceName, nn::os::EventClearMode eventClearMode);
    virtual ~Ipv4ConfigContext();

    //--------------------------------------------------------------------------
    /**
    *   @brief      Set IPv4 configuration to internal configuration in TSC.
    *   @details    Setting IPv4 configuration doesn't perform actual configuration. Actual
    *               configuration to the TCP/IP stack will be performed by @ref ApplyConfig.
    *   @pre        User has already called tsc::Initialize.
    *   @post       User needs to call @ref ApplyConfig to apply configuration
    *
    *   @param[in]  pConfig    Pointer to the memory where IP configuration information is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultLibraryNotInitialized        TSC library is not initialized yet
    *   @retval     ResultIpv4ConfigMethodInvalid      Passed IP configuration method is invalid
    *   @retval     ResultIpv4AddressInvalid           Passed IPv4 address is invalid
    *   @retval     ResultIpv4SubnetMaskInvalid        Passed subnet mask is invalid
    *   @retval     ResultIpv4DefaultGatewayInvalid    Passed default gateway address is invalid
    *   @retval     ResultIpv4PreferredDnsInvalid      Passed preferred DNS address is invalid
    *   @retval     ResultIpv4AlternativeDnsInvalid    Passed alternative DNS address is invalid
    */
    Result SetConfig(const Ipv4Config* pInConfig) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Get IPv4 configuration from internal configuration in TSC.
    *   @pre        User has already called tsc::Initialize.
    *   @post
    *
    *   @param[out] pConfig    Pointer to the memory where obtained IP information is stored
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    */
    Result GetConfig(Ipv4Config* pOutConfig) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Perform duplicate IP check (not supported yet)
    *   @details
    *   @pre        User has already called tsc::Initialize.
    *   @post
    *
    *   @param[in]  msecTimeout    Timeout in milli seconds
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    */
    Result CheckDuplicateIp(int msecTimeout) NN_NOEXCEPT;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Apply IP configuration to the TCP/IP stack.
    *   @details    No actual configuration is done until this API is called. TSC library maintains
    *               IP configuration parameter which is set by other TSC APIs internally.
    *               The internal configuration parameter will be first applied to the TCP/IP stack
    *               by this API.
    *
    *               This API returns immediately. The event obtained by @ref GetEventPointer will
    *               be notified after request completes. Actual result of @ref ApplyConfig will be
    *               obtained by @ref GetApplyResult after the event is notified.
    *               While operation for @ref ApplyConfig is performed in TSC library, it is not
    *               allowed to perform another configuration. This internal process can be aborted
    *               by @ref CancelApplyIpConfig.
    *
    *               @ref modeMask may be optionally passed to specify how IP configuration is done.
    *               @ref IpApplyModeMask will be used to specify the mode, otherwise 0 will
    *               be passed to it.
    *   @pre        User has already called tsc::Initialize, has configured IP information by
    *               @ref SetConfig, has set interface name by @ref SetInterfaceName and has confiured
    *               MTU value by @ref SetMtu.
    *   @post       User will wait configuration process gets done on the event obtained by
    *               @ref GetEventPointer and will check the result of configuration by
    *               @ref GetApplyResult.
    *
    *   @param[in]  modeMask       Mode to spacify how IP configuration is done
    *   @param[in]  msecTimeout    Timeout in milli seconds
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                      Success
    *   @retval     ResultLibraryNotInitialized        TSC library is not initialized yet
    *   @retval     ResultInterfaceNameInvalid         Interface name is not specified yet
    *   @retval     ResultIpv4ConfigMethodInvalid      Stored IP configuration method is invalid
    *   @retval     ResultIpv4AddressInvalid           Stored interface address is invalid
    *   @retval     ResultIpv4SubnetMaskInvalid        Stored subnet mask is invalid
    *   @retval     ResultIpv4DefaultGatewayInvalid    Stored default gateway address is invalid
    *   @retval     ResultIpv4PreferredDnsInvalid      Stored preferred DNS address is invalid
    *   @retval     ResultIpv4AlternativeDnsInvalid    Stored alternative DNS address is invalid
    *   @retval     ResultMtuInvalid                   Stored MTU value is invalid
    *   @retval     ResultInternalModuleFailed         Failed to process necessary modules
    *   @retval     ResultInvalidInternalLogic         Failed to process request due to internal problem
    *   @retval     ResultDuplicateIp                  TCP/IP stack detected a duplicate Ip on the network
    */
    Result ApplyConfig(int modeMask) NN_NOEXCEPT override;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Apply IP configuration to the TCP/IP stack with timeout.
    *   @details    Calling this API is same as calling the other version of @ref ApplyConfig which
    *               doesn't take timeout in argument after setting timeout by @ref SetTimeout.
    *               Timeout will be passed in milli seconds to @ref msecTimeout which will
    *               overwrite previously set timeout by @ref SetTimeout.
    *   @pre        Please refer the description in @ref ApplyConfig for further information.
    *   @post       Please refer the description in @ref ApplyConfig for further information.
    *   @param[in]  modeMask       Mode to spacify how IP configuration is done
    *   @param[in]  msecTimeout    Timeout in milli seconds
    *
    *   @return     Please refer the description in @ref ApplyConfig for futther information.
    */
    Result ApplyConfig(int modeMask, int msecTimeout) NN_NOEXCEPT override;

    //--------------------------------------------------------------------------
    /**
    *   @brief      Cancel ongoing @ref ApplyIpConfig operation.
    *   @details    This API aborts ongoing operation for @ref ApplyIpConfig inside TSC library.
    *   @pre        User has already called tsc::Initialize, has called @ref ApplyConfig and it
    *               doesn't return yet.
    *   @post
    *
    *   @return     Result of the process is returned
    *   @retval     ResultSuccess                  Success
    *   @retval     ResultLibraryNotInitialized    TSC library is not initialized yet
    *   @retval     ResultNoOngoingProcess         There's no operation to cancel
    *   @retval     ResultProcessAlreadyDone       @ref ApplyIpConfig has already completed
    *   @retval     ResultInternalModuleFailed     Failed with internal module
    */
    Result CancelApplyConfig() NN_NOEXCEPT override;
};

}}
