﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_TimeSpan.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Errno.h>
#include <nn/socket/socket_TypesPrivate.h>
#include <nn/socket/socket_Statistics.h>
#include <nn/socket/socket_ApiPrivateDeprecated.h>
#include <nn/socket/socket_ResolverOptionsPrivate.h>

namespace nn     {
namespace socket {


/**
    @brief          Duplicate an existing socket.

    @param[in]      socket          Socket ID
    @param[in]      ownerProcessId  Process ID

    @return         Returns non-negative socket descriptor or -1 if error occured.

*/
int DuplicateSocket(int socket, uint64_t ownerProcessId) NN_NOEXCEPT;

/**
    @brief           Retrieve resource statistics.

    @param[in]       type            Type of information requested.
    @param[out]      outBuffer       Buffer for returned output.
    @param[in]       outBufferLength Size of buffer.  Default size is DefaultStatisticsBufferSize.
    @param[in]       options         Statistic options.  See StatisticsOptions.

    @return          Returns the number of bytes written or -1 if an error occurred.

*/
int GetResourceStatistics(StatisticsType type, void* outBuffer, size_t outBufferLength, uint32_t options) NN_NOEXCEPT;

/**
    @brief          Opens a socket that is exempt from SocketShutdownAll.

    @param[in]      domain          Communication domain.
    @param[in]      type            Socket type.
    @param[in]      protocol        Protocol type.

    @return         Returns non-negative socket descriptor or -1 if error occured.

    @details        Opens a socket that is exempt from SocketShutdownAll(), unless the forced option is true for SocketShutdownAll().
                    This option can also by setting the socket option Option::So_Nn_Shutdown_Exempt to non-zero using SetSockOpt().
                    Using SocketExempt() is recommended over the @ref SetSockOpt() alternative due to possible race-conditions of being shutdown between
                    socket creation and setting of the Option::So_Nn_Shutdown_Exempt option.

    @see            Socket(), ShutdownAllSockets(), SetSockOpt()

*/
int SocketExempt(Family domain, Type type, Protocol protocol) NN_NOEXCEPT;

/**
    @brief          Cancel all resolver operations within this process.

    @details        Cancel all resolver operations within this process.

    @see            RequestCancelHandle()
*/
void CancelAll(void) NN_NOEXCEPT;

/**
    @brief          Look up @ref Family::Af_Inet host IP address by hostname.

    @param[in]      pName           The name of the host to look up (i.e. www.nintendo.com)
    @param[in]      options         An array of ResolverOption that contain the cancel handle and / or NSD
    @param[in]      optionsCount    the number of ResolverOption structures in the array

    @details        The return value of this function is thread-localized, this deviates from some POSIX implementations.
                    The return value is managed internally, if you wish to maintain the value between calls a deep copy must be made.

    @see            GetHError(), HStrError()

*/
HostEnt* GetHostEntByName(const char* pName, const nn::socket::ResolverOption* options, size_t optionsCount) NN_NOEXCEPT;

/**
    @brief          Get a list of IP addresses and port numbers.

    @param[in]      pNodeName       The name of the host or NULL.
    @param[in]      pServername     The name of the service or NULL.
    @param[in]      pHints          An optional pointer to an @ref AddrInfo structure.
    @param[out]     pResult         An out parameter containing 0, 1, or many (via @a ai_next) @ref AddrInfo structures.
    @param[in]      options         An array of ResolverOption that contain the cancel handle and / or NSD
    @param[in]      optionsCount    the number of ResolverOption structures in the array

    @return         Returns @ref AiErrno::EAi_Success on success or one of the error codes listed in @ref GAIStrError() if an error occurs.

    @details        GetAddrInfo() routine can be used to obtain list of IP addresses and port numbers for
                    host @a pNodename and service @a pServername. It provides more flexibility than @ref GetHostEntByName().

                    Please note that @a pNodename and @a pServername cannot both be NULL at the same time.

                    The @ref AddrInfo result @a pResult returned by this function can be deallocated via @ref FreeAddrInfo().

                    The optional @a pHints can be used to provide hints concerning the type of socket
                    that the caller supports or wishes to use. The caller can supply the
                    following structure elements in @a pHints:

                    <table>
                        <tr>
                            <th>Element</th>
                            <th>Description</th>
                        </tr>
                        <tr>
                            <td>@a ai_family</td>
                            <td>The protocol family that should be used. When @a ai_family is set to @ref Family::Af_Unspec,
                                it means the caller will accept any protocol family supported by the operating system.
                            </td>
                        </tr>
                        <tr>
                            <td>@a ai_socktype</td>
                            <td>Denotes the type of socket that is wanted: @ref Type::Sock_Stream, @ref Type::Sock_Dgram,
                                or @ref Type::Sock_Raw. When @a ai_socktype is zero the caller will accept any socket type.
                            </td>
                        </tr>
                        <tr>
                            <td>@a ai_protocol</td>
                            <td>Indicates which transport protocol is desired, @ref Protocol::IpProto_Udp or @ref Protocol::IpProto_Tcp. If
                                @a ai_protocol is zero the caller will accept any protocol.</td>
                        </tr>
                        <tr>
                            <td>@a ai_flags</td>
                            <td>The @a ai_flags field to which the hints parameter points shall be set to zero or be the
                                bitwise-inclusive OR of one or more of the values:
                                <table>
                                    <tr>
                                        <th>Flag</th>
                                        <th>Description</th>
                                    </tr>
                                    <tr>
                                        <td>@ref AddrInfoFlag::Ai_AddrConfig</td>
                                        <td>
                                            If the @ref AddrInfoFlag::Ai_AddrConfig bit is set, IPv4 addresses shall be
                                            returned only if an IPv4 address is configured on the local system, and
                                            IPv6 addresses shall be returned only if an IPv6 address is
                                            configured on the local system.
                                        </td>
                                    </tr>
                                    <tr>
                                        <td>@ref AddrInfoFlag::Ai_CanonName</td>
                                        <td>
                                            If the @ref AddrInfoFlag::Ai_CanonName bit is set, a successful call to
                                            @ref GetAddrInfo() will return a NULL-terminated string containing
                                            the canonical name of the specified hostname in the @a ai_canonname
                                            element of the first @ref AddrInfo structure returned.
                                        </td>
                                    </tr>
                                    <tr>
                                        <td>@ref AddrInfoFlag::Ai_NumericHost</td>
                                        <td>
                                            If the @ref AddrInfoFlag::Ai_NumericHost bit is set, it indicates that hostname
                                            should be treated as a numeric string defining an IPv4 or IPv6 address and
                                            resolution should be attempted.
                                        </td>
                                    </tr>
                                    <tr>
                                        <td>@ref AddrInfoFlag::Ai_NumericServ</td>
                                        <td>
                                            If the @ref AddrInfoFlag::Ai_NumericServ bit is set, then a non-NULL servname
                                            string supplied shall be a numeric port string. Otherwise, an
                                            @ref AiErrno::EAi_NoName error shall be returned. This bit shall prevent any
                                            type of name resolution service (for example, NIS+) from being invoked.
                                        </td>
                                    </tr>
                                    <tr>
                                        <td>@ref AddrInfoFlag::Ai_Passive</td>
                                        <td>
                                            If the @ref AddrInfoFlag::Ai_Passive bit is set it indicates that the
                                            returned socket address structure is intended for use in a call to
                                            @ref Bind(). In this case, if the hostname argument is the null pointer,
                                            then the IP address portion of the socket address structure will be set
                                            to @ref InAddr_Any for an IPv4 address. If the @ref AddrInfoFlag::Ai_Passive
                                            bit is not set, the returned socket address structure will be ready
                                            for use in a call to @ref Connect() for a connection-oriented protocol
                                            or @ref Connect(), @ref SendTo(), or @ref SendMsg() if a connectionless
                                            protocol was chosen. The IP address portion of the socket address
                                            structure will be set to the loopback address if hostname is the null
                                            pointer and @ref AddrInfoFlag::Ai_Passive is not set.
                                        </td>
                                    </tr>
                                </table>
                            </td>
                        </tr>
                    </table>

                    All other elements of the @ref AddrInfo structure passed via @a pHints must be
                    zero or the null pointer.

                    If @a pHints is the null pointer, GetAddrInfo() behaves as if the caller provided
                    an @ref AddrInfo structure with @a ai_family set to @ref Family::Af_Unspec and all other
                    elements set to zero or NULL.

    @see            FreeAddrInfo(), GetNameInfo(), GAIStrError()
*/
AiErrno GetAddrInfo(const char* pNodeName, const char* pServername, const AddrInfo* pHints, AddrInfo** pResult, const nn::socket::ResolverOption* options, size_t optionsCount) NN_NOEXCEPT;

/**
    @brief          Shutdown all open sockets across all proccesses and threads.

    @param[in]      forced          Whether or not to force the closure of Shutdown Exempt sockets.
                                    Should be set to false without good reason to do otherwise.

    @return         Returns the number of sockets shut down, or -1 if an error occured.

    @details        This function does the equivilant of calling @ref Shutdown() on all open sockets with the @a how argument set to @ref ShutdownMethod::Shut_RdWr.
                    Additionally, this call will trigger a forceful dropping of tcp connections after the shutdown is complete.
                    This will not shut down sockets opened with @ref SocketExempt() or sockets with the Option::So_Nn_Shutdown_Exempt option set unless
                    @a forced is set to true.

    @see            Shutdown(), SocketExempt(), SetSockOpt()

*/
int ShutdownAllSockets(bool forced) NN_NOEXCEPT;

/**
    @brief          Control I/O device

    @param[in]      fd              File descriptor for socket or device.
    @param[in]      command         Device control command.
    @param[in]      pData           Buffer containing command data.
    @param[in]      dataLength      Size of the buffer.

    @return         Returns result of the ioctl operation. If an error has occurred, a value of -1
                    is returned.

    @post           In the case of an error, the global variable @a errno will be set to one of the following:

    Error              | Description
    -------------------|------------
    @ref Errno::EBadf  | The @a fd argument is not a valid descriptor.
    @ref Errno::ENotTy | The specified request does not apply to the kind of object that the descriptor fd references.
    @ref Errno::ENotTy | The specified request does not apply to the kind of object that the descriptor fd references.
    @ref Errno::EInval | The request or @a pData argument is not valid.
    @ref Errno::EAcces | Command is not allowed.
    @ref Errno::EAgain | Resources to complete the request are temporarily unavailable.

    @details        See details of @ref Ioctl(int, IoctlCommand, void*, size_t).

    @see            Socket(), Accept(), GetLastError()
*/
inline int Ioctl(int fd, IoctlCommandPrivate command, void* pData, size_t dataLength) NN_NOEXCEPT
{
    return Ioctl(fd, static_cast<IoctlCommand>(command), pData, dataLength);
}

/**
    @brief          Get or set network stack state. Restricted API, do not use, to be moved to private header.

    @param[in]      pMibEntries     Management Information Base entries
    @param[in]      mibEntryCount   Number of entries for procesing
    @param[out]     pOldValue       Buffer to return old value
    @param[in,out]  pOldValueLength Size of the buffer for old value
    @param[in]      pNewValue       Buffer containing new value
    @param[in]      newValueLength  Size of the new value

    @return         Returns the value 0 if successful; otherwise the value -1 is returned.

    @post           In the case of an error, the global variable @a errno will be set to one of the following:

    Error               | Description
    --------------------|------------
    @ref Errno::EInval  | The @a pMibEntries array is less than two or greater than CTL_MAXNAME.
    @ref Errno::EInval  | A non-null @a pNewValue is given and its specified length in @a newValueLength is too large or too small.
    @ref Errno::EInval  | @a pNewValue is null with a non-zero @a newValueLength.
    @ref Errno::EFault  | The address range referred to by @a pOldValue and the value of @a pOldValueLength does not refer to a valid part of the process address space.
    @ref Errno::EFault  | The address range referred to by @a pNewValue and @a newValueLength does not refer to a valid part of the process address space.
    @ref Errno::ENoMem  | The length pointed to by @a pOldValueLength is too short to hold the requested value.
    @ref Errno::ENoMem  | The smaller of either the length pointed to by @a pOldValueLength or the estimated size of the returned data exceeds the system limit on locked memory.
    @ref Errno::ENoMem  | Locking the buffer @a pOldValue, or a portion of the buffer if the estimated size of the data to be returned is smaller, would cause the process to exceed its per-process locked memory limit.
    @ref Errno::ENotDir | The @a pMibEntries array specifies an intermediate rather than terminal name.
    @ref Errno::EIsDir  | The @a pMibEntries array specifies a terminal name, but the actual name is not terminal.
    @ref Errno::ENoEnt  | The @a pMibEntries array specifies a value that is unknown.
    @ref Errno::EPerm   | An attempt is made to set a read-only value.
    @ref Errno::EPerm   | A process without appropriate privilege attempts to set a value.
    @ref Errno::EAgain  | Resources to complete the request are temporarily unavailable.

    @details        The Sysctl() routine allows processes with appropriate privileges to modify
                    network stack behavior.

                    This function requires calling @ref Initialize() before it can be used.

    @see            GetLastError()
*/
int Sysctl(int* pMibEntries, size_t mibEntryCount, void* pOldValue, size_t* pOldValueLength, void* pNewValue, size_t newValueLength) NN_NOEXCEPT;

/**
    @brief          Open network device for reading/writing. Restricted API, do not use, to be moved to private header.

    @param[in]      path            Path to device
    @param[in]      flags           Ignored

    @return         Returns non-negative file descriptor, or -1 if an error occurred.

    @post           In the case of an error, the global variable @a errno will be set to one of the following:

    Error                  | Description
    -----------------------|------------
    @ref Errno::EOpNotSupp | Invalid path or device does not support open call.
    @ref Errno::EInval     | The @a path argument is null.
    @ref Errno::EFault     | The @a path argument points outside the process's allocated address space.
    @ref Errno::EAgain     | Resources to complete the request are temporarily unavailable.

    @details        The file name specified by @a path is opened for reading
                    and/or writing as specified by the argument @a flags, and the file descriptor
                    returned to the calling process. When operations on the file descriptor
                    are completed, it needs to be released by calling @ref Close().

                    In the current implementation only '/dev/bpf' device
                    can be accessed via Open(). @a flags argument is not used.

                    This function requires calling @ref Initialize() before it can be used.

    @see            Close(), GetLastError()
*/
int Open(const char* path, OpenFlag flags) NN_NOEXCEPT;

/**
    @brief          Convert a @ref SockAddr structure to a pair of host name and service strings.

    @param[in]      socketAddress       The @ref SockAddr structure socketAddress should point to a @ref SockAddrIn that is socketAddressLength bytes long.
    @param[in]      socketAddressLength The size of the socckaddr structure.
    @param[out]     host                The hostname buffer.
    @param[in]      hostLength          The length of the hostname buffer.
    @param[out]     service             The service name buffer.
    @param[in]      serviceLength       The length of the service name buffer.
    @param[in]      flags               Flags.
    @param[in]      options             An array of ResolverOption that contain the cancel handle and / or NSD.
    @param[in]      optionsCount        The number of ResolverOption structures in the array.

    @return         Returns @ref AiErrno::EAi_Success on success or one of the error codes listed in @ref GAIStrError() if an error occurs.

    @details        The GetNameInfo() function is used to convert a @ref SockAddr structure to a pair of host name and service strings.
                    It provides more flexibility than @ref GetHostEntByAddr(). The @ref AddrInfo result @a pResult
                    returned by this function can be deallocated via @ref FreeAddrInfo().

                    The @a flags argument is formed by OR'ing the following values from socket_Types.h:

                    Flag                              | Description
                    ----------------------------------|------------
                    @ref NameInfoFlag::Ni_NoFqdn      | A fully qualified domain name is not required for local hosts. The local part of the fully qualified domain name is returned instead.
                    @ref NameInfoFlag::Ni_NumericHost | Return the address in numeric form, as if calling @ref InetNtop(), instead of a host name.
                    @ref NameInfoFlag::Ni_NameReqd    | A name is required. If the host name cannot be found in DNS and this flag is set, a non-zero error code is returned. If the host name is not found and the flag is not set, the address is returned in numeric form.
                    @ref NameInfoFlag::Ni_NumericServ | The service name is returned as a digit string representing the port number.
                    @ref NameInfoFlag::Ni_Dgram       | Specifies that the service being looked up is a datagram service.

    @see            FreeAddrInfo(), GetAddrInfo(), GAIStrError()
*/
AiErrno GetNameInfo(const SockAddr* socketAddress, SockLenT socketAddressLength, char* host, size_t hostLength, char* service, size_t serviceLength, NameInfoFlag flags, const nn::socket::ResolverOption* options, size_t optionsCount) NN_NOEXCEPT;

/**
    @brief          Look up the address value for a host for @ref Family::Af_Inet.

    @param[in]      pAddress        This value shall be pointer to an @ref InAddr structure that contains a binary address in network byte order.
    @param[in]      length          The size of the structure.
    @param[in]      addressFamily   Currently only @ref Family::Af_Inet is supported.
    @param[in]      options         An array of ResolverOption that contain the cancel handle and / or NSD
    @param[in]      optionsCount    the number of ResolverOption structures in the array

    @return         Returns a pointer to a @ref HostEnt structure or NULL on error with error detail in @ref GetHError().

    @details        The return value of this function is thread-localized, this deviates from some POSIX implementations.
                    The return value is managed internally, if you wish to maintain the value between calls a deep copy must be made.

    @see            GetHError(), HStrError()
*/
HostEnt* GetHostEntByAddr(const void* pAddress, SockLenT length, Family addressFamily, const nn::socket::ResolverOption* options, size_t optionsCount) NN_NOEXCEPT;

/**
  @brief            This function gets an option with the provided key from the resolver

  @details          This function gets an option from the resolver. For most types of data you pass in a @ref
                    ResolverOptionKey and an empty @ref ResolverOption. If the @a key is a pointer type then
                    you must set data.pointerValue  in @a optionOut.

  @param[out]       pOptionOut This parameter is the @ref ResolverOption out parameter that will contain the result on success.

  @param[in]        key This parameter is the resolver option key corresponding with the option you want to get

  @return           0 on success, -1 on error.
 */
int ResolverGetOption(ResolverOption* pOptionOut, ResolverOptionKey key) NN_NOEXCEPT;

/**
 * @brief           This function sets an option into the resolver
 *
 * @details         This function sets @ref ResolverOption data. It is important that you set all values of the @a option.
 *
 * @param[in]       option This parameter is the @ref ResolverOption parameter containing data you wish to set into the resolver. .
 *
 * @return          0 on success, -1 on error.
 */
int ResolverSetOption(const ResolverOption& option) NN_NOEXCEPT;


/**
    @brief          Receive a message from a socket.

    @return         Returns -1, the global variable @a errno is set to @ref Errno::EOpNotSupp.

    @details        The RecvMsg() routine is used to receive a message from a socket.
                    It uses the @ref MsgHdr structure to minimize the number of directly supplied arguments.

                    In the current implementation this function is disabled.

    @see            Recv(), RecvFrom()
*/
ssize_t RecvMsg(int socket, MsgHdr* pOutMessage, MsgFlag flags) NN_NOEXCEPT;


/**
    @brief          Send a message from a socket.

    @return         Returns -1, the global variable @a errno is set to @ref Errno::EOpNotSupp.

    @details        The SendMsg() routine is used to send a message from a socket.
                    It uses the @ref MsgHdr structure to minimize the number of directly supplied arguments.

                    In the current implementation this function is disabled.

    @see            Send(), SendTo()
*/
ssize_t SendMsg(int socket, const MsgHdr* pMessage, MsgFlag flags) NN_NOEXCEPT;

/**
    @brief          Receive multiple messages on a socket.

    @param[in]      socket      The file descriptor of the socket to receive data from.
    @param[in]      pMsgvec     A pointer to an array of @ref MMsgHdr structures.
    @param[in]      vlen        The length of the array specified by pMsgvec.
    @param[in]      flags       The flags argument contains flags ORed together.
    @param[in]      pTimeout    Points to a @ref nn::TimeSpan object defining a timeout
                                for the receive operation.

    @return         On success, RecvMMsg() returns the number of messages received in @a pMsgvec.

                    On error, -1 is returned, and @a errno is set to indicate the error. In the case
                    of an error occurring after at least one message has been received,
                    RecvMMsg() will return the number of messages received.

    @details        The RecvMMsg() system call is an extension of @ref RecvMsg() that allows the caller
                    to receive multiple messages from a socket using a single system call.
                    (This can result in a drastic reduction in CPU usage.)

                    A further extension over @ref RecvMsg() is support for a timeout on the
                    receive operation.

                    The @a sockfd argument is the file descriptor of the socket to receive
                    data from.

                    The @a pMsgvec argument is a pointer to an array of @ref MMsgHdr structures.
                    The size of this array is specified in @a vlen.

                    The @a msg_hdr field is a @ref MsgHdr structure, as described in @ref RecvMsg().
                    The @a msg_len field is the number of bytes returned for the message in
                    the entry. This field has the same value as the return value of a
                    single @ref RecvMsg() on the header.

                    The @a pTimeout argument points to a @ref nn::TimeSpan object defining a timeout
                    (seconds plus nanoseconds) for the receive operation.
                    If @a pTimeout is NULL, then the operation blocks indefinitely.

                    A blocking @ref RecvMMsg() call blocks until @a vlen messages have been
                    received or until the timeout expires. A nonblocking call reads as
                    many messages as are available (up to the limit specified by @a vlen)
                    and returns immediately.

                    On return from RecvMMsg(), successive elements of @a pMsgvec are updated
                    to contain information about each received message: @a msg_len contains
                    the size of the received message; the subfields of @a msg_hdr are
                    updated as described in @ref RecvMsg(). The return value of the call
                    indicates the number of elements of @a pMsgvec that have been updated.

    @see            SendMMsg(), RecvMsg(), Socket()
*/
ssize_t RecvMMsg(int socket, MMsgHdr* pMsgvec, size_t vlen, MsgFlag flags, nn::TimeSpan *pTimeout) NN_NOEXCEPT;

/**
    @brief          Send multiple messages on a socket.

    @param[in]      socket      The file descriptor of the socket on which data is to be transmitted.
    @param[in]      msgvec      A pointer to an array of @ref MMsgHdr structures.
    @param[in]      vlen        The length of the array specified by @a msgvec.
    @param[in]      flags       The flags argument contains flags ORed together.

    @return         On success, SendMMsg() returns the number of messages sent from @a msgvec.
                    If this is less than @a vlen, the caller can retry with a further SendMMsg()
                    call to send the remaining messages.

                    On error, -1 is returned, and @a errno is set to indicate the error. In the case
                    of an error occurring after at least one message has been sent,
                    SendMMsg() will return the number of messages sent.

    @details        The SendMMsg() system call is an extension of @ref SendMsg() that allows the caller
                    to transmit multiple messages on a socket using a single system call.
                    (This can result in a drastic reduction in CPU usage.)

                   The @a sockfd argument is the file descriptor of the socket on which
                   data is to be transmitted.

                   The @a msgvec argument is a pointer to an array of @ref MMsgHdr objects.
                   The size of this array is specified in @a vlen.

                   The @a msg_hdr field is a @ref MsgHdr structure, as described in @ref SendMsg().
                   The @a msg_len field is used to return the number of bytes sent from the
                   message in @a msg_hdr (i.e., the same as the return value from a single
                   @ref SendMsg() call).

                   The @a flags argument contains flags ORed together. The flags are the
                   same as for @ref SendMsg().

                   A blocking SendMMsg() call blocks until @a vlen messages have been sent.
                   A nonblocking call sends as many messages as possible (up to the
                   limit specified by @a vlen) and returns immediately.

                   On return from SendMMsg(), the @a msg_len fields of successive elements
                   of @a msgvec are updated to contain the number of bytes transmitted from
                   the corresponding @a msg_hdr. The return value of the call indicates
                   the number of elements of @a msgvec that have been updated.

    @see           RecvMMsg(), SendMsg(), Socket()
*/
ssize_t SendMMsg(int socket, const MMsgHdr* msgvec, size_t vlen, MsgFlag flags) NN_NOEXCEPT;

}}
