﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn {
namespace socket {

/*
 * A Link-Level Sockaddr may specify the interface in one of two
 * ways: either by means of a system-provided index number (computed
 * anew and possibly differently on every reboot), or by a human-readable
 * string such as "il0" (for managerial convenience).
 *
 * Census taking actions, such as something akin to SIOCGCONF would return
 * both the index and the human name.
 *
 * High volume transactions (such as giving a link-level ``from'' address
 * in a recvfrom or recvmsg call) may be likely only to provide the indexed
 * form, (which requires fewer copy operations and less space).
 *
 * The form and interpretation  of the link-level address is purely a matter
 * of convention between the device driver and its consumers; however, it is
 * expected that all drivers for an interface of a given if_type will agree.
 */

/**
    Structure of a Link-Level sockaddr:
 */
struct SockAddrDl
{
    uint8_t     sdl_len;            ///< Total length of sockaddr.
    Family      sdl_family;         ///< nn::socket::Family::Af_Link.
    uint16_t    sdl_index;          ///< if != 0, system given index for interface.
    uint8_t     sdl_type;           ///< interface type.
    uint8_t     sdl_nlen;           ///< interface name length, no trailing 0 reqd.
    uint8_t     sdl_alen;           ///< link level address length.
    uint8_t     sdl_slen;           ///< link layer selector length.
    char        sdl_data[46];       ///< minimum work area, can be larger; contains both if name and ll address.
};

inline char * LlAddr(const SockAddrDl *pSad)
{
    return const_cast<char *>(pSad->sdl_data + pSad->sdl_nlen);
}

inline uint16_t LlIndex(const SockAddrDl *pSad)
{
    return pSad->sdl_index;
}

inline size_t Rt_RoundUp(size_t a)
{
    return (a > 0) ? (1 + ((a-1) | (sizeof(long) - 1))) : sizeof(long);
}

inline void Rt_Advance(char **x, nn::socket::SockAddr *n)
{
    *x += Rt_RoundUp(n->sa_len);
}

}} /* nn::socket */
