﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/* For certain IOCTL's BSD is passing embedded pointers in the
 * IOCTL structures. We extract the pointers, map them as exchange buffers
 * prior to sending command to the server. On the server side pointers in the
 * command block are set to point to the mapped exchange buffers.
 */

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/socket/net/bpf.h>
#include <nn/socket/sys/filio.h>
#include <nn/socket/sys/sockio.h>

namespace nn     {
namespace socket {
namespace detail {
namespace        {

const int IoctlEmbeddedSegmentCount = 4;

static inline int ExtractIoctlEmbeddedPointers(
                uint32_t command,
                void*    commandBlock,
                size_t   commandBlockLength,
                void*    embeddedBuffers[],
                size_t   embeddedBufferLengths[])
{
    uint32_t embeddedSegmentCount = 0;

    embeddedBuffers[0]       = commandBlock;
    embeddedBufferLengths[0] = commandBlockLength;
    embeddedSegmentCount++;

    if (commandBlockLength < IOCPARM_LEN(command))
    {
        return -1;
    }
    /* Extract embedded pointers */
    switch (command)
    {
        case SIOCGIFCONF:
        {
            NN_ABORT_UNLESS(commandBlockLength == sizeof(struct ifconf));
            embeddedBuffers[embeddedSegmentCount] = ((struct ifconf*)commandBlock)->ifc_buf;
            embeddedBufferLengths[embeddedSegmentCount] = ((struct ifconf*)commandBlock)->ifc_len;
            embeddedSegmentCount++;
            break;
        }
        case SIOCGIFINDEX:
        case SIOCSIFLLADDR:
        case SIOCAIFADDR:
        case SIOCDIFADDR:
        case SIOCSIFFLAGS:
        case SIOCGIFADDR:
        case SIOCGIFNETMASK:
        case SIOCGIFMEDIA:
        case SIOCGIFDUPINFO:
        case SIOCSIFMTU:
        case SIOCGIFMTU:
        case SIOCGIFMETRIC:
        case SIOCSIFMETRIC:
        case SIOCATMARK:
        case BIOCSETIF:
        case BIOCVERSION:
        case BIOCIMMEDIATE:
        case BIOCGBLEN:
        case BIOCSETF:
        case BIOCGDLT:
        case FIONREAD:
        case FIONWRITE:
        case FIONSPACE:
            /* No embedded pointers, only command block */
            break;
        default:
            /* If ioctl command is not defined here, we have not reviewed/checked whether
             * it passes any embedded pointers in command block.
             */
            NN_SDK_LOG("[net]: ioctl command %x has not been reviewed yet\n", command);
            return -1;
    }

    return embeddedSegmentCount;
}

static inline int RestoreIoctlEmbeddedPointers(
                uint32_t command,
                void*    embeddedBuffers[],
                size_t   embeddedBufferLengths[],
                uint32_t embeddedSegmentCount)
{
    void*  commandBlock       = embeddedBuffers[0];
    size_t commandBlockLength = embeddedBufferLengths[0];

    if (commandBlockLength < IOCPARM_LEN(command))
    {
        return -1;
    }

    /* Set embedded pointers in the main command block */
    switch (command)
    {
        case SIOCGIFCONF:
        {
            NN_ABORT_UNLESS(commandBlockLength == sizeof(struct ifconf));
            NN_ABORT_UNLESS(embeddedSegmentCount == 2);
            ((struct ifconf*)commandBlock)->ifc_buf = (char*)embeddedBuffers[1];
            ((struct ifconf*)commandBlock)->ifc_len = embeddedBufferLengths[1];
            break;
        }
        case SIOCGIFINDEX:
        case SIOCSIFLLADDR:
        case SIOCAIFADDR:
        case SIOCDIFADDR:
        case SIOCSIFFLAGS:
        case SIOCGIFADDR:
        case SIOCGIFNETMASK:
        case SIOCGIFMEDIA:
        case SIOCGIFDUPINFO:
        case SIOCSIFMTU:
        case SIOCGIFMTU:
        case SIOCGIFMETRIC:
        case SIOCSIFMETRIC:
        case SIOCATMARK:
        case BIOCSETIF:
        case BIOCVERSION:
        case BIOCIMMEDIATE:
        case BIOCGBLEN:
        case BIOCSETF:
        case BIOCGDLT:
        case FIONREAD:
        case FIONWRITE:
        case FIONSPACE:
            NN_ABORT_UNLESS(embeddedSegmentCount == 1);
            break;
        default:
            /* If ioctl command is not defined here, we have not reviewed/checked whether
             * it passes any embedded pointers in command block.
             */
            NN_SDK_LOG("[net]: ioctl command %x has not been reviewed yet\n", command);
            return -1;
    }
    return 0;
}

}}}}
