﻿/*-
 * Copyright (c) 1990, 1991, 1993
 *    The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from the Stanford/CMU enet packet filter,
 * (net/enet.c) distributed as part of 4.3BSD, and code contributed
 * to Berkeley by Steven McCanne and Van Jacobson both of Lawrence
 * Berkeley Laboratory.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      @(#)bpf.h    8.1 (Berkeley) 6/10/93
 *    @(#)bpf.h    1.34 (LBL)     6/16/96
 *
 * $FreeBSD$
 */

#pragma once

#include <nn/socket/sys/ioccom.h>

/* BSD style release date */
enum
{
    BPF_RELEASE    = 199606
};

typedef int32_t  bpf_int32;
typedef uint32_t bpf_u_int32;
typedef int64_t  bpf_int64;
typedef uint64_t bpf_u_int64;

/*
 * Alignment macros.  BPF_WORDALIGN rounds up to the next
 * even multiple of BPF_ALIGNMENT.
 */
enum
{
    BPF_ALIGNMENT = sizeof(long)
};

#define BPF_WORDALIGN(x)    (((x)+(BPF_ALIGNMENT-1))&~(BPF_ALIGNMENT-1))

enum
{
    BPF_MAXINSNS   = 512,
    BPF_MAXBUFSIZE = 0x80000,
    BPF_MINBUFSIZE = 32
};

/*
 * The instruction data structure.
 */
struct bpf_insn
{
    uint16_t        code;
    uint8_t         jt;
    uint8_t         jf;
    bpf_u_int32     k;
};

/*
 *  Structure for BIOCSETF.
 */
struct bpf_program
{
    uint32_t bf_len;
    struct bpf_insn bf_insns[BPF_MAXINSNS]; /* bf_len number of elements */
};

/*
 * Struct returned by BIOCGSTATS.
 */
struct bpf_stat
{
    uint32_t bs_recv;        /* number of packets received */
    uint32_t bs_drop;        /* number of packets dropped */
};

/*
 * Struct return by BIOCVERSION.  This represents the version number of
 * the filter language described by the instruction encodings below.
 * bpf understands a program iff kernel_major == filter_major &&
 * kernel_minor >= filter_minor, that is, if the value returned by the
 * running kernel has the same major number and a minor number equal
 * equal to or less than the filter being downloaded.  Otherwise, the
 * results are undefined, meaning an error may be returned or packets
 * may be accepted haphazardly.
 * It has nothing to do with the source code version.
 */
struct bpf_version
{
    uint16_t bv_major;
    uint16_t bv_minor;
};

/* Current version number of filter architecture. */
enum
{
    BPF_MAJOR_VERSION = 1,
    BPF_MINOR_VERSION = 1
};

/*
 * Historically, BPF has supported a single buffering model, first using mbuf
 * clusters in kernel, and later using malloc(9) buffers in kernel.  We now
 * support multiple buffering modes, which may be queried and set using
 * BIOCGETBUFMODE and BIOCSETBUFMODE.  So as to avoid handling the complexity
 * of changing modes while sniffing packets, the mode becomes fixed once an
 * interface has been attached to the BPF descriptor.
 */
enum
{
    BPF_BUFMODE_BUFFER = 1,     /* Kernel buffers with read(). */
    BPF_BUFMODE_ZBUF   = 2      /* Zero-copy buffers. */
};

/*-
 * Struct used by BIOCSETZBUF, BIOCROTZBUF: describes up to two zero-copy
 * buffer as used by BPF.
 */
struct bpf_zbuf
{
    void        *bz_bufa;       /* Location of 'a' zero-copy buffer. */
    void        *bz_bufb;       /* Location of 'b' zero-copy buffer. */
    size_t       bz_buflen;     /* Size of zero-copy buffers. */
};

#define BIOCGBLEN       _IOR('B', 102, uint32_t)
#define BIOCSBLEN       _IOWR('B', 102, uint32_t)
#define BIOCSETF        _IOW('B', 103, struct bpf_program)
#define BIOCFLUSH       _IO('B', 104)
#define BIOCPROMISC     _IO('B', 105)
#define BIOCGDLT        _IOR('B', 106, uint32_t)
#define BIOCGETIF       _IOR('B', 107, struct ifreq)
#define BIOCSETIF       _IOW('B', 108, struct ifreq)
#define BIOCSRTIMEOUT   _IOW('B', 109, struct timeval)
#define BIOCGRTIMEOUT   _IOR('B', 110, struct timeval)
#define BIOCGSTATS      _IOR('B', 111, struct bpf_stat)
#define BIOCIMMEDIATE   _IOW('B', 112, uint32_t)
#define BIOCVERSION     _IOR('B', 113, struct bpf_version)
#define BIOCGRSIG       _IOR('B', 114, uint32_t)
#define BIOCSRSIG       _IOW('B', 115, uint32_t)
#define BIOCGHDRCMPLT   _IOR('B', 116, uint32_t)
#define BIOCSHDRCMPLT   _IOW('B', 117, uint32_t)
#define BIOCGDIRECTION  _IOR('B', 118, uint32_t)
#define BIOCSDIRECTION  _IOW('B', 119, uint32_t)
#define BIOCSDLT        _IOW('B', 120, uint32_t)
#define BIOCGDLTLIST    _IOWR('B', 121, struct bpf_dltlist)
#define BIOCLOCK        _IO('B', 122)
#define BIOCSETWF       _IOW('B', 123, struct bpf_program)
#define BIOCFEEDBACK    _IOW('B', 124, uint32_t)
#define BIOCGETBUFMODE  _IOR('B', 125, uint32_t)
#define BIOCSETBUFMODE  _IOW('B', 126, uint32_t)
#define BIOCGETZMAX     _IOR('B', 127, size_t)
#define BIOCROTZBUF     _IOR('B', 128, struct bpf_zbuf)
#define BIOCSETZBUF     _IOW('B', 129, struct bpf_zbuf)
#define BIOCSETFNR      _IOW('B', 130, struct bpf_program)
#define BIOCGTSTAMP     _IOR('B', 131, uint32_t)
#define BIOCSTSTAMP     _IOW('B', 132, uint32_t)

/* Obsolete */
#define BIOCGSEESENT    BIOCGDIRECTION
#define BIOCSSEESENT    BIOCSDIRECTION

/* Packet directions */
enum bpf_direction
{
    BPF_D_IN,    /* See incoming packets */
    BPF_D_INOUT,    /* See incoming and outgoing packets */
    BPF_D_OUT    /* See outgoing packets */
};

/* Time stamping functions */
enum
{
    BPF_T_MICROTIME                 = 0x0000,
    BPF_T_NANOTIME                  = 0x0001,
    BPF_T_BINTIME                   = 0x0002,
    BPF_T_NONE                      = 0x0003,
    BPF_T_FORMAT_MASK               = 0x0003,
    BPF_T_NORMAL                    = 0x0000,
    BPF_T_FAST                      = 0x0100,
    BPF_T_MONOTONIC                 = 0x0200,
    BPF_T_MONOTONIC_FAST            = (BPF_T_FAST | BPF_T_MONOTONIC),
    BPF_T_FLAG_MASK                 = 0x0300
};

#define BPF_T_FORMAT(t)     ((t) & BPF_T_FORMAT_MASK)
#define BPF_T_FLAG(t)       ((t) & BPF_T_FLAG_MASK)
#define BPF_T_VALID(t) \
    ((t) == BPF_T_NONE || (BPF_T_FORMAT(t) != BPF_T_NONE &&    \
    ((t) & ~(BPF_T_FORMAT_MASK | BPF_T_FLAG_MASK)) == 0))

enum
{
    BPF_T_MICROTIME_FAST            = (BPF_T_MICROTIME | BPF_T_FAST),
    BPF_T_NANOTIME_FAST             = (BPF_T_NANOTIME | BPF_T_FAST),
    BPF_T_BINTIME_FAST              = (BPF_T_BINTIME | BPF_T_FAST),
    BPF_T_MICROTIME_MONOTONIC       = (BPF_T_MICROTIME | BPF_T_MONOTONIC),
    BPF_T_NANOTIME_MONOTONIC        = (BPF_T_NANOTIME | BPF_T_MONOTONIC),
    BPF_T_BINTIME_MONOTONIC         = (BPF_T_BINTIME | BPF_T_MONOTONIC),
    BPF_T_MICROTIME_MONOTONIC_FAST  = (BPF_T_MICROTIME | BPF_T_MONOTONIC_FAST),
    BPF_T_NANOTIME_MONOTONIC_FAST   = (BPF_T_NANOTIME | BPF_T_MONOTONIC_FAST),
    BPF_T_BINTIME_MONOTONIC_FAST    = (BPF_T_BINTIME | BPF_T_MONOTONIC_FAST)
};

/*
 * Structure prepended to each packet.
 */
struct bpf_ts
{
    bpf_int64       bt_sec;         /* seconds */
    bpf_u_int64     bt_frac;        /* fraction */
};
struct bpf_xhdr
{
    struct bpf_ts   bh_tstamp;      /* time stamp */
    bpf_u_int32     bh_caplen;      /* length of captured portion */
    bpf_u_int32     bh_datalen;     /* original length of packet */
    uint16_t        bh_hdrlen;      /* length of bpf header (this struct
                                       plus alignment padding) */
};
/* Obsolete */
struct bpf_hdr
{
    struct timeval  bh_tstamp;      /* time stamp */
    bpf_u_int32     bh_caplen;      /* length of captured portion */
    bpf_u_int32     bh_datalen;     /* original length of packet */
    uint16_t        bh_hdrlen;      /* length of bpf header (this struct
                                       plus alignment padding) */
};

/*
 * When using zero-copy BPF buffers, a shared memory header is present
 * allowing the kernel BPF implementation and user process to synchronize
 * without using system calls.  This structure defines that header.  When
 * accessing these fields, appropriate atomic operation and memory barriers
 * are required in order not to see stale or out-of-order data; see bpf(4)
 * for reference code to access these fields from userspace.
 *
 * The layout of this structure is critical, and must not be changed; if must
 * fit in a single page on all architectures.
 */
struct bpf_zbuf_header
{
    volatile uint32_t    bzh_kernel_gen;    /* Kernel generation number. */
    volatile uint32_t    bzh_kernel_len;    /* Length of data in the buffer. */
    volatile uint32_t    bzh_user_gen;      /* User generation number. */
    uint32_t _bzh_pad[5];
};

/*
 * Data-link level type codes.
 */
enum
{

    DLT_NULL                    = 0,    /* BSD loopback encapsulation */
    DLT_EN10MB                  = 1,    /* Ethernet (10Mb) */
    DLT_EN3MB                   = 2,    /* Experimental Ethernet (3Mb) */
    DLT_AX25                    = 3,    /* Amateur Radio AX.25 */
    DLT_PRONET                  = 4,    /* Proteon ProNET Token Ring */
    DLT_CHAOS                   = 5,    /* Chaos */
    DLT_IEEE802                 = 6,    /* IEEE 802 Networks */
    DLT_ARCNET                  = 7,    /* ARCNET */
    DLT_SLIP                    = 8,    /* Serial Line IP */
    DLT_PPP                     = 9,    /* Point-to-point Protocol */
    DLT_FDDI                    = 10,    /* FDDI */
    DLT_ATM_RFC1483             = 11,    /* LLC/SNAP encapsulated atm */
    DLT_RAW                     = 12,    /* raw IP */

/*
 * These are values from BSD/OS's "bpf.h".
 * These are not the same as the values from the traditional libpcap
 * "bpf.h"; however, these values shouldn't be generated by any
 * OS other than BSD/OS, so the correct values to use here are the
 * BSD/OS values.
 *
 * Platforms that have already assigned these values to other
 * DLT_ codes, however, should give these codes the values
 * from that platform, so that programs that use these codes will
 * continue to compile - even though they won't correctly read
 * files of these types.
 */
    DLT_SLIP_BSDOS              = 15,    /* BSD/OS Serial Line IP */
    DLT_PPP_BSDOS               = 16,    /* BSD/OS Point-to-point Protocol */

    DLT_ATM_CLIP                = 19,    /* Linux Classical-IP over ATM */

/*
 * These values are defined by NetBSD; other platforms should refrain from
 * using them for other purposes, so that NetBSD savefiles with link
 * types of 50 or 51 can be read as this type on all platforms.
 */
    DLT_PPP_SERIAL              = 50,    /* PPP over serial with HDLC encapsulation */
    DLT_PPP_ETHER               = 51,    /* PPP over Ethernet */

/*
 * Reserved for the Symantec Enterprise Firewall.
 */
    DLT_SYMANTEC_FIREWALL       = 99,

/*
 * Values between 100 and 103 are used in capture file headers as
 * link-layer header type LINKTYPE_ values corresponding to DLT_ types
 * that differ between platforms; don't use those values for new DLT_
 * new types.
 */

/*
 * Values starting with 104 are used for newly-assigned link-layer
 * header type values; for those link-layer header types, the DLT_
 * value returned by pcap_datalink() and passed to pcap_open_dead(),
 * and the LINKTYPE_ value that appears in capture files, are the
 * same.
 *
 * DLT_MATCHING_MIN is the lowest such value; DLT_MATCHING_MAX is
 * the highest such value.
 */
    DLT_MATCHING_MIN            = 104,

/*
 * This value was defined by libpcap 0.5; platforms that have defined
 * it with a different value should define it here with that value -
 * a link type of 104 in a save file will be mapped to DLT_C_HDLC,
 * whatever value that happens to be, so programs will correctly
 * handle files with that link type regardless of the value of
 * DLT_C_HDLC.
 *
 * The name DLT_C_HDLC was used by BSD/OS; we use that name for source
 * compatibility with programs written for BSD/OS.
 *
 * libpcap 0.5 defined it as DLT_CHDLC; we define DLT_CHDLC as well,
 * for source compatibility with programs written for libpcap 0.5.
 */
    DLT_C_HDLC                  = 104,    /* Cisco HDLC */
    DLT_CHDLC                   = DLT_C_HDLC,

    DLT_IEEE802_11              = 105,    /* IEEE 802.11 wireless */

/*
 * Values between 106 and 107 are used in capture file headers as
 * link-layer types corresponding to DLT_ types that might differ
 * between platforms; don't use those values for new DLT_ new types.
 */

/*
 * Frame Relay; BSD/OS has a DLT_FR with a value of 11, but that collides
 * with other values.
 * DLT_FR and DLT_FRELAY packets start with the Q.922 Frame Relay header
 * (DLCI, etc.).
 */
    DLT_FRELAY                  = 107,

/*
 * OpenBSD DLT_LOOP, for loopback devices; it's like DLT_NULL, except
 * that the AF_ type in the link-layer header is in network byte order.
 *
 * OpenBSD defines it as 12, but that collides with DLT_RAW, so we
 * define it as 108 here.  If OpenBSD picks up this file, it should
 * define DLT_LOOP as 12 in its version, as per the comment above -
 * and should not use 108 as a DLT_ value.
 */
    DLT_LOOP                    = 108,

/*
 * Values between 109 and 112 are used in capture file headers as
 * link-layer types corresponding to DLT_ types that might differ
 * between platforms; don't use those values for new DLT_ new types.
 */

/*
 * Encapsulated packets for IPsec; DLT_ENC is 13 in OpenBSD, but that's
 * DLT_SLIP_BSDOS in NetBSD, so we don't use 13 for it in OSes other
 * than OpenBSD.
 */
    DLT_ENC                     = 109,

/*
 * This is for Linux cooked sockets.
 */
    DLT_LINUX_SLL               = 113,

/*
 * Apple LocalTalk hardware.
 */
    DLT_LTALK                   = 114,

/*
 * Acorn Econet.
 */
    DLT_ECONET                  = 115,

/*
 * Reserved for use with OpenBSD ipfilter.
 */
    DLT_IPFILTER                = 116,

/*
 * Reserved for use in capture-file headers as a link-layer type
 * corresponding to OpenBSD DLT_PFLOG; DLT_PFLOG is 17 in OpenBSD,
 * but that's DLT_LANE8023 in SuSE 6.3, so we can't use 17 for it
 * in capture-file headers.
 */
    DLT_PFLOG                   = 117,

/*
 * Registered for Cisco-internal use.
 */
    DLT_CISCO_IOS               = 118,

/*
 * Reserved for 802.11 cards using the Prism II chips, with a link-layer
 * header including Prism monitor mode information plus an 802.11
 * header.
 */
    DLT_PRISM_HEADER            = 119,

/*
 * Reserved for Aironet 802.11 cards, with an Aironet link-layer header
 * (see Doug Ambrisko's FreeBSD patches).
 */
    DLT_AIRONET_HEADER          = 120,

/*
 * Reserved for use by OpenBSD's pfsync device.
 */
    DLT_PFSYNC                  = 121,

/*
 * Reserved for Siemens HiPath HDLC. XXX
 */
    DLT_HHDLC                   = 121,

/*
 * Reserved for RFC 2625 IP-over-Fibre Channel.
 */
    DLT_IP_OVER_FC              = 122,

/*
 * Reserved for Full Frontal ATM on Solaris.
 */
    DLT_SUNATM                  = 123,

/*
 * Reserved as per request from Kent Dahlgren <kent@praesum.com>
 * for private use.
 */
    DLT_RIO                     = 124,    /* RapidIO */
    DLT_PCI_EXP                 = 125,    /* PCI Express */
    DLT_AURORA                  = 126,    /* Xilinx Aurora link layer */

/*
 * BSD header for 802.11 plus a number of bits of link-layer information
 * including radio information.
 */
#ifndef DLT_IEEE802_11_RADIO
    DLT_IEEE802_11_RADIO        = 127,
#endif

/*
 * Reserved for TZSP encapsulation.
 */
    DLT_TZSP                    = 128,    /* Tazmen Sniffer Protocol */

/*
 * Reserved for Linux ARCNET.
 */
    DLT_ARCNET_LINUX            = 129,

/*
 * Juniper-private data link types.
 */
    DLT_JUNIPER_MLPPP           = 130,
    DLT_JUNIPER_MLFR            = 131,
    DLT_JUNIPER_ES              = 132,
    DLT_JUNIPER_GGSN            = 133,
    DLT_JUNIPER_MFR             = 134,
    DLT_JUNIPER_ATM2            = 135,
    DLT_JUNIPER_SERVICES        = 136,
    DLT_JUNIPER_ATM1            = 137,

/*
 * Apple IP-over-IEEE 1394, as per a request from Dieter Siegmund
 * <dieter@apple.com>.  The header that's presented is an Ethernet-like
 * header:
 *
 *    #define FIREWIRE_EUI64_LEN    8
 *    struct firewire_header {
 *        u_char  firewire_dhost[FIREWIRE_EUI64_LEN];
 *        u_char  firewire_shost[FIREWIRE_EUI64_LEN];
 *        u_short firewire_type;
 *    };
 *
 * with "firewire_type" being an Ethernet type value, rather than,
 * for example, raw GASP frames being handed up.
 */
    DLT_APPLE_IP_OVER_IEEE1394  = 138,

/*
 * Various SS7 encapsulations, as per a request from Jeff Morriss
 * <jeff.morriss[AT]ulticom.com> and subsequent discussions.
 */
    DLT_MTP2_WITH_PHDR          = 139,    /* pseudo-header with various info, followed by MTP2 */
    DLT_MTP2                    = 140,    /* MTP2, without pseudo-header */
    DLT_MTP3                    = 141,    /* MTP3, without pseudo-header or MTP2 */
    DLT_SCCP                    = 142,    /* SCCP, without pseudo-header or MTP2 or MTP3 */

/*
 * Reserved for DOCSIS.
 */
    DLT_DOCSIS                  = 143,

/*
 * Reserved for Linux IrDA.
 */
    DLT_LINUX_IRDA              = 144,

/*
 * Reserved for IBM SP switch and IBM Next Federation switch.
 */
    DLT_IBM_SP                  = 145,
    DLT_IBM_SN                  = 146,

/*
 * Reserved for private use.  If you have some link-layer header type
 * that you want to use within your organization, with the capture files
 * using that link-layer header type not ever be sent outside your
 * organization, you can use these values.
 *
 * No libpcap release will use these for any purpose, nor will any
 * tcpdump release use them, either.
 *
 * Do *NOT* use these in capture files that you expect anybody not using
 * your private versions of capture-file-reading tools to read; in
 * particular, do *NOT* use them in products, otherwise you may find that
 * people won't be able to use tcpdump, or snort, or Ethereal, or... to
 * read capture files from your firewall/intrusion detection/traffic
 * monitoring/etc. appliance, or whatever product uses that DLT_ value,
 * and you may also find that the developers of those applications will
 * not accept patches to let them read those files.
 *
 * Also, do not use them if somebody might send you a capture using them
 * for *their* private type and tools using them for *your* private type
 * would have to read them.
 *
 * Instead, ask "tcpdump-workers@tcpdump.org" for a new DLT_ value,
 * as per the comment above, and use the type you're given.
 */
    DLT_USER0                   = 147,
    DLT_USER1                   = 148,
    DLT_USER2                   = 149,
    DLT_USER3                   = 150,
    DLT_USER4                   = 151,
    DLT_USER5                   = 152,
    DLT_USER6                   = 153,
    DLT_USER7                   = 154,
    DLT_USER8                   = 155,
    DLT_USER9                   = 156,
    DLT_USER10                  = 157,
    DLT_USER11                  = 158,
    DLT_USER12                  = 159,
    DLT_USER13                  = 160,
    DLT_USER14                  = 161,
    DLT_USER15                  = 162,

/*
 * For future use with 802.11 captures - defined by AbsoluteValue
 * Systems to store a number of bits of link-layer information
 * including radio information:
 *
 *    http://www.shaftnet.org/~pizza/software/capturefrm.txt
 *
 * but it might be used by some non-AVS drivers now or in the
 * future.
 */
    DLT_IEEE802_11_RADIO_AVS    = 163,    /* 802.11 plus AVS radio header */

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.  The DLT_s are used
 * for passing on chassis-internal metainformation such as
 * QOS profiles, etc..
 */
    DLT_JUNIPER_MONITOR         = 164,

/*
 * Reserved for BACnet MS/TP.
 */
    DLT_BACNET_MS_TP            = 165,

/*
 * Another PPP variant as per request from Karsten Keil <kkeil@suse.de>.
 *
 * This is used in some OSes to allow a kernel socket filter to distinguish
 * between incoming and outgoing packets, on a socket intended to
 * supply pppd with outgoing packets so it can do dial-on-demand and
 * hangup-on-lack-of-demand; incoming packets are filtered out so they
 * don't cause pppd to hold the connection up (you don't want random
 * input packets such as port scans, packets from old lost connections,
 * etc. to force the connection to stay up).
 *
 * The first byte of the PPP header (0xff03) is modified to accomodate
 * the direction - 0x00 = IN, 0x01 = OUT.
 */
    DLT_PPP_PPPD                = 166,

/*
 * Names for backwards compatibility with older versions of some PPP
 * software; new software should use DLT_PPP_PPPD.
 */
    DLT_PPP_WITH_DIRECTION      = DLT_PPP_PPPD,
    DLT_LINUX_PPP_WITHDIRECTION = DLT_PPP_PPPD,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.  The DLT_s are used
 * for passing on chassis-internal metainformation such as
 * QOS profiles, cookies, etc..
 */
    DLT_JUNIPER_PPPOE           = 167,
    DLT_JUNIPER_PPPOE_ATM       = 168,

    DLT_GPRS_LLC                = 169,    /* GPRS LLC */
    DLT_GPF_T                   = 170,    /* GPF-T (ITU-T G.7041/Y.1303) */
    DLT_GPF_F                   = 171,    /* GPF-F (ITU-T G.7041/Y.1303) */

/*
 * Requested by Oolan Zimmer <oz@gcom.com> for use in Gcom's T1/E1 line
 * monitoring equipment.
 */
    DLT_GCOM_T1E1               = 172,
    DLT_GCOM_SERIAL             = 173,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.  The DLT_ is used
 * for internal communication to Physical Interface Cards (PIC)
 */
    DLT_JUNIPER_PIC_PEER        = 174,

/*
 * Link types requested by Gregor Maier <gregor@endace.com> of Endace
 * Measurement Systems.  They add an ERF header (see
 * http://www.endace.com/support/EndaceRecordFormat.pdf) in front of
 * the link-layer header.
 */
    DLT_ERF_ETH                 = 175,    /* Ethernet */
    DLT_ERF_POS                 = 176,    /* Packet-over-SONET */

/*
 * Requested by Daniele Orlandi <daniele@orlandi.com> for raw LAPD
 * for vISDN (http://www.orlandi.com/visdn/).  Its link-layer header
 * includes additional information before the LAPD header, so it's
 * not necessarily a generic LAPD header.
 */
    DLT_LINUX_LAPD              = 177,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 * The DLT_ are used for prepending meta-information
 * like interface index, interface name
 * before standard Ethernet, PPP, Frelay & C-HDLC Frames
 */
    DLT_JUNIPER_ETHER           = 178,
    DLT_JUNIPER_PPP             = 179,
    DLT_JUNIPER_FRELAY          = 180,
    DLT_JUNIPER_CHDLC           = 181,

/*
 * Multi Link Frame Relay (FRF.16)
 */
    DLT_MFR                     = 182,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 * The DLT_ is used for internal communication with a
 * voice Adapter Card (PIC)
 */
    DLT_JUNIPER_VP              = 183,

/*
 * Arinc 429 frames.
 * DLT_ requested by Gianluca Varenni <gianluca.varenni@cacetech.com>.
 * Every frame contains a 32bit A429 label.
 * More documentation on Arinc 429 can be found at
 * http://www.condoreng.com/support/downloads/tutorials/ARINCTutorial.pdf
 */
    DLT_A429                    = 184,

/*
 * Arinc 653 Interpartition Communication messages.
 * DLT_ requested by Gianluca Varenni <gianluca.varenni@cacetech.com>.
 * Please refer to the A653-1 standard for more information.
 */
    DLT_A653_ICM                = 185,

/*
 * USB packets, beginning with a USB setup header; requested by
 * Paolo Abeni <paolo.abeni@email.it>.
 */
    DLT_USB                     = 186,

/*
 * Bluetooth HCI UART transport layer (part H:4); requested by
 * Paolo Abeni.
 */
    DLT_BLUETOOTH_HCI_H4        = 187,

/*
 * IEEE 802.16 MAC Common Part Sublayer; requested by Maria Cruz
 * <cruz_petagay@bah.com>.
 */
    DLT_IEEE802_16_MAC_CPS      = 188,

/*
 * USB packets, beginning with a Linux USB header; requested by
 * Paolo Abeni <paolo.abeni@email.it>.
 */
    DLT_USB_LINUX               = 189,

/*
 * Controller Area Network (CAN) v. 2.0B packets.
 * DLT_ requested by Gianluca Varenni <gianluca.varenni@cacetech.com>.
 * Used to dump CAN packets coming from a CAN Vector board.
 * More documentation on the CAN v2.0B frames can be found at
 * http://www.can-cia.org/downloads/?269
 */
    DLT_CAN20B                  = 190,

/*
 * IEEE 802.15.4, with address fields padded, as is done by Linux
 * drivers; requested by Juergen Schimmer.
 */
    DLT_IEEE802_15_4_LINUX      = 191,

/*
 * Per Packet Information encapsulated packets.
 * DLT_ requested by Gianluca Varenni <gianluca.varenni@cacetech.com>.
 */
    DLT_PPI                     = 192,

/*
 * Header for 802.16 MAC Common Part Sublayer plus a radiotap radio header;
 * requested by Charles Clancy.
 */
    DLT_IEEE802_16_MAC_CPS_RADIO = 193,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 * The DLT_ is used for internal communication with a
 * integrated service module (ISM).
 */
    DLT_JUNIPER_ISM             = 194,

/*
 * IEEE 802.15.4, exactly as it appears in the spec (no padding, no
 * nothing); requested by Mikko Saarnivala <mikko.saarnivala@sensinode.com>.
 */
    DLT_IEEE802_15_4            = 195,

/*
 * Various link-layer types, with a pseudo-header, for SITA
 * (http://www.sita.aero/); requested by Fulko Hew (fulko.hew@gmail.com).
 */
    DLT_SITA                    = 196,

/*
 * Various link-layer types, with a pseudo-header, for Endace DAG cards;
 * encapsulates Endace ERF records.  Requested by Stephen Donnelly
 * <stephen@endace.com>.
 */
    DLT_ERF                     = 197,

/*
 * Special header prepended to Ethernet packets when capturing from a
 * u10 Networks board.  Requested by Phil Mulholland
 * <phil@u10networks.com>.
 */
    DLT_RAIF1                   = 198,

/*
 * IPMB packet for IPMI, beginning with the I2C slave address, followed
 * by the netFn and LUN, etc..  Requested by Chanthy Toeung
 * <chanthy.toeung@ca.kontron.com>.
 */
    DLT_IPMB                    = 199,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 * The DLT_ is used for capturing data on a secure tunnel interface.
 */
    DLT_JUNIPER_ST              = 200,

/*
 * Bluetooth HCI UART transport layer (part H:4), with pseudo-header
 * that includes direction information; requested by Paolo Abeni.
 */
    DLT_BLUETOOTH_HCI_H4_WITH_PHDR = 201,

/*
 * AX.25 packet with a 1-byte KISS header; see
 *
 *      http://www.ax25.net/kiss.htm
 *
 * as per Richard Stearn <richard@rns-stearn.demon.co.uk>.
 */
    DLT_AX25_KISS               = 202,

/*
 * LAPD packets from an ISDN channel, starting with the address field,
 * with no pseudo-header.
 * Requested by Varuna De Silva <varunax@gmail.com>.
 */
    DLT_LAPD                    = 203,

/*
 * Variants of various link-layer headers, with a one-byte direction
 * pseudo-header prepended - zero means "received by this host",
 * non-zero (any non-zero value) means "sent by this host" - as per
 * Will Barker <w.barker@zen.co.uk>.
 */
    DLT_PPP_WITH_DIR            = 204,     /* PPP - don't confuse with DLT_PPP_WITH_DIRECTION */
    DLT_C_HDLC_WITH_DIR         = 205,     /* Cisco HDLC */
    DLT_FRELAY_WITH_DIR         = 206,     /* Frame Relay */
    DLT_LAPB_WITH_DIR           = 207,     /* LAPB */

/*
 * 208 is reserved for an as-yet-unspecified proprietary link-layer
 * type, as requested by Will Barker.
 */

/*
 * IPMB with a Linux-specific pseudo-header; as requested by Alexey Neyman
 * <avn@pigeonpoint.com>.
 */
    DLT_IPMB_LINUX              = 209,

/*
 * FlexRay automotive bus - http://www.flexray.com/ - as requested
 * by Hannes Kaelber <hannes.kaelber@x2e.de>.
 */
    DLT_FLEXRAY                 = 210,

/*
 * Media Oriented Systems Transport (MOST) bus for multimedia
 * transport - http://www.mostcooperation.com/ - as requested
 * by Hannes Kaelber <hannes.kaelber@x2e.de>.
 */
    DLT_MOST                    = 211,

/*
 * Local Interconnect Network (LIN) bus for vehicle networks -
 * http://www.lin-subbus.org/ - as requested by Hannes Kaelber
 * <hannes.kaelber@x2e.de>.
 */
    DLT_LIN                     = 212,

/*
 * X2E-private data link type used for serial line capture,
 * as requested by Hannes Kaelber <hannes.kaelber@x2e.de>.
 */
    DLT_X2E_SERIAL              = 213,

/*
 * X2E-private data link type used for the Xoraya data logger
 * family, as requested by Hannes Kaelber <hannes.kaelber@x2e.de>.
 */
    DLT_X2E_XORAYA              = 214,

/*
 * IEEE 802.15.4, exactly as it appears in the spec (no padding, no
 * nothing), but with the PHY-level data for non-ASK PHYs (4 octets
 * of 0 as preamble, one octet of SFD, one octet of frame length+
 * reserved bit, and then the MAC-layer data, starting with the
 * frame control field).
 *
 * Requested by Max Filippov <jcmvbkbc@gmail.com>.
 */
    DLT_IEEE802_15_4_NONASK_PHY = 215,

/*
 * David Gibson <david@gibson.dropbear.id.au> requested this for
 * captures from the Linux kernel /dev/input/eventN devices. This
 * is used to communicate keystrokes and mouse movements from the
 * Linux kernel to display systems, such as Xorg.
 */
       DLT_LINUX_EVDEV          = 216,

/*
 * GSM Um and Abis interfaces, preceded by a "gsmtap" header.
 *
 * Requested by Harald Welte <laforge@gnumonks.org>.
 */
       DLT_GSMTAP_UM            = 217,
       DLT_GSMTAP_ABIS          = 218,

/*
 * MPLS, with an MPLS label as the link-layer header.
 * Requested by Michele Marchetto <michele@openbsd.org> on behalf
 * of OpenBSD.
 */
       DLT_MPLS                 = 219,

/*
 * USB packets, beginning with a Linux USB header, with the USB header
 * padded to 64 bytes; required for memory-mapped access.
 */
       DLT_USB_LINUX_MMAPPED    = 220,

/*
 * DECT packets, with a pseudo-header; requested by
 * Matthias Wenzel <tcpdump@mazzoo.de>.
 */
       DLT_DECT                 = 221,
/*
 * From: "Lidwa, Eric (GSFC-582.0)[SGT INC]" <eric.lidwa-1@nasa.gov>
 * Date: Mon, 11 May 2009 11:18:30 -0500
 *
 * DLT_AOS. We need it for AOS Space Data Link Protocol.
 *   I have already written dissectors for but need an OK from
 *   legal before I can submit a patch.
 *
 */
       DLT_AOS                  = 222,

/*
 * Wireless HART (Highway Addressable Remote Transducer)
 * From the HART Communication Foundation
 * IES/PAS 62591
 *
 * Requested by Sam Roberts <vieuxtech@gmail.com>.
 */
       DLT_WIHART               = 223,

/*
 * Fibre Channel FC-2 frames, beginning with a Frame_Header.
 * Requested by Kahou Lei <kahou82@gmail.com>.
 */
       DLT_FC_2                 = 224,

/*
 * Fibre Channel FC-2 frames, beginning with an encoding of the
 * SOF, and ending with an encoding of the EOF.
 *
 * The encodings represent the frame delimiters as 4-byte sequences
 * representing the corresponding ordered sets, with K28.5
 * represented as 0xBC, and the D symbols as the corresponding
 * byte values; for example, SOFi2, which is K28.5 - D21.5 - D1.2 - D21.2,
 * is represented as 0xBC 0xB5 0x55 0x55.
 *
 * Requested by Kahou Lei <kahou82@gmail.com>.
 */
       DLT_FC_2_WITH_FRAME_DELIMS = 225,
/*
 * Solaris ipnet pseudo-header; requested by Darren Reed <Darren.Reed@Sun.COM>.
 *
 * The pseudo-header starts with a one-byte version number; for version 2,
 * the pseudo-header is:
 *
 * struct dl_ipnetinfo {
 *     u_int8_t   dli_version;
 *     u_int8_t   dli_family;
 *     u_int16_t  dli_htype;
 *     u_int32_t  dli_pktlen;
 *     u_int32_t  dli_ifindex;
 *     u_int32_t  dli_grifindex;
 *     u_int32_t  dli_zsrc;
 *     u_int32_t  dli_zdst;
 * };
 *
 * dli_version is 2 for the current version of the pseudo-header.
 *
 * dli_family is a Solaris address family value, so it's 2 for IPv4
 * and 26 for IPv6.
 *
 * dli_htype is a "hook type" - 0 for incoming packets, 1 for outgoing
 * packets, and 2 for packets arriving from another zone on the same
 * machine.
 *
 * dli_pktlen is the length of the packet data following the pseudo-header
 * (so the captured length minus dli_pktlen is the length of the
 * pseudo-header, assuming the entire pseudo-header was captured).
 *
 * dli_ifindex is the interface index of the interface on which the
 * packet arrived.
 *
 * dli_grifindex is the group interface index number (for IPMP interfaces).
 *
 * dli_zsrc is the zone identifier for the source of the packet.
 *
 * dli_zdst is the zone identifier for the destination of the packet.
 *
 * A zone number of 0 is the global zone; a zone number of 0xffffffff
 * means that the packet arrived from another host on the network, not
 * from another zone on the same machine.
 *
 * An IPv4 or IPv6 datagram follows the pseudo-header; dli_family indicates
 * which of those it is.
 */
       DLT_IPNET                = 226,

/*
 * CAN (Controller Area Network) frames, with a pseudo-header as supplied
 * by Linux SocketCAN.  See Documentation/networking/can.txt in the Linux
 * source.
 *
 * Requested by Felix Obenhuber <felix@obenhuber.de>.
 */
       DLT_CAN_SOCKETCAN        = 227,

/*
 * Raw IPv4/IPv6; different from DLT_RAW in that the DLT_ value specifies
 * whether it's v4 or v6.  Requested by Darren Reed <Darren.Reed@Sun.COM>.
 */
    DLT_IPV4                    = 228,
    DLT_IPV6                    = 229,

/*
 * IEEE 802.15.4, exactly as it appears in the spec (no padding, no
 * nothing), and with no FCS at the end of the frame; requested by
 * Jon Smirl <jonsmirl@gmail.com>.
 */
    DLT_IEEE802_15_4_NOFCS      = 230,

/*
 * Raw D-Bus:
 *
 *    http://www.freedesktop.org/wiki/Software/dbus
 *
 * messages:
 *
 *    http://dbus.freedesktop.org/doc/dbus-specification.html#message-protocol-messages
 *
 * starting with the endianness flag, followed by the message type, etc.,
 * but without the authentication handshake before the message sequence:
 *
 *    http://dbus.freedesktop.org/doc/dbus-specification.html#auth-protocol
 *
 * Requested by Martin Vidner <martin@vidner.net>.
 */
    DLT_DBUS                    = 231,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 */
    DLT_JUNIPER_VS              = 232,
    DLT_JUNIPER_SRX_E2E         = 233,
    DLT_JUNIPER_FIBRECHANNEL    = 234,

/*
 * DVB-CI (DVB Common Interface for communication between a PC Card
 * module and a DVB receiver).  See
 *
 *    http://www.kaiser.cx/pcap-dvbci.html
 *
 * for the specification.
 *
 * Requested by Martin Kaiser <martin@kaiser.cx>.
 */
    DLT_DVB_CI                  = 235,

/*
 * Variant of 3GPP TS 27.010 multiplexing protocol (similar to, but
 * *not* the same as, 27.010).  Requested by Hans-Christoph Schemmel
 * <hans-christoph.schemmel@cinterion.com>.
 */
    DLT_MUX27010                = 236,

/*
 * STANAG 5066 D_PDUs.  Requested by M. Baris Demiray
 * <barisdemiray@gmail.com>.
 */
    DLT_STANAG_5066_D_PDU       = 237,

/*
 * Juniper-private data link type, as per request from
 * Hannes Gredler <hannes@juniper.net>.
 */
    DLT_JUNIPER_ATM_CEMIC       = 238,

/*
 * NetFilter LOG messages
 * (payload of netlink NFNL_SUBSYS_ULOG/NFULNL_MSG_PACKET packets)
 *
 * Requested by Jakub Zawadzki <darkjames-ws@darkjames.pl>
 */
    DLT_NFLOG                   = 239,

/*
 * Hilscher Gesellschaft fuer Systemautomation mbH link-layer type
 * for Ethernet packets with a 4-byte pseudo-header and always
 * with the payload including the FCS, as supplied by their
 * netANALYZER hardware and software.
 *
 * Requested by Holger P. Frommer <HPfrommer@hilscher.com>
 */
    DLT_NETANALYZER             = 240,

/*
 * Hilscher Gesellschaft fuer Systemautomation mbH link-layer type
 * for Ethernet packets with a 4-byte pseudo-header and FCS and
 * with the Ethernet header preceded by 7 bytes of preamble and
 * 1 byte of SFD, as supplied by their netANALYZER hardware and
 * software.
 *
 * Requested by Holger P. Frommer <HPfrommer@hilscher.com>
 */
    DLT_NETANALYZER_TRANSPARENT = 241,

/*
 * IP-over-Infiniband, as specified by RFC 4391.
 *
 * Requested by Petr Sumbera <petr.sumbera@oracle.com>.
 */
    DLT_IPOIB                   = 242,

/*
 * MPEG-2 transport stream (ISO 13818-1/ITU-T H.222.0).
 *
 * Requested by Guy Martin <gmsoft@tuxicoman.be>.
 */
    DLT_MPEG_2_TS               = 243,

/*
 * ng4T GmbH's UMTS Iub/Iur-over-ATM and Iub/Iur-over-IP format as
 * used by their ng40 protocol tester.
 *
 * Requested by Jens Grimmer <jens.grimmer@ng4t.com>.
 */
    DLT_NG40                    = 244,

/*
 * Pseudo-header giving adapter number and flags, followed by an NFC
 * (Near-Field Communications) Logical Link Control Protocol (LLCP) PDU,
 * as specified by NFC Forum Logical Link Control Protocol Technical
 * Specification LLCP 1.1.
 *
 * Requested by Mike Wakerly <mikey@google.com>.
 */
    DLT_NFC_LLCP                = 245,

/*
 * 245 is used as LINKTYPE_PFSYNC; do not use it for any other purpose.
 *
 * DLT_PFSYNC has different values on different platforms, and all of
 * them collide with something used elsewhere.  On platforms that
 * don't already define it, define it as 245.
 */
    DLT_MATCHING_MAX            = 246    /* highest value in the "matching" range */
};

/*
 * DLT and savefile link type values are split into a class and
 * a member of that class.  A class value of 0 indicates a regular
 * DLT_/LINKTYPE_ value.
 */
#define DLT_CLASS(x)            ((x) & 0x03ff0000)

/*
 * The instruction encodings.
 */

/* instruction classes */
#define BPF_CLASS(code)     ((code) & 0x07)

/* ld/ldx fields */
#define BPF_SIZE(code)      ((code) & 0x18)
#define BPF_MODE(code)      ((code) & 0xe0)

/* alu/jmp fields */
#define BPF_OP(code)        ((code) & 0xf0)

#define BPF_SRC(code)       ((code) & 0x08)

/* ret - BPF_K and BPF_X also apply */
#define BPF_RVAL(code)      ((code) & 0x18)

/* misc */
#define BPF_MISCOP(code)    ((code) & 0xf8)

enum
{
    BPF_ETHCOOK     = 0x00,
    BPF_WHOLEPACKET = 0xff,
    BPF_LD          = 0x00,
    BPF_LDX         = 0x01,
    BPF_ST          = 0x02,
    BPF_STX         = 0x03,
    BPF_ALU         = 0x04,
    BPF_JMP         = 0x05,
    BPF_RET         = 0x06,
    BPF_MISC        = 0x07,
    BPF_W           = 0x00,
    BPF_H           = 0x08,
    BPF_B           = 0x10,
    BPF_IMM         = 0x00,
    BPF_ABS         = 0x20,
    BPF_IND         = 0x40,
    BPF_MEM         = 0x60,
    BPF_LEN         = 0x80,
    BPF_MSH         = 0xa0,
    BPF_ADD         = 0x00,
    BPF_SUB         = 0x10,
    BPF_MUL         = 0x20,
    BPF_DIV         = 0x30,
    BPF_OR          = 0x40,
    BPF_AND         = 0x50,
    BPF_LSH         = 0x60,
    BPF_RSH         = 0x70,
    BPF_NEG         = 0x80,
    BPF_JA          = 0x00,
    BPF_JEQ         = 0x10,
    BPF_JGT         = 0x20,
    BPF_JGE         = 0x30,
    BPF_JSET        = 0x40,
    BPF_K           = 0x00,
    BPF_X           = 0x08,
    BPF_A           = 0x10,
    BPF_TAX         = 0x00,
    BPF_TXA         = 0x80,
};

/*
 * Macros for insn array initializers.
 */
#define BPF_STMT(code, k)           { (uint16_t)(code), 0, 0, k }
#define BPF_JUMP(code, k, jt, jf)   { (uint16_t)(code), jt, jf, k }

/*
 * Structure to retrieve available DLTs for the interface.
 */
struct bpf_dltlist
{
    uint32_t    bfl_len;        /* number of bfd_list array */
    uint32_t    *bfl_list;      /* array of DLTs */
};


/*
 * Number of scratch memory words (for BPF_LD|BPF_MEM and BPF_ST).
 */
enum
{
    BPF_MEMWORDS = 16
};
