﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief プロキシオブジェクト用のアロケータユーティリティを定義します。
*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/sf_MemoryResource.h>
#include <nn/sf/sf_LmemUtility.h>
#include <new>
#include <type_traits>
#include <mutex>
#include <nn/sf/detail/sf_StaticMutex.h>

namespace nn { namespace sf {

namespace detail {

class ProxyObjectAllocatorImpl
{
private:

    lmem::HeapCommonHead m_HeapCommonHead;
    UnitHeapMemoryResource m_MemoryResource;

public:

    explicit ProxyObjectAllocatorImpl(void* buffer, size_t bufferSize, size_t unitSize) NN_NOEXCEPT
        : m_MemoryResource(lmem::CreateUnitHeap(buffer, bufferSize, unitSize, lmem::CreationOption_ThreadSafe, DefaultAlignment, &m_HeapCommonHead))
    {
    }

    ~ProxyObjectAllocatorImpl() NN_NOEXCEPT
    {
        lmem::DestroyUnitHeap(m_MemoryResource.GetHandle());
    }

    MemoryResource* GetMemoryResource() NN_NOEXCEPT
    {
        return &m_MemoryResource;
    }
};

const Bit32 ProxyObjectAllocatorMagicNumber = 0x16354534;

}

/**
    @brief プロキシオブジェクトをアロケーションする際に必要な最大のバイト数を表す定数です。
*/
const size_t ProxyObjectAllocationSize = 64;

/**
    @brief プロキシオブジェクト専用のアロケータを作成するためのユーティリティです。

    @tparam ObjectCountMax 作成されたアロケータからアロケートできるプロキシオブジェクトの最大数を指定します。

    @details
     ObjectCountMax 個のプロキシオブジェクトを作成できるだけのアロケータを作成します。
     アロケータへのアクセスは GetMemoryResource() によって MemoryResource の形で行うことができます。
     この MemoryResource からは、ProxyObjectAllocationSize バイトを超えるメモリ確保はできません。

     この型の変数はグローバルに配置し、必ず NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER マクロを使用して初期化してください。
     ライブラリコードのロード完了時にこの変数の初期化が終わっていることが保証されます。

     さらに GetMemoryResource() を呼ぶ前には Initialize() を呼んで、アロケータを初期化する必要があります。

     Initialize() は複数回呼び出すことができ Finalize() を同じ回数だけ呼ぶまでの間、アロケータは有効です。

     このオブジェクトは、内部にアロケート用バッファを含むため、
     おおよそ ProxyObjectAllocationSize * ObjectCountMax バイトの比較的大きな領域を占有します。
     スタック上などに置かないようにご注意ください。

     フィールドメンバには直接アクセスしないでください。
     特に、現状は、lmem ライブラリのユニットヒープを実装に使用していますが、
     この実装は今後変更される可能性があります。
     関連するフィールドメンバに直接アクセスしないようにしてください。
*/
template <size_t ObjectCountMax>
struct ProxyObjectAllocator
{
private:

    void AssertInitializerUsed() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_magicNumber == detail::ProxyObjectAllocatorMagicNumber, "This needs to be initialized by NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER");
    }

public:

    detail::ProxyObjectAllocatorImpl* _pImpl;
    mutable detail::StaticMutex _mutex;
    int _initializeCount;
    Bit32 _magicNumber;
    typename std::aligned_storage<sizeof(detail::ProxyObjectAllocatorImpl)>::type _implBuffer;
    typename std::aligned_storage<ProxyObjectAllocationSize * (ObjectCountMax == 0 ? 1 : ObjectCountMax)>::type _buffer;

    /**
        @brief nn::sf::ProxyObjectAllocator のインスタンスの初期化用マクロです。
    */
    #define NN_SF_PROXY_OBJECT_ALLOCATOR_INITIALIZER \
        { \
            nullptr, \
            NN_SF_DETAIL_STATIC_MUTEX_INITIALIZER(false), \
            0, \
            ::nn::sf::detail::ProxyObjectAllocatorMagicNumber, \
            {}, \
            {}, \
        }

    /**
        @brief アロケータを初期化します。

        @post 初期化済み状態である。
    */
    void Initialize() NN_NOEXCEPT
    {
        AssertInitializerUsed();
        std::lock_guard<decltype(_mutex)> lk(_mutex);
        if (_initializeCount == 0)
        {
            this->_pImpl = new (&_implBuffer) detail::ProxyObjectAllocatorImpl(&_buffer, sizeof(_buffer), ProxyObjectAllocationSize);
            this->_initializeCount = 1;
        }
        else
        {
            ++_initializeCount;
        }
    }

    /**
        @brief アロケータを終了します。

        @pre (これまでの Finalize() の呼び出し回数) < (Initialize() の呼び出し回数)
        @pre アロケータから確保された全てのメモリが、解放されている。

        @post (Finalize() の呼び出し回数) == (Initialize() の呼び出し回数) のとき、未初期化状態となる
    */
    void Finalize() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(_initializeCount > 0, "Must be initialized.");
        std::lock_guard<decltype(_mutex)> lk(_mutex);
        --_initializeCount;
        if (_initializeCount == 0)
        {
            _pImpl->~ProxyObjectAllocatorImpl();
            this->_pImpl = nullptr;
        }
    }

    /**
        @brief アロケータへのアクセスを MemoryResource の形で取得します。

        @return アロケータに対応する MemoryResource を返します。

        @pre 初期化済み状態である。

        @details
         内部で保持するアロケータへの MemoryResource を取得します。

         この関数を呼ぶためには、あらかじめ Initialize() を呼んでおく必要があります。

         アロケーションポリシーとして MemoryResourceAllocationPolicy を指定した場合に、この関数の戻り値を使用できます。
         また MemoryResourceStaticAllocator::Initialize() の引数として使用することもできます。

        @see MemoryResource, Initialize(), Finalize()
    */
    MemoryResource* GetMemoryResource() const NN_NOEXCEPT
    {
        AssertInitializerUsed();
        NN_SDK_REQUIRES_NOT_NULL(_pImpl);
        return _pImpl->GetMemoryResource();
    }
};

}}
