﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスインターフェイスの出力引数を表すための型を定義します。
*/

#include <nn/nn_Common.h>
#include <type_traits>

namespace nn { namespace sf {

template <typename>
struct EnablesOut : public std::integral_constant<bool, false>
{
};

/**
    @brief サービスインターフェイスにおいて出力引数を表すクラステンプレートです。

    @tparam T 出力される値の型を指定します。
    @tparam Enabled 内部で使用される型のため、常に省略してください。

    @details
     このクラステンプレートは、出力される型 T に対して特殊化されています。

     - Trivial な型
     - SharedPointer @<Interface>
     - NativeHandle

     これらの型以外を型引数にとることはできません。
*/
template <typename T, typename Enabled = void>
class Out
{
    static_assert(!std::is_same<T, T>::value, "[SF-BASE-InvalidOutType:NotSupported] Out<T>: T is not supported.");
};

/**
    @brief Tirivial である型に対する Out の特殊化です。

    @tparam T 出力される値の Tirivial な型を指定します。
*/
template <typename T>
class Out<T, typename std::enable_if<std::is_trivial<T>::value || EnablesOut<T>::value>::type>
{
private:

    T* m_P;

public:

    /**
        @brief コンストラクタ: 指定したポインタで初期化します。

        @param[in] p 出力バッファを表すポインタ

        @details
         T* からの暗黙変換であるため、T のポインタを直接変換することが可能です。
    */
    NN_IMPLICIT Out(T* p) NN_NOEXCEPT
        : m_P(p)
    {
    }

    /**
        @brief 値をセットします。

        @param[in] x セットする値を指定します。
    */
    void Set(const T& x) const NN_NOEXCEPT
    {
        *this->m_P = x;
    }

    /**
        @brief セットされた値を取得します。

        @return セットされた値への const 参照を返します。
    */
    const T& Get() const NN_NOEXCEPT
    {
        return *m_P;
    }

    /**
        @brief 内部に設定されているポインタを直接取得します。

        @return 内部に設定されているポインタ。
    */
    T* GetPointer() const NN_NOEXCEPT
    {
        return m_P;
    }

    /**
        @brief -> 演算子: 内部に設定されているポインタを直接取得します。

        @return 内部に設定されているポインタを返します。
    */
    T* operator->() const NN_NOEXCEPT
    {
        return m_P;
    }

    /**
        @brief 逆参照演算子: 内部に設定されているポインタの逆参照を取得します。

        @return 内部に設定されているポインタの逆参照を返します。
    */
    T& operator*() const NN_NOEXCEPT
    {
        return *m_P;
    }

};

template <typename T>
class Out<T*>
{
    static_assert(!std::is_same<T, T>::value, "[SF-BASE-InvalidOutType:Pointer] Out<T*> is not supported.");
};

}}


namespace nn {

class Result;

namespace sf {

template <>
struct EnablesOut<Result> : public std::true_type
{
};

}

}
