﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスインターフェイスのインスタンスを生成するためのファクトリクラスを定義します。
*/

#include <nn/sf/sf_Types.h>
#include <nn/sf/impl/detail/sf_ImplTemplateBase.h>
#include <nn/sf/sf_ObjectImplFactory.h>
#include <memory>
#include <utility>
#include <nn/sf/detail/sf_CommonUtil.h>
#include <type_traits>
#include <nn/sf/sf_DefaultAllocationPolicy.h>
#include <nn/sf/sf_MemoryResource.h>

namespace nn { namespace sf {

namespace detail {

    template <typename T, typename Enabled = void>
    struct IsSmartPointer
        : public std::false_type
    {
    };

    template <typename T>
    struct IsSmartPointer<T, typename std::enable_if<T::IsTreatedAsSmartPointerBySf>::type>
        : public std::true_type
    {
    };

    template <typename T, typename Deleter>
    struct IsSmartPointer<std::unique_ptr<T, Deleter>>
        : public ::std::true_type
    {
    };

    template <typename T>
    struct IsSmartPointer<std::shared_ptr<T>>
        : public std::true_type
    {
    };

    template <typename Impl, typename Enabled = void>
    struct UnmanagedEmplacedImplHolderBaseGetter
    {
        typedef Impl Type;
    };

    template <typename Impl>
    struct UnmanagedEmplacedImplHolderBaseGetter<Impl, typename std::enable_if<std::is_abstract<Impl>::value>::type>
    {
        class Impl2 : public Impl
        {
        public:
            using Impl::Impl;
        private:
            virtual void AddReference() NN_NOEXCEPT NN_OVERRIDE {}
            virtual void Release() NN_NOEXCEPT NN_OVERRIDE {}
        };
        typedef Impl2 Type;
    };

    template <typename Impl>
    class UnmanagedEmplacedImplHolder
    {
        template <typename, typename, typename, typename, typename>
        friend class impl::detail::ImplTemplateBaseT;
    private:
        typedef typename UnmanagedEmplacedImplHolderBaseGetter<Impl>::Type Impl2;
        static_assert(!std::is_abstract<Impl2>::value, "[SF-BASE-InvalidImplType:Abstract] Impl is abstract.");
        Impl2 m_Impl;
        template <typename... Args>
        explicit UnmanagedEmplacedImplHolder(Args&&... args) NN_NOEXCEPT
            : m_Impl{std::forward<Args>(args)...}
        {
        }
    public:
        static Impl* GetImplPointer(UnmanagedEmplacedImplHolder* pHolder) NN_NOEXCEPT
        {
            return &pHolder->m_Impl;
        }
    };

    template <typename Impl, typename Enabled = void>
    class EmplacedImplHolder
        : private Impl
    {
        template <typename, typename, typename, typename, typename>
        friend class impl::detail::ImplTemplateBaseT;
    private:
        template <typename... Args>
        explicit EmplacedImplHolder(Args&&... args) NN_NOEXCEPT
             : Impl{std::forward<Args>(args)...}
        {
        }
    public:
        static Impl* GetImplPointer(EmplacedImplHolder* pHolder) NN_NOEXCEPT
        {
            return pHolder;
        }
        template <typename Interface>
        static Impl* GetEmplacedImplPointerImpl(const nn::sf::SharedPointer<Interface>& p) NN_NOEXCEPT
        {
            typedef impl::detail::ImplTemplateBase<Interface, Interface, EmplacedImplHolder, EmplacedImplHolder> Base;
            return static_cast<Base*>(p.Get());
        }
    };

    template <typename Impl>
    class EmplacedImplHolder<Impl, typename std::enable_if<IsSmartPointer<Impl>::value>::type>
        : private Impl
    {
        template <typename, typename, typename, typename, typename>
        friend class impl::detail::ImplTemplateBaseT;
    private:
        template <typename... Args>
        explicit EmplacedImplHolder(Args&&... args) NN_NOEXCEPT
             : Impl{std::forward<Args>(args)...}
        {
        }
    public:
        static auto GetImplPointer(EmplacedImplHolder* pHolder) NN_NOEXCEPT
        {
            return static_cast<Impl*>(pHolder)->operator->();
        }
        template <typename Interface>
        static Impl* GetEmplacedImplPointerImpl(const nn::sf::SharedPointer<Interface>& p) NN_NOEXCEPT
        {
            typedef impl::detail::ImplTemplateBase<Interface, Interface, EmplacedImplHolder, EmplacedImplHolder> Base;
            return static_cast<Base*>(p.Get());
        }
    };

    template <typename SmartPointer>
    using SmartPointerHolder = EmplacedImplHolder<SmartPointer>;

    template <typename T>
    class ManagedPointerHolder
    {
        template <typename, typename, typename, typename, typename>
        friend class impl::detail::ImplTemplateBaseT;
    private:
        T* m_P;
        explicit ManagedPointerHolder(T* p) NN_NOEXCEPT
            : m_P(p)
        {
        }
        ~ManagedPointerHolder() NN_NOEXCEPT
        {
            m_P->Release();
        }
        static T* GetImplPointer(ManagedPointerHolder* pHolder) NN_NOEXCEPT
        {
            return pHolder->m_P;
        }
    };

    template <typename T>
    class UnmanagedPointerHolder
    {
        template <typename, typename, typename, typename, typename>
        friend class impl::detail::ImplTemplateBaseT;
    private:
        T* m_P;
        explicit UnmanagedPointerHolder(T* p) NN_NOEXCEPT
            : m_P(p)
        {
        }
        static T* GetImplPointer(UnmanagedPointerHolder* pHolder) NN_NOEXCEPT
        {
            return pHolder->m_P;
        }
    };

}

/**
    @brief 寿命管理をしないインスタンスクラステンプレートです。

    @tparam Interface インターフェイスを指定します。
    @tparam Impl 実装クラスを指定します。

    @details
     このクラスは Interface を基底クラスに持ち、Interface の各メソッド関数を実装します。
     ISharedObject::AddReference() と ISharedObject::Release() はオーバロードしますが、これらは空であり、寿命管理は行われません。
     このクラスは内部に Impl クラスを保持し、初期化時にコンストラクトすることが可能です。

     このクラスのインスタンスは、グローバル管理領域にオブジェクトを置くなど、
     サービスフレームワーク外によってオブジェクトが有効であることが保証されている際に使用されます。
     このインスタンスをサービスフレームワークで使用する際には GetShared() によって共有ポインタを得てください。
*/
template <typename Interface, typename Impl>
class UnmanagedServiceObject final
    : public impl::detail::ImplTemplateBase<Interface, Interface, detail::UnmanagedEmplacedImplHolder<Impl>, detail::UnmanagedEmplacedImplHolder<Impl>>
{
private:

    typedef impl::detail::ImplTemplateBase<Interface, Interface, detail::UnmanagedEmplacedImplHolder<Impl>, detail::UnmanagedEmplacedImplHolder<Impl>> ImplBase;

public:

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
    /**
        @brief コンストラクタ: 引数を Impl のコンストラクタに転送して初期化します。

        @tparam Args Impl が受け取れる引数型列を指定します。
        @param[in] args 引数列を指定します。

        @details
         Impl(std::foward<Args>(args)...) 相当の呼び出しによって、内部に保持される Impl オブジェクトが初期化されます。
    */
    template <typename... Args>
    explicit UnmanagedServiceObject(Args&&... args) NN_NOEXCEPT;
#else
    using ImplBase::ImplBase;
#endif

    /**
        @brief オブジェクトの参照を増やします。
    */
    virtual void AddReference() NN_NOEXCEPT NN_OVERRIDE
    {
        // nop
    }

    /**
        @brief オブジェクトの参照を減らします。
    */
    virtual void Release() NN_NOEXCEPT NN_OVERRIDE
    {
        // nop
    }

    /**
        @brief 内部にある Impl オブジェクトへのポインタを取得します。

        @return 内部にある Impl オブジェクトへのポインタを返します。
    */
    Impl& GetImpl() NN_NOEXCEPT
    {
        return *detail::UnmanagedEmplacedImplHolder<Impl>::GetImplPointer(this);
    }

    /**
        @brief このオブジェクトに対する共有ポインタを作成します。

        @return このオブジェクトをに対する共有ポインタを返します。
    */
    SharedPointer<Interface> GetShared() NN_NOEXCEPT
    {
        return SharedPointer<Interface>(this, false);
    }

};

/**
    @brief ポインタによって与えられたオブジェクトを実装とするインスタンスクラステンプレートです。

    @tparam Interface インターフェイスを指定します。
    @tparam T 実装クラスを指定します。T* 型が Interface に適合する関数を持つ必要があります。

    @details
     このクラスは Interface を基底クラスに持ち、Interface の各メソッド関数を実装します。
     ISharedObject::AddReference() と ISharedObject::Release() はオーバロードしますが、これらは空であり、寿命管理は行われません。

     このクラスのインスタンスは、ポインタの指すオブジェクトがグローバル管理領域にオブジェクトを置かれているなど。
     サービスフレームワーク外によってオブジェクトが有効であることが保証されている際に使用されます。
     このインスタンスをサービスフレームワークで使用する際には GetShared() によって共有ポインタを得てください。
*/
template <typename Interface, typename T>
class UnmanagedServiceObjectByPointer final
    : public impl::detail::ImplTemplateBase<Interface, Interface, detail::UnmanagedPointerHolder<T>, detail::UnmanagedPointerHolder<T>>
{
private:

    typedef impl::detail::ImplTemplateBase<Interface, Interface, detail::UnmanagedPointerHolder<T>, detail::UnmanagedPointerHolder<T>> ImplBase;

public:

    /**
        @brief コンストラクタ: ポインタを使って初期化します。
    */
    explicit UnmanagedServiceObjectByPointer(T* p) NN_NOEXCEPT
        : ImplBase(p)
    {
    }

    /**
        @brief このオブジェクトに対する共有ポインタを作成します。

        @return このオブジェクトをに対する共有ポインタを返します。
    */
    SharedPointer<Interface> GetShared() NN_NOEXCEPT
    {
        return SharedPointer<Interface>(this, false);
    }

    virtual void AddReference() NN_NOEXCEPT NN_OVERRIDE
    {
        // nop
    }

    virtual void Release() NN_NOEXCEPT NN_OVERRIDE
    {
        // nop
    }

};

/**
    @brief 汎用のオブジェクトファクトリクラステンプレートです。

    @tparam AllocationPolicy アロケーションポリシーを指定します。
    @tparam Enabled 内部用のパラメータです。常に省略してください。

    @details
     AllocationPolicy::HasStatefulAllocator が true であるか false であるかによって、実装が変更されます。
     詳しくはそれぞれの特殊化を参照してください。
*/
template <typename AllocationPolicy, typename Enabled = void>
class ObjectFactory
{
    // doxygen バグ対策のためメインテンプレートを空定義しておく。
    // 宣言だけだと class でなく singleton として解釈されてしまう。
};

/**
    @brief 実装クラスのインスタンスを埋め込んで生成されたサービスオブジェクト専用の共有ポインタを表します。

    @tparam Interface インターフェイスを指定します。
    @tparam Impl 実装クラスを指定します。

    @details
     このクラスは SharedPointer<Interface> を継承しており SharedPointer<Interface> として振る舞うことができます。
     また、 SharedPointer<Interface> への変換が可能です。

     加えて GetImpl() によって実装クラスへの参照を取得することが可能です。
*/
template <typename Interface, typename Impl>
class EmplacedRef
    : public SharedPointer<Interface>
{
    template <typename, typename>
    friend class ObjectFactory;
public:

    /**
        @brief デフォルトコンストラクタ: 何も指さない EmplacedRef として初期化します。

        @post static_cast<bool>(*this) == false
    */
    EmplacedRef() NN_NOEXCEPT
    {
    }

    /**
        @brief 実装クラスのインスタンスへの参照を取得します。

        @return 実装クラスのインスタンスへの参照を返します。
    */
    Impl& GetImpl() const NN_NOEXCEPT
    {
        return *detail::EmplacedImplHolder<Impl>::template GetEmplacedImplPointerImpl<Interface>(*this);
    }

private:

    explicit EmplacedRef(Interface* p, bool addReference) NN_NOEXCEPT
        : SharedPointer<Interface>(p, addReference)
    {
    }

};

/**
    @brief 状態を持たないアロケータを使用してオブジェクトを生成するファクトリクラスです。

    @tparam AllocationPolicy アロケーションポリシーを指定します。
*/
template <typename AllocationPolicy>
class ObjectFactory<AllocationPolicy, typename std::enable_if<!AllocationPolicy::HasStatefulAllocator>::type>
{
private:

    template <typename Interface, class Holder, typename Arg>
    static nn::sf::SharedPointer<Interface> CreateSharedForPointer(Arg p) NN_NOEXCEPT
    {
        typedef impl::detail::ImplTemplateBase<Interface, Interface, Holder, Holder> Base;
        return SharedPointer<Interface>(ObjectImplFactory<Base, AllocationPolicy>::Create(std::forward<Arg>(p)), false);
    }

public:

    /**
        @brief 実装クラスを埋め込んでインターフェイス実装オブジェクトを生成します。

        @tparam Interface インターフェイスを指定します。
        @tparam Impl 実装クラスを指定します。
        @tparam Args Impl が受け取れる引数型列を指定します。
        @param[in] args 引数列を指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         Impl のオブジェクトを内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
         Impl(std::foward<Args>(args)...) 相当の呼び出しによって、内部に保持される Impl オブジェクトが初期化されます。
         返り値の EmplacedRef は、GetImpl() で内部 Impl オブジェクト参照の取得、および、SharedPointer への暗黙変換が可能です。
    */
    template <typename Interface, typename Impl, typename... Args>
    static EmplacedRef<Interface, Impl> CreateSharedEmplaced(Args&&... args) NN_NOEXCEPT
    {
        typedef impl::detail::ImplTemplateBase<Interface, Interface, detail::EmplacedImplHolder<Impl>, detail::EmplacedImplHolder<Impl>> Base;
        return EmplacedRef<Interface, Impl>(ObjectImplFactory<Base, AllocationPolicy>::Create(std::forward<Args>(args)...), false);
    }

    /**
        @brief 指定されたユーザ型を元に共有オブジェクトを作成します。

        @tparam UserSharedObject ユーザオブジェクト型を指定します。ISharedObject を継承している必要があります。
        @tparam Args UserSharedObject のコンストラクタが受け取れる引数型列を指定します。
        @param[in] args 引数列を指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         UserSharedObject のオブジェクトを内部に持つオブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
         UserSharedObject(std::foward<Args>(args)...) 相当の呼び出しによって、内部に保持される UserSharedObject オブジェクトが初期化されます。
    */
    template <typename UserSharedObject, typename... Args>
    static SharedPointer<UserSharedObject> CreateUserSharedObject(Args&&... args) NN_NOEXCEPT
    {
        return SharedPointer<UserSharedObject>(ObjectImplFactory<UserSharedObject, AllocationPolicy>::Create(std::forward<Args>(args)...), false);
    }

    /**
        @brief CreateSharedEmplaced() によって生成されたオブジェクトの実装へのポインタを取得します。

        @tparam Impl 実装クラスを指定します。
        @tparam Interface インターフェイスを指定します。
        @param[in] p 同じ型引数の CreateSharedEmplaced() から生成されたオブジェクトへの共有ポインタを指定します。

        @return Impl へのポインタを返します。

        @pre p が同じ型引数の CreateSharedEmplaced() から生成されたオブジェクトへの共有ポインタである

        @details
         代わりに EmplacedRef::GetImpl() を使用することを推奨します。
    */
    template <typename Impl, typename Interface>
    static Impl* GetEmplacedImplPointer(const nn::sf::SharedPointer<Interface>& p) NN_NOEXCEPT
    {
        return detail::EmplacedImplHolder<Impl>::template GetEmplacedImplPointerImpl<Interface>(p);
    }

    /**
        @brief スマートポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

        @tparam Interface インターフェイスを指定します。
        @tparam SmartPointer 実装クラスを指定します。operator->() の結果が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
        @param[in] p 実装となるポインタを指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         SmartPointer を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
         この関数の呼び出しによって p への参照は返り値のオブジェクトによって保持され、
         そのデストラクタによってその参照が削除されます。
    */
    template <typename Interface, typename SmartPointer>
    static nn::sf::SharedPointer<Interface> CreateShared(SmartPointer&& p) NN_NOEXCEPT
    {
        return CreateSharedForPointer<Interface, detail::SmartPointerHolder<typename std::decay<decltype(p)>::type>>(std::forward<SmartPointer>(p));
    }

    // 未テスト
    /**
        @brief ポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

        @tparam Interface インターフェイスを指定します。
        @tparam T 実装クラスを指定します。T* 型が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
        @param[in] p 実装となるポインタを指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         T* を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
         生成オブジェクトのデストラクタでは、delete p が呼ばれ、破棄されます。
    */
    template <typename Interface, typename T>
    static nn::sf::SharedPointer<Interface> CreateShared(T* p) NN_NOEXCEPT
    {
        return CreateSharedForPointer<Interface, detail::ManagedPointerHolder<T>>(p);
    }

    /**
        @brief ポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

        @tparam Interface インターフェイスを指定します。
        @tparam T 実装クラスを指定します。T* 型が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
        @param[in] p 実装となるポインタを指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         T* を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされていますが、
         生成オブジェクトのデストラクタでは、インターフェイス実装オブジェクトの破棄だけを行い、
         p に対しての破棄処理は行いません。
    */
    template <typename Interface, typename T>
    static nn::sf::SharedPointer<Interface> CreateSharedWithoutManagement(T* p) NN_NOEXCEPT
    {
        return CreateSharedForPointer<Interface, detail::UnmanagedPointerHolder<T>>(p);
    }

};

/**
    @brief 状態を持つアロケータを使用してオブジェクトを生成するファクトリクラスです。

    @tparam AllocationPolicy アロケーションポリシーを指定します。
*/
template <typename AllocationPolicy>
class ObjectFactory<AllocationPolicy, typename std::enable_if<AllocationPolicy::HasStatefulAllocator>::type>
{
public:

    /**
        @brief AllocationPolicy から得た、アロケータの型です。
    */
    typedef typename AllocationPolicy::Allocator Allocator;

private:

    template <typename Interface, class Holder, typename Arg>
    static nn::sf::SharedPointer<Interface> CreateSharedForPointer(Allocator* allocator, Arg p) NN_NOEXCEPT
    {
        typedef impl::detail::ImplTemplateBase<Interface, Interface, Holder, Holder> Base;
        return SharedPointer<Interface>(ObjectImplFactory<Base, AllocationPolicy>::Create(allocator, std::forward<Arg>(p)), false);
    }

public:

    /**
        @brief 実装クラスを埋め込んでインターフェイス実装オブジェクトを生成します。

        @tparam Interface インターフェイスを指定します。
        @tparam Impl 実装クラスを指定します。
        @tparam Args Impl が受け取れる引数型列を指定します。
        @param[in] allocator アロケータへのポインタを指定します。
        @param[in] args 引数列を指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         Impl のオブジェクトを内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
         Impl(std::foward<Args>(args)...) 相当の呼び出しによって、内部に保持される Impl オブジェクトが初期化されます。
    */
    template <typename Interface, typename Impl, typename... Args>
    static EmplacedRef<Interface, Impl> CreateSharedEmplaced(Allocator* allocator, Args&&... args) NN_NOEXCEPT
    {
        typedef impl::detail::ImplTemplateBase<Interface, Interface, detail::EmplacedImplHolder<Impl>, detail::EmplacedImplHolder<Impl>> Base;
        return EmplacedRef<Interface, Impl>(ObjectImplFactory<Base, AllocationPolicy>::Create(allocator, std::forward<Args>(args)...), false);
    }

    /**
        @brief 指定されたユーザ型を元に共有オブジェクトを作成します。

        @tparam UserSharedObject ユーザオブジェクト型を指定します。ISharedObject を継承している必要があります。
        @tparam Args UserSharedObject のコンストラクタが受け取れる引数型列を指定します。
        @param[in] allocator アロケータへのポインタを指定します。
        @param[in] args 引数列を指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         UserSharedObject のオブジェクトを内部に持つオブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
         UserSharedObject(std::foward<Args>(args)...) 相当の呼び出しによって、内部に保持される UserSharedObject オブジェクトが初期化されます。
    */
    template <typename UserSharedObject, typename... Args>
    static SharedPointer<UserSharedObject> CreateUserSharedObject(Allocator* allocator, Args&&... args) NN_NOEXCEPT
    {
        return SharedPointer<UserSharedObject>(ObjectImplFactory<UserSharedObject, AllocationPolicy>::Create(allocator, std::forward<Args>(args)...), false);
    }

    /**
        @brief CreateSharedEmplaced() によって生成されたオブジェクトの実装へのポインタを取得します。

        @tparam Impl 実装クラスを指定します。
        @tparam Interface インターフェイスを指定します。本テンプレート引数を明示的に指定する必要はありません。
        @param[in] p 同じ型引数の CreateSharedEmplaced() から生成されたオブジェクトへの共有ポインタを指定します。

        @return Impl へのポインタを返します。

        @pre p が同じ型引数の CreateSharedEmplaced() から生成されたオブジェクトへの共有ポインタである

        @details
         代わりに EmplacedRef::GetImpl() を使用することを推奨します。
    */
    template <typename Impl, typename Interface>
    static Impl* GetEmplacedImplPointer(const nn::sf::SharedPointer<Interface>& p) NN_NOEXCEPT
    {
        return detail::EmplacedImplHolder<Impl>::template GetEmplacedImplPointerImpl<Interface>(p);
    }

    /**
        @brief スマートポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

        @tparam Interface インターフェイスを指定します。
        @tparam SmartPointer 実装クラスを指定します。operator->() の結果が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
        @param[in] allocator アロケータへのポインタを指定します。
        @param[in] p 実装となるポインタを指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         SmartPointer を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
    */
    template <typename Interface, typename SmartPointer>
    static nn::sf::SharedPointer<Interface> CreateShared(Allocator* allocator, SmartPointer&& p) NN_NOEXCEPT
    {
        return CreateSharedForPointer<Interface, detail::SmartPointerHolder<typename std::decay<decltype(p)>::type>>(allocator, std::forward<SmartPointer>(p));
    }

    // 未テスト
    /**
        @brief ポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

        @tparam Interface インターフェイスを指定します。
        @tparam T 実装クラスを指定します。T* 型が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
        @param[in] allocator アロケータへのポインタを指定します。
        @param[in] p 実装となるポインタを指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         T* を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
         生成オブジェクトのデストラクタでは、delete p が呼ばれ、破棄されます。
    */
    template <typename Interface, typename T>
    static nn::sf::SharedPointer<Interface> CreateShared(Allocator* allocator, T* p) NN_NOEXCEPT
    {
        return CreateSharedForPointer<Interface, detail::ManagedPointerHolder<T>>(allocator, p);
    }

    /**
        @brief ポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

        @tparam Interface インターフェイスを指定します。
        @tparam T 実装クラスを指定します。T* 型が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
        @param[in] allocator アロケータへのポインタを指定します。
        @param[in] p 実装となるポインタを指定します。

        @return 生成したオブジェクトへの共有ポインタを返します。

        @details
         T* を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
         この際オブジェクトのアロケーションには、指定したアロケーションポリシーが使用されます。
         このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされていますが、
         生成オブジェクトのデストラクタでは、インターフェイス実装オブジェクトの破棄だけを行い、
         p に対しての破棄処理は行いません。
    */
    template <typename Interface, typename T>
    static nn::sf::SharedPointer<Interface> CreateSharedWithoutManagement(Allocator* allocator, T* p) NN_NOEXCEPT
    {
        return CreateSharedForPointer<Interface, detail::UnmanagedPointerHolder<T>>(allocator, p);
    }

};

/**
    @brief アロケータ状態を内部に持つファクトリクラスです。
*/
template <typename AllocationPolicy>
class StatefulObjectFactory
{
public:

    typedef typename AllocationPolicy::Allocator Allocator;

private:

    typedef ObjectFactory<AllocationPolicy> StaticObjectFactory;
    Allocator* m_Allocator;

public:

    explicit StatefulObjectFactory(Allocator* allocator) NN_NOEXCEPT
        : m_Allocator(allocator)
    {
    }

    template <typename Interface, typename Impl, typename... Args>
    EmplacedRef<Interface, Impl> CreateSharedEmplaced(Args&&... args) NN_NOEXCEPT
    {
        return StaticObjectFactory::template CreateSharedEmplaced<Interface, Impl>(m_Allocator, std::forward<Args>(args)...);
    }

    template <typename Impl, typename Interface>
    static Impl* GetEmplacedImplPointer(const nn::sf::SharedPointer<Interface>& p) NN_NOEXCEPT
    {
        return StaticObjectFactory::template GetEmplacedImplPointer<Impl, Interface>(p);
    }

    template <typename Interface, typename SmartPointer>
    nn::sf::SharedPointer<Interface> CreateShared(SmartPointer&& p) NN_NOEXCEPT
    {
        return StaticObjectFactory::template CreateShared<Interface>(m_Allocator, std::forward<SmartPointer>(p));
    }

    // 未テスト
    template <typename Interface, typename T>
    nn::sf::SharedPointer<Interface> CreateShared(T* p) NN_NOEXCEPT
    {
        return StaticObjectFactory::template CreateShared<Interface>(m_Allocator, p);
    }

    template <typename Interface, typename T>
    nn::sf::SharedPointer<Interface> CreateSharedWithoutManagement(T* p) NN_NOEXCEPT
    {
        return StaticObjectFactory::template CreateSharedWithoutManagement<Interface>(m_Allocator, p);
    }

};


/**
    @brief DefaultAllocationPolicy を使用するオブジェクトファクトリです。
*/
typedef ObjectFactory<DefaultAllocationPolicy> DefaultObjectFactory;

/**
    @brief MemoryResource を毎回引数にとるオブジェクトファクトリです。
*/
typedef ObjectFactory<MemoryResourceAllocationPolicy> MemoryResourceObjectFactory;

//! @name サービスオブジェクト生成
//! @{

/**
    @brief 実装クラスを埋め込んでインターフェイス実装オブジェクトを生成します。

    @tparam Interface インターフェイスを指定します。
    @tparam Impl 実装クラスを指定します。
    @tparam Args Impl が受け取れる引数型列を指定します。
    @param[in] args 引数列を指定します。

    @return 生成したオブジェクトへの共有ポインタを返します。

    @details
        Impl のオブジェクトを内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
        この際オブジェクトのアロケーションには DefaultAllocationPolicy が使用されます。
        このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
        Impl(std::foward<Args>(args)...) 相当の呼び出しによって、内部に保持される Impl オブジェクトが初期化されます。
        返り値の EmplacedRef は、GetImpl() で内部 Impl オブジェクト参照の取得、および、SharedPointer への暗黙変換が可能です。
*/
template <typename Interface, typename Impl, typename... Args>
inline EmplacedRef<Interface, Impl> CreateSharedObjectEmplaced(Args&&... args) NN_NOEXCEPT
{
    return DefaultObjectFactory::CreateSharedEmplaced<Interface, Impl>(std::forward<Args>(args)...);
}

/**
    @brief 実装クラスを埋め込んでインターフェイス実装オブジェクトを生成します。

    @tparam Interface インターフェイスを指定します。
    @tparam Impl 実装クラスを指定します。
    @tparam Args Impl が受け取れる引数型列を指定します。
    @param[in] pMemoryResource MemoryResource を指定します。
    @param[in] args 引数列を指定します。

    @return 生成したオブジェクトへの共有ポインタを返します。

    @details
        Impl のオブジェクトを内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
        この際オブジェクトのアロケーションには pMemoryResource で指定された MemoryResource が使用されます。
        このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
        Impl(std::foward<Args>(args)...) 相当の呼び出しによって、内部に保持される Impl オブジェクトが初期化されます。
        返り値の EmplacedRef は、GetImpl() で内部 Impl オブジェクト参照の取得、および、SharedPointer への暗黙変換が可能です。
*/
template <typename Interface, typename Impl, typename... Args>
inline EmplacedRef<Interface, Impl> CreateSharedObjectEmplaced(MemoryResource* pMemoryResource, Args&&... args) NN_NOEXCEPT
{
    return MemoryResourceObjectFactory::CreateSharedEmplaced<Interface, Impl>(pMemoryResource, std::forward<Args>(args)...);
}

/**
    @brief スマートポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

    @tparam Interface インターフェイスを指定します。
    @tparam SmartPointer 実装クラスを指定します。operator->() の結果が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
    @param[in] p 実装となるポインタを指定します。

    @return 生成したオブジェクトへの共有ポインタを返します。

    @details
        SmartPointer を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
        この際オブジェクトのアロケーションには DefaultAllocationPolicy が使用されます。
        このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
*/
template <typename Interface, typename SmartPointer>
inline nn::sf::SharedPointer<Interface> CreateSharedObject(SmartPointer&& p) NN_NOEXCEPT
{
    return DefaultObjectFactory::CreateShared<Interface, SmartPointer>(std::forward<SmartPointer>(p));
}

/**
    @brief スマートポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

    @tparam Interface インターフェイスを指定します。
    @tparam SmartPointer 実装クラスを指定します。operator->() の結果が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
    @param[in] pMemoryResource MemoryResource を指定します。
    @param[in] p 実装となるポインタを指定します。

    @return 生成したオブジェクトへの共有ポインタを返します。

    @details
        SmartPointer を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
        この際オブジェクトのアロケーションには pMemoryResource で指定された MemoryResource が使用されます。
        このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
*/
template <typename Interface, typename SmartPointer>
inline nn::sf::SharedPointer<Interface> CreateSharedObject(MemoryResource* pMemoryResource, SmartPointer&& p) NN_NOEXCEPT
{
    return MemoryResourceObjectFactory::CreateShared<Interface, SmartPointer>(pMemoryResource, std::forward<SmartPointer>(p));
}

// 未テスト
/**
    @brief ポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

    @tparam Interface インターフェイスを指定します。
    @tparam T 実装クラスを指定します。T* 型が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
    @param[in] p 実装となるポインタを指定します。

    @return 生成したオブジェクトへの共有ポインタを返します。

    @details
        T* を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
        この際オブジェクトのアロケーションには DefaultAllocationPolicy が使用されます。
        このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされています。
        生成オブジェクトのデストラクタでは、delete p が呼ばれ、破棄されます。
*/
template <typename Interface, typename T>
inline nn::sf::SharedPointer<Interface> CreateSharedObject(T* p) NN_NOEXCEPT
{
    return DefaultObjectFactory::CreateShared<Interface, T>(std::move(p));
}

/**
    @brief ポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

    @tparam Interface インターフェイスを指定します。
    @tparam T 実装クラスを指定します。T* 型が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
    @param[in] p 実装となるポインタを指定します。

    @return 生成したオブジェクトへの共有ポインタを返します。

    @details
        T* を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
        この際オブジェクトのアロケーションには DefaultAllocationPolicy が使用されます。
        このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされていますが、
        生成オブジェクトのデストラクタでは、インターフェイス実装オブジェクトの破棄だけを行い、
        p に対しての破棄処理は行いません。
*/
template <typename Interface, typename T>
inline nn::sf::SharedPointer<Interface> CreateSharedObjectWithoutManagement(T* p) NN_NOEXCEPT
{
    return DefaultObjectFactory::CreateSharedWithoutManagement<Interface, T>(std::move(p));
}

/**
    @brief ポインタによって与えられたオブジェクトを実装とするインターフェイス実装オブジェクトを返します。

    @tparam Interface インターフェイスを指定します。
    @tparam T 実装クラスを指定します。T* 型が Interface に適合する関数を持つ必要があります。本テンプレート引数を明示的に指定する必要はありません。
    @param[in] pMemoryResource MemoryResource を指定します。
    @param[in] p 実装となるポインタを指定します。

    @return 生成したオブジェクトへの共有ポインタを返します。

    @details
        T* を内部に持つ Interface 実装オブジェクトを生成し、そのオブジェクトへの共有ポインタを返します。
        この際オブジェクトのアロケーションには pMemoryResource で指定された MemoryResource が使用されます。
        このオブジェクトは、参照カウントによる実装で ISharedObject::AddReference() と ISharedObject::Release() がオーバロードされていますが、
        生成オブジェクトのデストラクタでは、インターフェイス実装オブジェクトの破棄だけを行い、
        p に対しての破棄処理は行いません。
*/
template <typename Interface, typename T>
inline nn::sf::SharedPointer<Interface> CreateSharedObjectWithoutManagement(MemoryResource* pMemoryResource, T* p) NN_NOEXCEPT
{
    return MemoryResourceObjectFactory::CreateSharedWithoutManagement<Interface, T>(pMemoryResource, std::move(p));
}

//! @}

}}

/**
    @brief マクロを書いた型をスマートポインタとして扱うことを宣言します。

    @details
     本マクロで指定された型を Emplaced 系のオブジェクト作成関数に渡した場合、
     type::operator->() で取得した値に対してインターフェイスの実装関数を検索します。
*/
#define NN_SF_MARK_THIS_AS_SMART_POINTER \
    public: \
        static constexpr bool IsTreatedAsSmartPointerBySf = true
